package com.paystack.android_sdk.ui

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.IconButtonDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.androiddesignsystem.PaystackTheme
import kotlinx.coroutines.delay

@Composable
internal fun PaymentsUi(
    viewModel: PaymentsViewModel,
    modifier: Modifier = Modifier,
    onPaymentComplete: (Charge) -> Unit,
    onPaymentCancelled: () -> Unit,
    onError: (error: Throwable, reference: String?) -> Unit
) {
    val paymentUiState by viewModel.paymentUiState.collectAsState()
    val transactionAccessData by viewModel.transactionAccessData.collectAsState()

    if (paymentUiState.isTerminal) {
        LaunchedEffect("autoClose") {
            delay(AUTO_CLOSE_TIMEOUT)
            when (val state = paymentUiState) {
                is PaymentUiState.Success -> onPaymentComplete(state.charge)
                is PaymentUiState.Error -> onError(state.error, transactionAccessData?.reference)
                else -> { /* Do nothing */
                }
            }
        }
    }

    Column(modifier = modifier) {
        Row(
            modifier = Modifier.padding(
                top = PaystackTheme.dimensions.spacing.half,
                start = PaystackTheme.dimensions.spacing.half,
                end = PaystackTheme.dimensions.spacing.half,
            )
        ) {
            IconButton(
                onClick = onPaymentCancelled,
                colors = IconButtonDefaults.iconButtonColors(
                    contentColor = PaystackTheme.colors.neutralColors.gray01
                )
            ) {
                Icon(
                    painter = painterResource(R.drawable.pstk_ic_close),
                    contentDescription = stringResource(R.string.pstk_close_icon)
                )
            }
        }

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        PaymentUiContent(
            paymentUiState = paymentUiState,
            modifier = Modifier.weight(1f),
            onComplete = viewModel::onTransactionComplete,
            onError = { error -> onError(error, transactionAccessData?.reference) },
        )
    }
}

private const val AUTO_CLOSE_TIMEOUT = 2000L
