package com.oogbox.support.orm.core.data;

import android.database.Cursor;

import com.oogbox.support.orm.BaseModel;
import com.oogbox.support.orm.core.factory.BaseModelCursor;
import com.oogbox.support.orm.core.factory.BaseModelCursorUtilsImpl;
import com.oogbox.support.orm.core.helper.M2MTable;
import com.oogbox.support.orm.core.helper.RecordWrapper;
import com.oogbox.support.orm.core.types.OEnum;
import com.oogbox.support.orm.core.types.OManyToMany;
import com.oogbox.support.orm.core.types.OManyToOne;
import com.oogbox.support.orm.core.types.OOneToMany;
import com.oogbox.support.orm.core.types.helper.OFieldType;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class ORecord extends RecordWrapper<ORecord> {

    private BaseModel model;

    public ORecord setModel(BaseModel model) {
        this.model = model;
        return this;
    }

    public String getEnumValue(String key) {
        String value = getString(key);
        if (value != null && model != null) {
            OFieldType type = model.getColumn(key);
            if (type != null) {
                OEnum oEnum = (OEnum) type;
                return oEnum.getEnumVal(value);
            }
        }
        return value;
    }

    public ORecord readMany2One(String key) {
        Integer recordId = getInt(key);
        if (recordId != null && model != null) {
            OManyToOne m2o = (OManyToOne) model.getColumn(key);
            BaseModel refModel = model.createModel(m2o.getRefModel());
            if (refModel != null) {
                return refModel.browse(recordId);
            }
        }
        return null;
    }

    public Integer readCountMany2Many(String key) {
        if (model != null) {
            OManyToMany col = (OManyToMany) model.getColumn(key);
            Integer recordId = getInt("_id");
            if (recordId != null) {
                M2MTable m2MTable = new M2MTable(model.getContext(), model, col);
                return m2MTable.readCount(recordId);
            }
        }
        return 0;
    }

    public Integer readCountOne2Many(String key) {
        if (model != null) {
            OOneToMany col = (OOneToMany) model.getColumn(key);
            Integer recordId = getInt("_id");
            if (recordId != null) {
                BaseModel refModel = model.createModel(col.getRefModel());
                return refModel.count(col.getRefColumn() + " = ? ", new String[]{recordId + ""});
            }
        }
        return 0;
    }

    public List<ORecord> readOne2Many(String key) {
        if (model != null) {
            OOneToMany col = (OOneToMany) model.getColumn(key);
            Integer recordId = getInt("_id");
            if (recordId != null) {
                BaseModel refModel = model.createModel(col.getRefModel());
                return refModel.select(col.getRefColumn() + " = ? ", new String[]{recordId + ""});
            }
        }
        return Collections.emptyList();
    }


    public List<ORecord> readMany2Many(String key) {
        if (model != null) {
            OManyToMany col = (OManyToMany) model.getColumn(key);
            Integer recordId = getInt("_id");
            if (recordId != null) {
                M2MTable m2MTable = new M2MTable(model.getContext(), model, col);
                return m2MTable.readAll(recordId);
            }
        }
        return Collections.emptyList();
    }

    public static ORecord fromCursor(Cursor cr) {
        ORecord record = new ORecord();
        List<String> relatedFields = new ArrayList<>();
        List<String> computeFields = new ArrayList<>();
        for (String column : cr.getColumnNames()) {
            if (cr instanceof BaseModelCursorUtilsImpl) {
                int index = cr.getColumnIndex(column);
                BaseModelCursorUtilsImpl modelCursor = (BaseModelCursorUtilsImpl) cr;
                if (modelCursor.isComputeType(index)) {
                    computeFields.add(column);
                } else if (modelCursor.isRelatedType(index)) {
                    relatedFields.add(column);
                } else {
                    record.put(column, cursorValue(column, cr));
                }
            } else {
                record.put(column, cursorValue(column, cr));
            }
        }
        if (!relatedFields.isEmpty()) {
            BaseModelCursorUtilsImpl modelCursor = (BaseModelCursorUtilsImpl) cr;
            for (String column : relatedFields) {
                record.put(column, modelCursor.getRelatedValue(cr.getColumnIndex(column)));
            }
        }
        if (!computeFields.isEmpty()) {
            BaseModelCursorUtilsImpl modelCursor = (BaseModelCursorUtilsImpl) cr;
            for (String column : computeFields) {
                record.put(column, modelCursor.getComputeValue(record, cr.getColumnIndex(column)));
            }
        }
        return record;
    }

    public static Object cursorValue(String column, Cursor cr) {
        Object value = false;
        int index = cr.getColumnIndex(column);
        if (index >= 0) {
            switch (cr.getType(index)) {
                case BaseModelCursor.FIELD_TYPE_NULL:
                    value = null;
                    break;
                case BaseModelCursor.FIELD_TYPE_STRING:
                    value = cr.getString(index);
                    break;
                case BaseModelCursor.FIELD_TYPE_INTEGER:
                    value = cr.getInt(index);
                    break;
                case BaseModelCursor.FIELD_TYPE_FLOAT:
                    value = cr.getFloat(index);
                    break;
                case BaseModelCursor.FIELD_TYPE_BLOB:
                    value = cr.getBlob(index);
                    break;
            }
        }
        return value;
    }
}
