package com.nimbusds.jose.jwk.loader;


import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import net.jcip.annotations.Immutable;
import org.apache.logging.log4j.Logger;


/**
 * JOSE and PKCS#11 configuration. Supports system properties override.
 *
 * <p>Example configuration:
 *
 * <pre>
 * jose.allowWeakKeys = false
 * pkcs11.enable = true
 * pkcs11.configFile = /WEB-INF/hsm.cfg
 * pkcs11.password = secret
 * </pre>
 */
@Immutable
class JOSEConfiguration {
	
	
	/**
	 * The name of the JOSE configuration file.
	 */
	static final String FILENAME = "/WEB-INF/jose.properties";
	
	
	/**
	 * If {@code true} weak RSA keys shorter than 2048 bits will be
	 * allowed. The default value if {@code false}.
	 */
	private final boolean allowWeakKeys;
	
	
	/**
	 * PKCS#11 enabled / disabled.
	 */
	private final boolean pkcs11Enabled;
	
	
	/**
	 * The PKCS#11 configuration, specified by file name or inline (with
	 * optional additional BASE64URL encoding), {@code null} if not
	 * specified.
	 */
	private final String pkcs11ConfigFile;
	
	
	/**
	 * The PKCS#11 password, empty if none or not applicable.
	 */
	private final char[] pkcs11KeyStorePassword;
	
	
	/**
	 * Creates new default JOSE configuration.
	 */
	public JOSEConfiguration() {
		this(new Properties());
	}
	
	
	/**
	 * Creates new JOSE configuration with the specified properties. System
	 * properties override is disabled.
	 *
	 * @param props The configuration properties, {@code null} if none.
	 */
	public JOSEConfiguration(final Properties props) {
		
		this(props, false);
	}
	
	
	/**
	 * Creates new JOSE configuration with the specified properties.
	 *
	 * @param props    The configuration properties, {@code null} if none.
	 * @param override Enables system properties override.
	 */
	public JOSEConfiguration(final Properties props, final boolean override) {
		
		var finalProps = new Properties();
		
		if (props != null) {
			finalProps.putAll(props);
		}
		
		// Override with matching system property
		if (override) {
			System.getProperties().forEach((key, value) -> {
				if (key.toString().startsWith("jose.") || key.toString().startsWith("pkcs11.")) {
					finalProps.put(key, value);
				}
			});
		}
		
		allowWeakKeys = "true".equalsIgnoreCase(finalProps.getProperty("jose.allowWeakKeys"));
		
		if ("true".equalsIgnoreCase(finalProps.getProperty("pkcs11.enable"))) {
			
			pkcs11Enabled = true;
			
			pkcs11ConfigFile = finalProps.getProperty("pkcs11.configFile");
			
			if (pkcs11ConfigFile == null || pkcs11ConfigFile.trim().isEmpty()) {
				throw new RuntimeException("PKCS#11 is enabled by pkcs11.enable, " +
					"but no PKCS#11 configuration is specified in pkcs11.configFile");
			}
			
			String pw = finalProps.getProperty("pkcs11.password");
			if (pw == null || pw.trim().isEmpty()) {
				pkcs11KeyStorePassword = "".toCharArray();
			} else {
				pkcs11KeyStorePassword = pw.trim().toCharArray();
			}
			
		} else {
			pkcs11Enabled = false;
			pkcs11ConfigFile = null;
			pkcs11KeyStorePassword = "".toCharArray();
		}
	}
	
	
	/**
	 * Checks if weak keys are allowed.
	 *
	 * @return {@code true} if allowed, else {@code false}.
	 */
	boolean isAllowWeakKeys() {
		
		return allowWeakKeys;
	}
	
	
	/**
	 * Checks if PKCS#11 (HSM) is enabled.
	 *
	 * @return {@code true} if enabled, else {@code false}.
	 */
	boolean isPKCS11Enabled() {
		
		return pkcs11Enabled;
	}
	
	
	/**
	 * Returns the PKCS#11 (HSM) configuration, specified as file name or
	 * inline (with optional additional BASE64URL encoding).
	 *
	 * <p>See https://docs.oracle.com/en/java/javase/11/security/pkcs11-reference-guide1.html#GUID-C4ABFACB-B2C9-4E71-A313-79F881488BB9
	 *
	 * @return The PKCS#11 configuration, {@code null} if not specified.
	 */
	String getPKCS11ConfigurationFile() {
		
		return pkcs11ConfigFile;
	}
	
	
	/**
	 * Returns the PKCS#11 (HSM) password (or PIN).
	 *
	 * @return The password, empty array if none.
	 */
	char[] getPKCS11KeyStorePassword() {
		
		return pkcs11KeyStorePassword;
	}
	
	
	/**
	 * Returns {@code true} if a PKCS#11 (HSM) password is configured.
	 *
	 * @return {@code true} if a password is configured, {@code false} if
	 *         not.
	 */
	boolean hasPKCS11KeyStorePassword() {
		
		return getPKCS11KeyStorePassword().length > 0;
	}
	
	
	/**
	 * Logs the configuration at INFO level.
	 *
	 * @param logger The logger.
	 */
	void log(final Logger logger) {
		
		if (logger == null) {
			return;
		}
		
		logger.info("[SE0000] JOSE configuration: Allow weak RSA keys: {}", isAllowWeakKeys());
		logger.info("[SE0001] JOSE configuration: PKCS#11 enabled: {}", isPKCS11Enabled());
		
		if (isPKCS11Enabled()) {
			logger.info("[SE0002] JOSE configuration: PKCS#11 configuration file: {}", getPKCS11ConfigurationFile());
			logger.info("[SE0003] JOSE configuration: PKCS#11 password configured: {}", hasPKCS11KeyStorePassword());
		}
	}
	
	
	/**
	 * Loads a JOSE configuration using the specified file input stream
	 * source. System properties override is enabled.
	 *
	 * @param fisSource The file input source. Must not be {@code null}.
	 *
	 * @return The JOSE configuration.
	 *
	 * @throws IOException If an I/O exception is encountered.
	 */
	static JOSEConfiguration load(final FileInputStreamSource fisSource)
		throws IOException {
		
		InputStream is = fisSource.getInputSteam(FILENAME);
		
		var properties = new Properties();
		
		if (is != null) {
			properties.load(is);
		}
		
		return new JOSEConfiguration(properties, true);
	}
}
