/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent.android.sample;

import android.app.ActivityManager;
import android.content.Context;
import android.os.Build;
import android.os.Debug;
import android.os.Process;
import com.newrelic.agent.android.harvest.AgentHealth;
import com.newrelic.agent.android.logging.AgentLog;
import com.newrelic.agent.android.logging.AgentLogManager;
import com.newrelic.agent.android.metric.Metric;
import com.newrelic.agent.android.stats.TicToc;
import com.newrelic.agent.android.tracing.ActivityTrace;
import com.newrelic.agent.android.tracing.Sample;
import com.newrelic.agent.android.tracing.TraceLifecycleAware;
import com.newrelic.agent.android.tracing.TraceMachine;
import com.newrelic.agent.android.util.NamedThreadFactory;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;

public class Sampler
implements TraceLifecycleAware,
Runnable {
    protected static final long SAMPLE_FREQ_MS = 100L;
    protected static final long SAMPLE_FREQ_MS_MAX = 250L;
    private static final int[] PID = new int[]{Process.myPid()};
    private static final int KB_IN_MB = 1024;
    private static final AgentLog log = AgentLogManager.getAgentLog();
    private static final ReentrantLock samplerLock = new ReentrantLock();
    protected static Sampler sampler;
    protected static boolean cpuSamplingDisabled;
    private final ActivityManager activityManager;
    private final EnumMap<Sample.SampleType, Collection<Sample>> samples = new EnumMap(Sample.SampleType.class);
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor(new NamedThreadFactory("Sampler"));
    protected final AtomicBoolean isRunning = new AtomicBoolean(false);
    protected long sampleFreqMs = 100L;
    protected ScheduledFuture sampleFuture;
    private Long lastCpuTime;
    private Long lastAppCpuTime;
    private RandomAccessFile procStatFile;
    private RandomAccessFile appStatFile;
    private Metric samplerServiceMetric;

    protected Sampler(Context context) {
        this.activityManager = (ActivityManager)context.getSystemService("activity");
        this.samples.put(Sample.SampleType.MEMORY, new ArrayList());
        this.samples.put(Sample.SampleType.CPU, new ArrayList());
    }

    public static void init(Context context) {
        samplerLock.lock();
        try {
            if (sampler == null) {
                sampler = Sampler.provideSampler(context);
                Sampler.sampler.sampleFreqMs = 100L;
                Sampler.sampler.samplerServiceMetric = new Metric("samplerServiceTime");
                TraceMachine.addTraceListener(sampler);
                if (Build.VERSION.SDK_INT >= 24) {
                    cpuSamplingDisabled = true;
                    log.debug("CPU sampling not supported in Android 8 and above.");
                }
                log.debug("Sampler initialized");
            }
        }
        catch (Exception e) {
            log.error("Sampler init failed: " + e.getMessage());
            Sampler.shutdown();
        }
        finally {
            samplerLock.unlock();
        }
    }

    protected static Sampler provideSampler(Context context) {
        return new Sampler(context);
    }

    public static void start() {
        samplerLock.lock();
        try {
            if (sampler != null) {
                sampler.schedule();
                log.debug("Sampler started");
            }
        }
        finally {
            samplerLock.unlock();
        }
    }

    public static void stop() {
        samplerLock.lock();
        try {
            if (sampler != null) {
                sampler.stop(false);
                log.debug("Sampler stopped");
            }
        }
        finally {
            samplerLock.unlock();
        }
    }

    public static void stopNow() {
        samplerLock.lock();
        try {
            if (sampler != null) {
                sampler.stop(true);
                log.debug("Sampler hard stopped");
            }
        }
        finally {
            samplerLock.unlock();
        }
    }

    public static void shutdown() {
        samplerLock.lock();
        try {
            if (sampler != null) {
                TraceMachine.removeTraceListener(sampler);
                Sampler.stopNow();
                sampler = null;
                log.debug("Sampler shutdown");
            }
        }
        finally {
            samplerLock.unlock();
        }
    }

    @Override
    public void run() {
        try {
            if (this.isRunning.get()) {
                this.sample();
            }
        }
        catch (Exception e) {
            log.error("Caught exception while running the sampler", e);
            AgentHealth.noticeException(e);
        }
    }

    protected void schedule() {
        samplerLock.lock();
        try {
            if (!this.isRunning.get()) {
                this.clear();
                this.sampleFuture = this.scheduler.scheduleWithFixedDelay(this, 0L, this.sampleFreqMs, TimeUnit.MILLISECONDS);
                this.isRunning.set(true);
                log.debug(String.format("Sampler scheduler started; sampling will occur every %d ms.", this.sampleFreqMs));
            }
        }
        catch (Exception e) {
            log.error("Sampler scheduling failed: " + e.getMessage());
            AgentHealth.noticeException(e);
        }
        finally {
            samplerLock.unlock();
        }
    }

    protected void stop(boolean immediate) {
        samplerLock.lock();
        try {
            if (this.isRunning.get()) {
                this.isRunning.set(false);
                if (this.sampleFuture != null) {
                    this.sampleFuture.cancel(immediate);
                }
                this.resetCpuSampler();
                log.debug("Sampler canceled");
            }
        }
        catch (Exception e) {
            log.error("Sampler stop failed: " + e.getMessage());
            AgentHealth.noticeException(e);
        }
        finally {
            samplerLock.unlock();
        }
    }

    protected static boolean isRunning() {
        if (sampler == null || Sampler.sampler.sampleFuture == null) {
            return false;
        }
        return !Sampler.sampler.sampleFuture.isDone();
    }

    protected void monitorSamplerServiceTime(double serviceTime) {
        this.samplerServiceMetric.sample(serviceTime);
        Double serviceTimeAvg = this.samplerServiceMetric.getTotal() / (double)this.samplerServiceMetric.getCount();
        if (serviceTimeAvg > (double)this.sampleFreqMs) {
            log.debug("Sampler: sample service time has been exceeded. Increase by 10%");
            this.sampleFreqMs = Math.min((long)((float)this.sampleFreqMs * 1.1f), 250L);
            if (this.sampleFuture != null) {
                this.sampleFuture.cancel(true);
            }
            this.sampleFuture = this.scheduler.scheduleWithFixedDelay(this, 0L, this.sampleFreqMs, TimeUnit.MILLISECONDS);
            log.debug(String.format("Sampler scheduler restarted; sampling will now occur every %d ms.", this.sampleFreqMs));
            this.samplerServiceMetric.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void sample() {
        TicToc timer = new TicToc();
        samplerLock.lock();
        try {
            Sample cpuSample;
            timer.tic();
            Sample memorySample = Sampler.sampleMemory();
            if (memorySample != null) {
                this.getSampleCollection(Sample.SampleType.MEMORY).add(memorySample);
            }
            if ((cpuSample = this.sampleCpu()) != null) {
                this.getSampleCollection(Sample.SampleType.CPU).add(cpuSample);
            }
        }
        catch (Exception e) {
            log.error("Sampling failed: " + e.getMessage());
            AgentHealth.noticeException(e);
        }
        finally {
            samplerLock.unlock();
        }
        this.monitorSamplerServiceTime(timer.toc());
    }

    protected void clear() {
        for (Collection<Sample> sampleCollection : this.samples.values()) {
            sampleCollection.clear();
        }
    }

    public static Sample sampleMemory() {
        if (sampler == null) {
            return null;
        }
        return Sampler.sampleMemory(Sampler.sampler.activityManager);
    }

    public static Sample sampleMemory(ActivityManager activityManager) {
        try {
            int totalPss;
            Debug.MemoryInfo[] memInfo = activityManager.getProcessMemoryInfo(PID);
            if (memInfo.length > 0 && (totalPss = memInfo[0].getTotalPss()) >= 0) {
                Sample sample = new Sample(Sample.SampleType.MEMORY);
                sample.setSampleValue((double)totalPss / 1024.0);
                return sample;
            }
        }
        catch (Exception e) {
            log.error("Sample memory failed: " + e.getMessage());
            AgentHealth.noticeException(e);
        }
        return null;
    }

    protected static Sample sampleCpuInstance() {
        if (sampler == null) {
            return null;
        }
        return sampler.sampleCpu();
    }

    public Sample sampleCpu() {
        if (cpuSamplingDisabled) {
            return null;
        }
        try {
            if (this.procStatFile == null || this.appStatFile == null) {
                this.appStatFile = new RandomAccessFile("/proc/" + PID[0] + "/stat", "r");
                this.procStatFile = new RandomAccessFile("/proc/stat", "r");
            } else {
                this.procStatFile.seek(0L);
                this.appStatFile.seek(0L);
            }
            String procStatString = this.procStatFile.readLine();
            String appStatString = this.appStatFile.readLine();
            String[] procStats = procStatString.split(" ");
            String[] appStats = appStatString.split(" ");
            long cpuTime = Long.parseLong(procStats[2]) + Long.parseLong(procStats[3]) + Long.parseLong(procStats[4]) + Long.parseLong(procStats[5]) + Long.parseLong(procStats[6]) + Long.parseLong(procStats[7]) + Long.parseLong(procStats[8]);
            long appTime = Long.parseLong(appStats[13]) + Long.parseLong(appStats[14]);
            if (this.lastCpuTime == null && this.lastAppCpuTime == null) {
                this.lastCpuTime = cpuTime;
                this.lastAppCpuTime = appTime;
                return null;
            }
            Sample sample = new Sample(Sample.SampleType.CPU);
            sample.setSampleValue((double)(appTime - this.lastAppCpuTime) / (double)(cpuTime - this.lastCpuTime) * 100.0);
            this.lastCpuTime = cpuTime;
            this.lastAppCpuTime = appTime;
            return sample;
        }
        catch (Exception e) {
            cpuSamplingDisabled = true;
            log.debug("Exception hit while CPU sampling: " + e.getMessage());
            AgentHealth.noticeException(e);
            return null;
        }
    }

    private void resetCpuSampler() {
        this.lastCpuTime = null;
        this.lastAppCpuTime = null;
        if (this.appStatFile != null && this.procStatFile != null) {
            try {
                this.appStatFile.close();
                this.procStatFile.close();
                this.appStatFile = null;
                this.procStatFile = null;
            }
            catch (IOException e) {
                log.debug("Exception hit while resetting CPU sampler: " + e.getMessage());
                AgentHealth.noticeException(e);
            }
        }
    }

    public static Map<Sample.SampleType, Collection<Sample>> copySamples() {
        EnumMap<Sample.SampleType, Collection<Sample>> copy;
        samplerLock.lock();
        try {
            if (sampler == null) {
                samplerLock.unlock();
                HashMap<Sample.SampleType, Collection<Sample>> hashMap = new HashMap<Sample.SampleType, Collection<Sample>>();
                return hashMap;
            }
            copy = new EnumMap<Sample.SampleType, Collection<Sample>>(Sampler.sampler.samples);
            for (Sample.SampleType key : Sampler.sampler.samples.keySet()) {
                copy.put(key, new ArrayList<Sample>(Sampler.sampler.samples.get((Object)key)));
            }
        }
        finally {
            samplerLock.unlock();
        }
        return Collections.unmodifiableMap(copy);
    }

    private Collection<Sample> getSampleCollection(Sample.SampleType type) {
        return this.samples.get((Object)type);
    }

    @Override
    public void onEnterMethod() {
        if (this.isRunning.get()) {
            return;
        }
        Sampler.start();
    }

    @Override
    public void onExitMethod() {
    }

    @Override
    public void onTraceStart(ActivityTrace activityTrace) {
        Sampler.start();
    }

    @Override
    public void onTraceComplete(final ActivityTrace activityTrace) {
        this.scheduler.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    Sampler.this.stop(true);
                    activityTrace.setVitals(Sampler.copySamples());
                    Sampler.this.clear();
                }
                catch (RuntimeException e) {
                    log.error(e.toString());
                }
            }
        });
    }

    @Override
    public void onTraceRename(ActivityTrace activityTrace) {
    }

    static {
        cpuSamplingDisabled = false;
    }
}

