package com.networkingx.okhttpclient

import com.networkingx.request.ContentType
import com.networkingx.response.ErrorCode
import com.networkingx.internal.Generic
import com.networkingx.NetworkRequest
import com.networkingx.client.Client
import com.networkingx.log.Logger
import com.networkingx.log.Logger.w
import com.networkingx.policies.RetryPolicy
import com.networkingx.request.Request
import com.networkingx.request.RequestBody
import com.networkingx.response.OnError
import com.networkingx.response.OnSuccess
import com.networkingx.response.Response
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.RequestBody.Companion.toRequestBody
import okio.BufferedSource
import okio.ByteString
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException
import java.io.InputStream
import java.io.Reader
import java.text.ParseException
import java.util.concurrent.TimeUnit

class OkHttpClient : Client {
    private var client: OkHttpClient? = null

    @Suppress("unused")
    constructor(retryPolicy: RetryPolicy) : super(retryPolicy) {
        createClient(retryPolicy)
    }

    constructor() : super() {
        createClient()
    }

    @Suppress("unused")
    constructor(client: OkHttpClient?) : super() {
        this.client = client
    }

    private fun createClient(retryPolicy: RetryPolicy = this.retryPolicy) {
        client = OkHttpClient.Builder()
            .connectTimeout(
                retryPolicy.getCurrentTimeout().toLong(),
                TimeUnit.MILLISECONDS
            )
            .readTimeout(
                retryPolicy.getCurrentTimeout().toLong(),
                TimeUnit.MILLISECONDS
            )

            .addInterceptor(object : Interceptor {
                @Throws(IOException::class)
                override fun intercept(chain: Interceptor.Chain): okhttp3.Response {
                    val request = chain.request()
                    // try the request
                    var response = chain.proceed(request)
                    var tryCount = 0

                    while (!response.isSuccessful && tryCount < retryPolicy.getRetryCount()) {
                        w("intercept", "Request is not successful - $tryCount")
                        tryCount++
                        // Request customization: add request headers
                        val requestBuilder = request.newBuilder()
                            .headers(request.headers)
                            .method(request.method, request.body)
                        val newRequest = requestBuilder.build()
                        // retry the request
                        response = chain.proceed(newRequest)
                    }
                    // otherwise just pass the original response on
                    return response
                }
            }).build()
    }

    override fun canHandleRequest(url: String, method: Int): Boolean {
        return method != NetworkRequest.Method.OPTIONS && method != NetworkRequest.Method.TRACE
    }

    override fun <F> makeRequest(
        request: Request,
        onSuccess: OnSuccess<F>?,
        onError: OnError?
    ) {
        Logger.d(TAG, "Tag:${request.requestData.reqTAG}")
        Logger.d(TAG, "${request.requestData.reqTAG} request Url: ${request.requestData.requestUrl}")
        Logger.d(TAG, "${request.requestData.reqTAG} request Json Params: ${request.requestData.requestBody.body}")
        Logger.d(TAG, "${request.requestData.reqTAG} request Header: ${request.requestData.requestHeader}")
        val builder = okhttp3.Request.Builder()
            .url(request.requestData.requestUrl)
            .tag(request.requestData.reqTAG)
        request.requestData.requestHeader?.apply {
            for ((key1, value) in this) {
                builder.addHeader(key1, value ?: "")
            }
        }
        val requestBody = getRequestBody(request.requestData.requestBody)
        when (request.requestData.method) {
            NetworkRequest.Method.GET -> builder.get()
            NetworkRequest.Method.POST -> builder.post(requestBody)
            NetworkRequest.Method.DELETE -> builder.delete(requestBody)
            NetworkRequest.Method.HEAD -> builder.head()
            NetworkRequest.Method.PATCH -> builder.patch(requestBody)
            NetworkRequest.Method.PUT -> builder.put(requestBody)
            NetworkRequest.Method.OPTIONS -> {
                throw IllegalArgumentException(
                    "okHttp does not support Options request type," +
                            " Use " +
                            "volley request Manager for this type of request"
                )
            }
            NetworkRequest.Method.TRACE -> throw IllegalArgumentException(
                "okHttp does not support trace request type, " +
                        "Use " +
                        "volley request Manager for this type of request"
            )
        }
          request.retryPolicy?.let { createClient(it) }
        client!!.newCall(builder.build()).enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                Logger.e(TAG, request.requestData.reqTAG + " onErrorResponse >> errorCode: "
                        + ErrorCode.UNKNOWN_ERROR)
                onError?.invoke(java.lang.NullPointerException("unknown error"), ErrorCode.UNKNOWN_ERROR)
            }

            @Throws(IOException::class)
            override fun onResponse(call: Call, response: okhttp3.Response) {
                if (!response.isSuccessful) {
                    Logger.e(TAG, request.requestData.reqTAG +
                            " onErrorResponse >> errorCode: " + response.code)
                    onError?.invoke(NullPointerException("failed response"), response.code)
                    return
                }
                val body: ResponseBody? = response.body
                if (body == null) {
                    Logger.e(TAG, "onResponse jsonObject: null")
                    onError?.invoke(java.lang.NullPointerException("null response"),
                        ErrorCode.RESPONSE_NULL)
                    return
                }
                val responseHeaders = response.headers
                val headers: HashMap<String, String>
                headers = HashMap(responseHeaders.size)
                var i = 0
                val size = responseHeaders.size
                while (i < size) {
                    headers[responseHeaders.name(i)] = responseHeaders.value(i)
                    i++
                }
                var responseValue: Any? = null
                onSuccess ?: return
                try {
                    when {
                        Generic<OnSuccess<String?>>()
                            .checkType(onSuccess) || Generic<OnSuccess<String>>()
                            .checkType(
                                onSuccess
                            ) -> {
                            responseValue = body.string()
                        }
                        Generic<OnSuccess<ByteArray?>>()
                            .checkType(onSuccess) || Generic<OnSuccess<ByteArray>>()
                            .checkType(
                                onSuccess
                            ) -> {
                            responseValue = body.bytes()
                        }
                        Generic<OnSuccess<ByteString?>>()
                            .checkType(onSuccess) || Generic<OnSuccess<ByteString>>()
                            .checkType(
                                onSuccess
                            ) -> {
                            responseValue = body.byteString()
                        }
                        Generic<OnSuccess<InputStream?>>()
                            .checkType(onSuccess) || Generic<OnSuccess<InputStream>>()
                            .checkType(
                                onSuccess
                            ) -> {
                            responseValue = body.byteStream()
                        }
                        Generic<OnSuccess<Reader?>>()
                            .checkType(onSuccess) || Generic<OnSuccess<Reader>>()
                            .checkType(
                                onSuccess
                            ) -> {
                            responseValue = body.charStream()
                        }
                        Generic<OnSuccess<BufferedSource?>>()
                            .checkType(onSuccess) || Generic<OnSuccess<BufferedSource>>()
                            .checkType(
                                onSuccess
                            ) -> {
                            responseValue = body.source().apply { close() }
                        }
                        else -> {
                            onError?.invoke(
                                ParseException("unexpected response format", 0),
                                ErrorCode.PARSE_ERROR
                            )
                        }
                    }
                    Logger.d(TAG, "onResponse jsonObject: ${responseValue.toString()}")
                    onSuccess.invoke(
                        Response(
                            responseValue, headers, response.code,
                            response.receivedResponseAtMillis - response.sentRequestAtMillis,
                            Response.LoadedFrom.NETWORK
                        ) as Response<F>
                    )
                } catch (e: JSONException) {
                    Logger.e(
                        TAG, request.requestData.reqTAG + " onErrorResponse >> errorCode: "
                                + ErrorCode.PARSE_ERROR
                    )
                    e.printStackTrace()
                    onError?.invoke(
                        ParseException("unexpected response format", 0),
                        ErrorCode.PARSE_ERROR
                    )
                }
            }
        })
    }


    override fun cancelPendingRequests(tag: String?) {
        val calls = client!!.dispatcher.queuedCalls()
        for (call in calls) {
            if (call.request().tag() == tag && !call.isCanceled()) {
                call.cancel()
            }
        }
    }

    override fun cancelAllRequests() {
        client!!.dispatcher.cancelAll()
    }

    private fun getRequestBody(requestBody: RequestBody): okhttp3.RequestBody {

        return if (!requestBody.body.isNullOrEmpty()) {
            requestBody.body!!.toRequestBody(requestBody.mediaType.toMediaType())
        } else if (requestBody.bodyParams != null && requestBody.encodedBodyParams != null) {
            val builder = FormBody.Builder()
            for ((key, value) in requestBody.bodyParams!!.entries) {
                builder.add(key, value)
            }
            for ((key, value) in requestBody.encodedBodyParams!!.entries) {
                builder.addEncoded(key, value)
            }
            builder.build()
        } else {
            val params = requestBody.body ?: JSONObject().toString()
            params.toRequestBody(requestBody.mediaType.toMediaType())
        }

    }

    @Suppress("unused")
    companion object {
        val JSON: MediaType = ContentType.JSON.toString().toMediaType()
        val STRING: MediaType = ContentType.STRING.toString().toMediaType()
    }
}