package com.netcore.android.workmgr

import android.content.Context
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.Smartech
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventCommonDataDump
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTApiService
import com.netcore.android.network.SMTEnumHttpMethodType
import com.netcore.android.network.models.SMTPushAmpResponse
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTResponse
import com.netcore.android.notification.*
import com.netcore.android.notification.SMTNotificationGeneratorProvider
import com.netcore.android.notification.SMTNotificationParser
import com.netcore.android.notification.SMTNotificationType
import com.netcore.android.notification.SMTPNHandler
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.preference.SMTGUIDPreferenceHelper
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference
import java.util.concurrent.TimeUnit
import java.util.*


internal class SMTPushAmpWorker(context: Context, params: WorkerParameters) : Worker(context, params) {

    private val TAG: String = SMTPushAmpWorker::class.java.simpleName
    private var mNotificationModel: SMTNotificationData? = null

    override fun doWork(): Result {
        val context = applicationContext
        // if SDK is not active cancel PushAmp worker
        val arePushampConditionsSatisfied = SMTWorkerScheduler.getInstance().arePushampConditionsSatisfied(context)
        if (!arePushampConditionsSatisfied) {
            SMTLogger.i(TAG, "Pushamp is disabled : $arePushampConditionsSatisfied")
            SMTWorkerScheduler.getInstance().cancelPushAmp(context)
            return Result.failure()
        }

        SMTLogger.v(TAG, "PushAmp is running")
        val smtResponse = makePushAmpApiCall(context)

        if (!isStopped) {
            when (smtResponse != null && smtResponse.isSuccess) {
                true -> {
                    handlePushAmpApiSuccess(context, smtResponse)
                    Result.success()
                }
                false -> {
                    Result.failure()
                }
            }
        }
        return Result.success()
    }

    override fun onStopped() {
        super.onStopped()
        SMTLogger.v(TAG, "PushAmp worker is stopped.")
    }

    /**
     * Makes push amp api call
     */
    private fun makePushAmpApiCall(context: Context): SMTResponse {
        // if push amp is not enabled then return
        val requestBuilder = SMTRequest.Builder()
                .setHttpMethod(SMTEnumHttpMethodType.GET)
                .setBaseUrl(SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_BASE_URL_PUSHAMP))
                .setEndPoint(getPushAmpEndPoint(context))
                .setApiId(SMTRequest.SMTApiTypeID.PUSH_AMPLIFICATION)
        return SMTApiService(requestBuilder.build()).makeApiCall()
    }

    /**
     * Provides Push Amp end point
     * @return String - API end point
     */
    private fun getPushAmpEndPoint(context: Context): String {
        val requestTime = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_PA_REQUEST_TIME, TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString())
        val guId = SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_GUID)
        val smtEventCommonDataDump = SMTEventCommonDataDump(context)
        val urlParams = smtEventCommonDataDump.getURLParameters()
        return "${SMTConfigConstants.PUSHAMP_API_PATH}?$urlParams&${SMTEventParamKeys.SMT_GUID}=$guId&${SMTEventParamKeys.SMT_REQUEST_TIME}=$requestTime&${SMTEventParamKeys.SMT_OS_NAME}=android&${SMTEventParamKeys.SMT_APP_BUNDLE_ID}=${smtEventCommonDataDump.getAppBundleId()}"
    }


    /**
     * handles push amp api success
     * schedules next push amp api call
     * @param response - API response
     */
    private fun handlePushAmpApiSuccess(context: Context, response: SMTResponse) {
        // Iterate through Push Amp notification to check if already shown to user or not
        // if not then show it to user, record into notification and event table
        val smtPreference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val pushAmpResponse = response as SMTPushAmpResponse
        SMTLogger.i(TAG, "PushAmp: ${pushAmpResponse.pushAmpData?.toString()}")
        pushAmpResponse.pushAmpData?.let { value ->
            if (!value.paEnabled) {
                SMTWorkerScheduler.getInstance().cancelPushAmp(context)
            }
        }

        // store push amp settings
        smtPreference.setBoolean(SMTPreferenceConstants.IS_PUSH_AMP_ENABLED, pushAmpResponse.pushAmpData?.paEnabled
                ?: true)
        smtPreference.setInt(SMTPreferenceConstants.PUSH_AMP_INTERVAL, pushAmpResponse.pushAmpData?.paInterval
                ?: SMTConfigConstants.DEFAULT_PUSH_AMP_INTERVAL)

        val areNotificationEnabled = SMTCommonUtility.areNotificationsEnabled(context)
        val isUserPNOptedIn = smtPreference.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION, true)
        val isNotificationListenerEnabled = smtPreference.getInt(SMTPreferenceConstants.SMT_MF_IS_NOTIFICATION_LISTENER_ENABLED, 0)
        if (areNotificationEnabled && isUserPNOptedIn) {
            SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_PA_REQUEST_TIME, TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString())

            val mPushAmpNotificationList = pushAmpResponse.pushAmpData?.notificationString
            if (mPushAmpNotificationList?.isNotEmpty() == true) {
                setPushAmpNotification(context, mPushAmpNotificationList, isNotificationListenerEnabled)
            }
            try {
                val mScheduledNotificationList = pushAmpResponse.pushAmpData?.scheduledNotification
                if (mScheduledNotificationList?.isNotEmpty() == true) {
                    Collections.sort(mScheduledNotificationList, SMTNotificationSorter())
                    mScheduledNotificationList?.let {
                        val mSmtScheduledNotification = SMTScheduleNotification()
                        mSmtScheduledNotification.parseScheduledNotification(context, mScheduledNotificationList, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP, false)
                    }
                }
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }
    }

    /*Pushamp Notification is inserted to DB with collapse key if exist
    *@param context :Context
    *@param mPushAmpNotificationList: ArrayList of String
    *@param isNotificationListenerEnabled: callback to app
    * It fetches the latest data based on publish time if collapse key is found in payload
    * */
    private fun setPushAmpNotification(context: Context, mPushAmpNotificationList: ArrayList<String>, isNotificationListenerEnabled: Int) {
        val mParser = SMTNotificationParser()
        val dbService = SMTDataBaseService.getInstance(WeakReference(context))
        val mUniqueCollapseKeyList = HashSet<String>()
        for (notif in mPushAmpNotificationList) {
            val mData = mParser.parse(notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
            val mTrid = mData?.mTrid
            if (mTrid != null) {
                val isTridExist = dbService.findNotificationWithId(mData.mTrid, mData.mSourceType)
                if (isNotificationListenerEnabled == 1) {
                    Smartech.getInstance(WeakReference(context)).getSMTNotificationListener()?.getSmartechNotifications(notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
                }
                if (isTridExist) {
                    return
                } else {
                    val mCollapse = mData.mCollapse
                    val mPublishDate = mData.mPublishedTimeStamp
                    if (mCollapse != null && mPublishDate != null) {
                        if (mCollapse.isNotEmpty()) {
                            mData.notificationId = SMTCommonUtility.getNotifyId(context, mCollapse)
                            dbService.insertPNToNotificationTable(mData.mTrid, notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP, mData)
                            mUniqueCollapseKeyList.add(mCollapse)
                        } else {
                            SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(context, notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
                        }
                    } else {
                        SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(context, notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
                    }
                    SMTEventRecorder.getInstance(context).recordNotificationDelivery(mData.mTrid, notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP, mData)
                }
            }
        }
        if (mUniqueCollapseKeyList.size > 0) {
            val mStrCollapse = SMTCommonUtility.getCollapseKeyListAsString(mUniqueCollapseKeyList)
            dbService.getUpdatedPNGroupByCollapseKey(mStrCollapse)?.forEach {
                SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(context, it.mPayload!!, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
            }
        }
    }


    /**
     * Provides app version Name
     * @param context - App context
     * @return String - App version name
     */
    private fun getAppVersionName(context: Context): String {
        val packageInfo: PackageInfo
        try {
            packageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            return packageInfo.versionName
        } catch (e: PackageManager.NameNotFoundException) {
            SMTLogger.e(TAG, "Unable to get app version name, error :- $e")
        }
        return ""
    }


    private fun deleteMedia(mNotificationModel: SMTNotificationData) {
        if (mNotificationModel.mNotificationType.equals(SMTNotificationType.CAROUSEL_PORTRAIT.type) || mNotificationModel.mNotificationType.equals(SMTNotificationType.CAROUSEL_LANDSCAPE.type)) {
            mNotificationModel.mCarouselList?.let { setup ->
                setup.forEach { item ->
                    item.mMediaLocalPath?.let {
                        SMTCommonUtility.deleteFile(it)
                    }

                }
            }
        } else {
            mNotificationModel.mMediaLocalPath?.let {
                SMTCommonUtility.deleteFile(it)
            }
        }
    }

    /*
    * It sorts the schedule notification from pushamp based on publish time;
    * there can be a scenario where in pushamp the notifications can come in random order
    * */
    internal class SMTNotificationSorter : Comparator<String> {
        val mParser = SMTNotificationParser()

        override fun compare(o1: String?, o2: String?): Int {
            val mO1 = mParser.parse(o1!!, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
            val mO2 = mParser.parse(o2!!, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)

            val mDate1 = SMTCommonUtility.convertStringToDate(mO1?.mPublishedTimeStamp)
            val mDate2 = SMTCommonUtility.convertStringToDate(mO2?.mPublishedTimeStamp)

            if (mDate1?.before(mDate2)!!) {
                return -1
            } else {
                return 1
            }
        }
    }
}




