package com.netcore.android.workmgr

import android.content.Context
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.netcore.android.SMTActivityLifecycleCallback
import com.netcore.android.logger.SMTLogger
import com.netcore.android.utility.SMTCommonUtility

/**
 *  This worker is created to sync events in background.
 *
 *  @author Netcore
 *
 *  @Description This worker will always run in background and its job is to check status of EventSyncWorker.
 *  If EventSyncWorker is not running then start the EventSyncWorker
 *
 *  By default, It's a periodic worker which will run after every 15 minutes interval which can be changed from SMTWorkerScheduler class.
 *
 */

internal class BackgroundSyncWorker(context: Context, param: WorkerParameters) : Worker(context, param) {

    val TAG: String = BackgroundSyncWorker::class.java.simpleName
    lateinit var context: Context
    override fun doWork(): Result {
        try {
            context = applicationContext

            val eventWorkerRunningStatus = SMTWorkerScheduler.getInstance().isEventWorkerRunning(context)
            SMTLogger.v(TAG, "Event worker Status : $eventWorkerRunningStatus")
            val isAppInForeground = SMTActivityLifecycleCallback.getInstance().isAppInForeground()

            // SDK is inactive so cancel running existing workers
            if (!SMTCommonUtility.checkPanelAndSDKActiveStatus(context)) {
                SMTLogger.w(TAG, "SDK / Panel is inactive")
                SMTWorkerScheduler.getInstance().cancelBackgroundSyncWorker(context)
                return Result.failure()
            }

            if (!eventWorkerRunningStatus) {
                if (!isAppInForeground) {
                    SMTLogger.i(TAG, "EventSync worker started.")
                    SMTWorkerScheduler.getInstance().scheduleEventWorker(context)
                }
            }
        } catch (e: Exception) {
            SMTLogger.v(TAG, e.localizedMessage)
        }
        return Result.success()
    }

    override fun onStopped() {
        super.onStopped()
        SMTLogger.v(TAG, "Background sync worker stopped")
    }
}