package com.netcore.android.notification.services

import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import android.util.Log
import androidx.core.app.JobIntentService
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationType
import com.netcore.android.notification.SMTScheduleNotification
import com.netcore.android.notification.SMTScheduledPNReceiver
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import com.netcore.android.workmgr.SMTWorkerScheduler
import java.lang.ref.WeakReference
import java.text.SimpleDateFormat
import java.util.*

class SMTAlarmService : JobIntentService() {


    override fun onHandleWork(intent: Intent) {
        val TAG = SMTAlarmService::class.java.simpleName
        lateinit var notifData: SMTNotificationData
        lateinit var payload: String
        var isFromBootReciever = false

        try {
            intent.let {
                notifData = it.getParcelableExtra(SMTScheduleNotification.PUT_EXTRA_NOTIF_DATA)
                payload = it.getStringExtra(SMTScheduleNotification.PUT_EXTRA_PAYLOAD)
                isFromBootReciever = it.getBooleanExtra(SMTScheduleNotification.PUT_EXTRA_IS_FROM_BOOT, false)
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }

        try {
            val trId = notifData.mTrid
            val scheduledTimePN = notifData.mScheduledDate
            val collapse = notifData.mCollapse

            if (trId.isNotEmpty()) {
                scheduledTimePN?.let {
                    // rowId of table is use to set unique request code for broad cast notification
                    val id = getID(collapse, notifData.notificationId, notifData.mTrid)
                    val alarmMgr = this.getSystemService(Context.ALARM_SERVICE) as AlarmManager
                    if (id > 0) {
                        val mIntent = Intent(this, SMTScheduledPNReceiver::class.java).apply {
                            putExtra(SMTScheduleNotification.NOTIFICATION_DATA, payload)
                            putExtra(SMTScheduleNotification.SOURCE_TYPE, notifData.mSourceType)
                        }

                        //check if already alarm exist with respective ids
                        val mPIntent = getPendingIntent(id, PendingIntent.FLAG_NO_CREATE)
                        var mPendingIntent: PendingIntent
                        mPendingIntent = if (mPIntent != null) {
                            Log.d("MCOLLAPSE 3", "ALARM CANCEL" + notifData.mTitle)
                            alarmMgr.cancel(getPendingIntent(id, PendingIntent.FLAG_CANCEL_CURRENT))
                            deleteOldFiles(trId, collapse)
                            updateStatusByCollapse(trId, collapse)
                            PendingIntent.getBroadcast(this, id, mIntent, PendingIntent.FLAG_UPDATE_CURRENT)
                        } else {
                            Log.d("ALARM FRESH", "CREATED PENDING INTENT")
                            PendingIntent.getBroadcast(this, id, mIntent, 0)
                        }

                        val scheduledTime = getUtcDateTime(it)
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                            Log.d("ALARM FRESH", "${notifData.mTitle} =====> ${scheduledTime.time}")
                            alarmMgr.setExact(AlarmManager.RTC_WAKEUP, scheduledTime.time, mPendingIntent)
                        } else {
                            alarmMgr.set(AlarmManager.RTC_WAKEUP, scheduledTime.time, mPendingIntent)
                        }
                    }
                }
            }
        } catch (e: java.lang.Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
        if (!isFromBootReciever) {
            notifData.mTrid.let {
                SMTEventRecorder.getInstance(this).recordNotificationDelivery(it, payload, notifData.mSourceType, notifData)
            }
            SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(this)
        }
    }

    private fun updateStatusByCollapse(trId: String, collapse: String?) {
        if (collapse != null) {
            if (collapse.isNotEmpty())
                SMTDataBaseService.getInstance(WeakReference(applicationContext)).updateSchedulePNByCollapseKey(trId, collapse)
        }
    }

    private fun getID(collapse: String?, notificationId: Int, trid: String): Int {
        return if (collapse != null) {
            if (collapse.isNotEmpty()) {
                notificationId
            } else {
                SMTDataBaseService.getInstance(WeakReference(this)).findRowIdByTrid(trid)
            }
        } else {
            SMTDataBaseService.getInstance(WeakReference(this)).findRowIdByTrid(trid)
        }
    }

    /*the method deletes if there is any older scheduled notification got cancelled or updated the files are manually deleted
    * */
    private fun deleteOldFiles(trid: String, collapse: String?) {
        val mList = SMTDataBaseService.getInstance(WeakReference(this)).getScheduledNotificationWithNotCurrentTRID(trid, collapse)
        mList?.let {
            if (it.size > 0) {
                it.forEach {
                    when (it.notificationData.mNotificationType) {
                        SMTNotificationType.CAROUSEL_PORTRAIT.type, SMTNotificationType.CAROUSEL_LANDSCAPE.type -> {
                            it.notificationData.mCarouselList?.forEach {
                                it.mMediaLocalPath?.let {
                                    Log.d("CAROUSEL PATH ", it)
                                    SMTCommonUtility.deleteFile(it)
                                }
                            }
                        }

                        SMTNotificationType.AUDIO.type, SMTNotificationType.GIF.type, SMTNotificationType.BIG_IMAGE.type -> {
                            it.notificationData.mMediaLocalPath?.let {
                                Log.d("GIF_AUDIO_IMG ", it)
                                SMTCommonUtility.deleteFile(it)
                            }
                        }
                    }
                }
            }
        }
    }

    fun getPendingIntent(id: Int, flag: Int): PendingIntent? {
        return PendingIntent.getBroadcast(this, id, Intent(this, SMTScheduledPNReceiver::class.java), flag)
    }

    companion object {
        val jobId = 1001

        fun enqueueWork(context: Context, intent: Intent) {
            enqueueWork(context, SMTAlarmService::class.java, jobId, intent)
        }
    }

    private fun getUtcDateTime(scheduledTimePN: String): Date {
        var simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
        simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return simpleDateFormat.parse(scheduledTimePN)
    }

}