package com.netcore.android.notification

import android.content.Context
import android.content.Intent
import android.util.Log
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.notification.services.SMTAlarmService
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONArray
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.text.SimpleDateFormat
import java.util.*


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Handles the incoming notification
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */

class SMTScheduleNotification {
    //    val mScheduleNotification: Any
    val TAG = SMTScheduleNotification::class.java.name


    /**
     * @param context - requires to query DB
     * @param scheduledNotification - ArrayList of Scheduled Notification
     * @param isFromBootReciever - Whether Device was restarted or timezone changed or time changed
     *  Two Keys are important for scheduled notification
     *  SOURCE TYPE : PUSHAMP / FCM
     *  IS SCHEDULED PN : 1 / 0 (for Normal PN)
     *  Whenever Alarm is triggered from getExtras we fetch notification json and what source type it was as we dont get source in notification.
     */
    @Synchronized
    internal fun parseScheduledNotification(context: Context, scheduledNotification: ArrayList<String>?, sourceType: Int, isFromBootReciever: Boolean) {
        try {
            if (!scheduledNotification.isNullOrEmpty()) {
                val dbService = SMTDataBaseService.getInstance(WeakReference(context))
                for (payload in scheduledNotification) {
                    val jsonObject = JSONObject(payload)
                    jsonObject.put(SMTNotificationConstants.NOTIF_SOURCE_TYPE, sourceType)
                    jsonObject.put(SMTNotificationConstants.NOTIF_IS_SCHEDULED_PN, 1)

                    // clearing model class on new local notification
                    val mNotificationModel = getNotificationModel(jsonObject.toString(), sourceType)
                    if (mNotificationModel?.mTrid != null) {
                        if (!dbService.findNotificationWithId(mNotificationModel.mTrid, sourceType)) {
                            setNotificationInDatabase(context, jsonObject.toString(), mNotificationModel, isFromBootReciever)
                        }
                    }
                    Thread.sleep(800L)
                }
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }

    }

    /**
     * @param context - requires to query DB
     * @param mNotificationModel - Notification Model
     * @param isFromBootReciever - Whether Device was restarted or timezone changed or time changed
     *
     * Download Media and store locally and get the media path.
     */
    private fun setNotificationInDatabase(context: Context, payload: String, mNotificationModel: SMTNotificationData?, isFromBootReciever: Boolean) {
        try {

            mNotificationModel?.let {
                if (isValidScheduledDate(it)) {
                    SMTMediaDownloadManager().downloadMedia(context, it, object : SMTMediaDownloadManager.MediaDownloadListener {
                        override fun onDownloadSuccess(notification: SMTNotificationData) {
                            setMediaPathIfItHasImages(context, payload, notification, isFromBootReciever)
                        }

                        override fun onDownloadFailed(notification: SMTNotificationData) {
                            setMediaPathIfItHasImages(context, payload, notification, isFromBootReciever)
                        }
                    })
                }
            }

        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }


    /**
     * @param context - requires to query DB
     * @param mNotificationModel - Notification Model
     * @param isFromBootReciever - Whether Device was restarted or timezone changed or time changed
     * Schedule Notification Should Download the media files if it has.
     * Once downloaded we receive a media path and then only insert in db else there wont be images displayed in Notification
     */
    fun setMediaPathIfItHasImages(context: Context, payload: String, mNotificationModel: SMTNotificationData, isFromBootReciever: Boolean) {
        val mPayload = JSONObject(payload)
        when (mNotificationModel.mNotificationType) {
            SMTNotificationType.CAROUSEL_PORTRAIT.type, SMTNotificationType.CAROUSEL_LANDSCAPE.type -> {
                val mCarouselArray = JSONArray()
                mNotificationModel.mCarouselList?.forEach {
                    val mCarouselObject = JSONObject()
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_ID, it.id)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_DEEPLINK_KEY, it.imgDeeplink)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_MSG_KEY, it.imgMsg)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_TITLE_KEY, it.imgTitle)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_URL_KEY, it.imgUrl)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_DOWNLOAD_STATUS, it.mDownloadStatus)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY, it.mMediaLocalPath)
                    mCarouselArray.put(mCarouselObject)
                }

                val mKey = SMTNotificationConstants.NOTIF_CAROUSEL_KEY
                val mCustomPayload = SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY
                if (mPayload.has(mCustomPayload)) {
                    val mPPayload = mPayload.getJSONObject(mCustomPayload)
                    if (mPPayload.has(mKey)) {
                        mPPayload.remove(mKey)
                        mPPayload.put(mKey, mCarouselArray)
                    }
                }
            }
            SMTNotificationType.BIG_IMAGE.type, SMTNotificationType.AUDIO.type, SMTNotificationType.GIF.type -> {
                val mCustomPayload = SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY
                if (mPayload.has(mCustomPayload)) {
                    val mPPayload = mPayload.getJSONObject(mCustomPayload)
                    mPPayload.put(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY, mNotificationModel.mMediaLocalPath)
                    mPPayload.put(SMTNotificationConstants.NOTIF_DOWNLOAD_STATUS, mNotificationModel.mDownloadStatus)
                }
            }
        }

        checkCollapseAndInsertInDB(context, mNotificationModel, mPayload, isFromBootReciever)
    }


    /**
     * @param context - To Query DB
     * @param mNotificationModel - To get Collapse key
     * @param isFromBootReciever - Whether Device was restarted or timezone changed or time changed
     * check collapse is not empty if true the get notifier id from DB where collapse key matches
     */
    @Synchronized
    private fun checkCollapseAndInsertInDB(context: Context, mNotificationModel: SMTNotificationData, mPayload: JSONObject, isFromBootReciever: Boolean) {
        //if source coming from FCM then insert if coming from pushamp then its already inserted directly schedule the notification
        // Inserting schedule notification inside database
        val mCollapse = mNotificationModel.mCollapse
        if (!mCollapse.isNullOrEmpty()) {
            val mId = SMTCommonUtility.getNotifyId(context, mCollapse)
            mNotificationModel.notificationId = mId
        }

        insertNotification(context, mPayload.toString(), mNotificationModel, isFromBootReciever)
    }


    /**
     *inserts scheduled notification in DB
     * with status scheduled
     * @param context - Context
     * @param payload - Notification Payload
     * @param mNotificationModel - NotificationModel with MediaPath
     * @param isFromBootReciever - Whether Device was restarted or timezone changed or time changed
     * once inserted schedules the notification
     */
    @Synchronized
    private fun insertNotification(context: Context, payload: String, mNotificationModel: SMTNotificationData, isFromBootReciever: Boolean) {
        if (mNotificationModel.mScheduledDate != null && mNotificationModel.mTtl != null) {
            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            mNotificationModel.mScheduledPNStatus = SMTNotificationConstants.NOTIF_IS_SCHEDULED
            dbService.insertPNToNotificationTable(mNotificationModel.mTrid, payload, mNotificationModel.mSourceType, mNotificationModel)

            scheduleNotification(context, payload, mNotificationModel, isFromBootReciever)
        }
    }

    /**
     * scheduled local notfication
     * @param context - pass context
     * @param notifData - notification data
     * @param isFromBootReciever -journey from boot receiver true or false
     */
    private fun scheduleNotification(context: Context, payload: String, notifData: SMTNotificationData, isFromBootReciever: Boolean) {
        val intent = Intent()
        Log.d("MCOLLAPSE 2", notifData.mTitle)
        intent.putExtra(PUT_EXTRA_NOTIF_DATA, notifData)
        intent.putExtra(PUT_EXTRA_PAYLOAD, payload)
        intent.putExtra(PUT_EXTRA_IS_FROM_BOOT, isFromBootReciever)
        SMTAlarmService.enqueueWork(context, intent)
    }

    /**
     *provides a local timezone date by converting utc
     * @param scheduledTimePN - UTC Datetime
     */
    private fun getUtcDateTime(scheduledTimePN: String): Date {
        var simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
        simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return simpleDateFormat.parse(scheduledTimePN)
    }

    /**
     * Provides notification data model after parsing the notification payload
     */
    fun getNotificationModel(notifData: String, sourceType: Int): SMTNotificationData? {
        return SMTNotificationParser().parse(notifData, sourceType)
    }

    /**
     *@param context - To get instance of DB
     *on Boot or time change or timezone change the method is executed to fetch all scheduled notification whose status is 's'
     */
    fun onBootComplete(context: Context) {
        try {
            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            val mScheduledNotificationList = dbService.getScheduledNotification()
            mScheduledNotificationList?.forEach {
                if (isValidScheduledDate(it.notificationData)) {
                    scheduleNotification(context, it.payload, it.notificationData, true)
                } else {
                    SMTLogger.d(TAG, "Scheduled Notification has been Expired")
                }
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    /**
     * @param mNotificationModel - NotificationModel to get TTL and Scheduled Date of a Notification
     * before downloading the media files validate if the scheduled date is before expiry date
     * @return true or false if scheduled date is valid
     * */
    private fun isValidScheduledDate(mNotificationModel: SMTNotificationData): Boolean {
        val mScheduleDate = mNotificationModel.mScheduledDate
        val mTtl = mNotificationModel.mTtl
        return if (mScheduleDate != null && mTtl != null) {
            getUtcDateTime(mScheduleDate).before(getUtcDateTime(mTtl))
        } else {
            false
        }
    }

    /**
     * @param context - To query DB
     * @param mTrid - TRID of a Notification
     * @param status - Where clause status = 's'/'c'/'r'
     * update the schedule notification to 'r' once rendered
     */
    fun updateScheduledNotification(context: Context, mTrid: String, status: String) {
        try {
            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            dbService.updateScheduledStatus(mTrid, status)
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }
/*
    fun setSNotification(context: Context, mData: SMTNotificationData, notif: String, source: Int) {
        val mCollapse = mData.mCollapse
        val mPublishDate = mData.mPublishedTimeStamp
        val dbService = SMTDataBaseService.getInstance(WeakReference(context))
        if (isValidScheduledDate(mData)) {
            val jsonObject = JSONObject(notif)
            jsonObject.put(SMTNotificationConstants.NOTIF_SOURCE_TYPE, source)
            jsonObject.put(SMTNotificationConstants.NOTIF_IS_SCHEDULED_PN, 1)
            if (mCollapse != null && mPublishDate != null) {
                if (mCollapse.isNotEmpty()) {
                    mData.notificationId = SMTCommonUtility.getNotifyId(context, mCollapse)
                    Log.d("SCHEDULE", "ID : ${mData.notificationId}")
                    mData.mScheduledPNStatus = SMTNotificationConstants.NOTIF_IS_CANCELLED
                    dbService.insertPNToNotificationTable(mData.mTrid, jsonObject.toString(), mData.mSourceType, mData)
                } else {
                    Log.d("SCHEDULE", "ID  2: AutoGenerated")
                    dbService.insertSchedulePNToTable(mData.mTrid, jsonObject.toString(), mData.mSourceType,
                            mData.mScheduledDate!!, SMTNotificationConstants.NOTIF_IS_SCHEDULED, mData.mTtl!!)
                }
            } else {
                dbService.insertSchedulePNToTable(mData.mTrid, jsonObject.toString(), mData.mSourceType,
                        mData.mScheduledDate!!, SMTNotificationConstants.NOTIF_IS_SCHEDULED, mData.mTtl!!)
            }
            SMTEventRecorder.getInstance(context).recordNotificationDelivery(mData.mTrid, jsonObject.toString(), SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP, mData)
        } else {
            Log.d(TAG, "Date has been expired")
        }
    }*/

    companion object {
        val PUT_EXTRA_NOTIF_DATA = "notif_data"
        val PUT_EXTRA_IS_FROM_BOOT = "is_from_boot_reciever"
        val PUT_EXTRA_PAYLOAD = "payload"
        const val NOTIFICATION_DATA = "notification_data"
        const val SOURCE_TYPE = "source_type"
    }
}