package com.netcore.android.notification

import android.content.Context
import android.content.Intent
import android.content.pm.ResolveInfo
import android.util.Log
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.SMTFirebaseMessagingService
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.utility.SMTGWSource
import org.json.JSONObject

//@Suppress("PrivatePropertyName")
/**
 * Base class for receiving messages from Firebase Cloud Messaging.
 * Methods are invoked on a background thread.
 * It also handles the broadcasting the received messages or token to any other FirebaseMessagingservice registered in the application
 */
internal class SMTFbMessagingService : FirebaseMessagingService() {

    private val TAG: String = SMTFbMessagingService::class.java.simpleName

    /**
     * This method is get invoked when there a token is first generated, and again if token changes.
     */
    override fun onNewToken(token: String) {
        super.onNewToken(token)
        SMTLogger.internal(TAG, "onNewToken called and new token is : $token")
        token.let {
            // store FCM TOKEM for later usage
            SMTLogger.i(TAG, "FCM TOKEN: $token")

            var payloadHashMap = HashMap<String, Any>()
            payloadHashMap[SMTEventParamKeys.SMT_GWSOURCE] = SMTGWSource.FCM.value
            SMTNotificationUtility.getInstance().setPushToken(this, token, SMTGWSource.FCM)
        }

        notifyOnNewTokenToServices(token, applicationContext)
    }

    /**
     * This is invoked when a notification message is received
     */
    override fun onMessageReceived(remoteMessage: RemoteMessage) {

        try {
            SMTLogger.v(TAG, "New Push Notification received")

            var isFromSmartech = false
            remoteMessage.let { message ->
                message.data.let { data ->
                    SMTLogger.i(TAG, "Notification payload  $data")

                    // Handling empty object
                    val objPayload = try {
                        JSONObject(data.toString())
                    } catch (e: java.lang.Exception) {
                        SMTLogger.internal(TAG, e.localizedMessage)
                        JSONObject()
                    }

                    isFromSmartech = SMTNotificationUtility.getInstance().checkNotificationSource(objPayload)

                    SMTLogger.i(TAG, "Is Notification From Smartech: $isFromSmartech")
                    if (isFromSmartech) {
                        SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(applicationContext, data.toString(), SMTNotificationSourceType.NOTIFICATION_SOURCE_PN, false)
                    }
                }
            }

            // if the notification is not from Smartech, notify other services
            if (!isFromSmartech) {
                notifyOnMessageReceivedToServices(remoteMessage, applicationContext)
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    override fun onDeletedMessages() {
        super.onDeletedMessages()
        notifyDeletedMessagesToServices(applicationContext)
    }

    override fun onMessageSent(p0: String) {
        super.onMessageSent(p0)
        notifyOnMessageSentToServices(p0, applicationContext)
    }

    override fun onSendError(p0: String, p1: java.lang.Exception) {
        super.onSendError(p0, p1)
        notifyOnSendErrorToServices(p0, p1, applicationContext)
    }

    private fun notifyOnMessageReceivedToServices(remoteMessage: RemoteMessage?, context: Context) {
        val serviceList = getRegisteredSMTFirebaseMessagingService()
        if (serviceList.size > 0) {
            serviceList.forEach { it.onMessageReceived(context, remoteMessage) }
        }

    }

    /**
     * Notifying to any other FirebaseMessagingServices who has registered in the user's application
     */
    private fun notifyOnNewTokenToServices(token: String?, context: Context) {
        val serviceList = getRegisteredSMTFirebaseMessagingService()
        if (serviceList.size > 0) {
            serviceList.forEach { it.onNewToken(context, token) }
        }

    }

    /**
     * Notifying to any other FirebaseMessagingServices who has registered in the user's application
     */
    private fun notifyDeletedMessagesToServices(context: Context) {
        val serviceList = getRegisteredSMTFirebaseMessagingService()
        if (serviceList.size > 0) {
            serviceList.forEach { it.onDeletedMessages(context) }
        }

    }

    /**
     * Notifying to any other FirebaseMessagingServices who has registered in the user's application
     */
    private fun notifyOnMessageSentToServices(value: String?, context: Context) {
        val serviceList = getRegisteredSMTFirebaseMessagingService()
        if (serviceList.size > 0) {
            serviceList.forEach { it.onMessageSent(context, value) }
        }

    }

    /**
     * Notifying to any other FirebaseMessagingServices who has registered in the user's application
     */
    private fun notifyOnSendErrorToServices(var1: String?, var2: Exception?, context: Context) {
        val serviceList = getRegisteredSMTFirebaseMessagingService()
        if (serviceList.size > 0) {
            serviceList.forEach { it.onSendError(context, var1, var2) }
        }

    }

    /**
     * Method to get all the FirebaseMessagingServices registered in this application apart from SMTbMessagingService
     */
    private fun getRegisteredSMTFirebaseMessagingService(): MutableList<SMTFirebaseMessagingService> {

        var fbServiceList = mutableListOf<SMTFirebaseMessagingService>()
        val iterator = applicationContext.packageManager.queryIntentServices(Intent("com.google.firebase.MESSAGING_EVENT"), 0).iterator()

        while (iterator.hasNext()) {

            val localResolveInfo: ResolveInfo = iterator.next() as ResolveInfo
            if (localResolveInfo.serviceInfo?.packageName.equals(applicationContext.packageName) && "com.netcore.android.notification.SMTFbMessagingService" != localResolveInfo.serviceInfo.name) {
                var serivice = Class.forName(localResolveInfo.serviceInfo.name).newInstance()
                if (serivice is SMTFirebaseMessagingService) {
                    fbServiceList.add(serivice)
                }
            }
        }
        return fbServiceList
    }
}