package com.netcore.android.inbox.helpers

import android.content.Context
import android.content.Intent
import android.media.AudioAttributes
import android.media.AudioFocusRequest
import android.media.AudioManager
import android.media.MediaPlayer
import android.os.Build
import android.os.Handler
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.models.SMTNotificationData
import java.io.IOException

@Suppress("DEPRECATION")
internal class SMTMediaPlayer(val context: Context) : MediaPlayer.OnCompletionListener, MediaPlayer.OnPreparedListener, MediaPlayer.OnErrorListener, MediaPlayer.OnSeekCompleteListener, AudioManager.OnAudioFocusChangeListener {

    private val TAG = SMTMediaPlayer::class.java.simpleName
    var seekHandler = Handler()
    private var mediaPlayer: MediaPlayer? = null
    private var notification: SMTNotificationData? = null
    private var mIsPrepared = false
    private lateinit var audioManager: AudioManager
    private lateinit var audioFocusRequest: AudioFocusRequest
    val focusLock = Any()
    var playbackDelayed = false
    var playbackNowAuthorized = false

    override fun onCompletion(mp: MediaPlayer?) {
        mIsPrepared = false
        stopPlayer()
    }

    override fun onPrepared(mp: MediaPlayer?) {
        mIsPrepared = true
        if (notification != null) {
            play(notification!!)
        }
    }

    override fun onError(mp: MediaPlayer?, what: Int, extra: Int): Boolean {
        mIsPrepared = false
        when (what) {
            MediaPlayer.MEDIA_ERROR_NOT_VALID_FOR_PROGRESSIVE_PLAYBACK -> {
                SMTLogger.d("SMTInboxAudioPlayerService", "MEDIA ERROR NOT VALID FOR PROGRESSIVE PLAYBACK " + extra)
                notifyPlayerError(notification)
            }
            MediaPlayer.MEDIA_ERROR_SERVER_DIED -> {
                SMTLogger.d("SMTInboxAudioPlayerService", "MEDIA ERROR SERVER DIED " + extra)
                notifyPlayerError(notification)
            }
            MediaPlayer.MEDIA_ERROR_UNKNOWN -> {
                SMTLogger.d("SMTInboxAudioPlayerService", "MEDIA ERROR UNKNOWN " + extra)
                notifyPlayerError(notification)
            }

        }
        return true
    }


    fun play(notification: SMTNotificationData) {
        if (this.notification != null && mediaPlayer != null) {
            if (this.notification!!.mTrid != notification.mTrid) {
                stopPlayer()
                this.notification = notification
                initSetup(notification)
//                initMediaPlayer(notification)
            } else {
                playMedia()
            }
        } else {
            this.notification = notification
            initSetup(notification)
        }

    }

    private fun initSetup(notification: SMTNotificationData) {
        if (playbackNowAuthorized) {
            initMediaPlayer(notification)
        } else if (!playbackDelayed) {
            requestAudioFocus()
        }

    }

    private fun playMedia() {

        if (mediaPlayer != null) {
            if (mIsPrepared && !mediaPlayer!!.isPlaying) {
                mediaPlayer!!.start()
                notifyAudioPlaying()
            } else if (!mIsPrepared) {
                initMediaPlayer(notification!!)
            }
        }
    }


    fun stop(notification: SMTNotificationData) {
        if (this.notification != null && this.notification!!.mTrid == notification.mTrid) {
            stopPlayer()
//            stopPlayerService()
        }

    }


    fun stopPlayer() {

        if (mediaPlayer != null) {
            if (mIsPrepared && mediaPlayer!!.isPlaying) {
                mediaPlayer!!.stop()
                mediaPlayer!!.release()
                mediaPlayer = null
                mIsPrepared = false

            }

        }
        seekHandler.removeCallbacks(moveSeekBarThread)
        removeAudioFocus()
        notifyAudioStop()
        notification = null
    }

    fun pause(notification: SMTNotificationData) {
        if (this.notification != null && this.notification!!.mTrid == notification.mTrid) {
            pauseMedia()
        }

    }

    private fun pauseMedia() {
        if (mIsPrepared && mediaPlayer!!.isPlaying) {
            mediaPlayer!!.pause()
        }
    }

    private fun notifyAudioStop() {

        if (notification != null) {
            val intent = Intent(notification!!.mTrid)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_ACTION, SMTInboxAudioPlayerService.AUDIO_STOPPED)
            sendLocalBroadCast(intent)

        }
    }


    private fun notifyAudioPlaying() {

        if (notification != null) {
            val intent = Intent(notification!!.mTrid)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_ACTION, SMTInboxAudioPlayerService.AUDIO_PLAYING)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_PROGRESS_RATE, mediaPlayer!!.currentPosition)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_MAX_DURATION, mediaPlayer!!.duration)
            sendLocalBroadCast(intent)

        }
    }


    private fun notifyPlayerError(notification: SMTNotificationData?) {
        if (notification != null) {
            val intent = Intent(notification.mTrid)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_ACTION, SMTInboxAudioPlayerService.AUDIO_ERROR)
            sendLocalBroadCast(intent)

        }

    }

    override fun onSeekComplete(mp: MediaPlayer) {
        //Invoked indicating the completion of a seek operation.

    }


    private fun initMediaPlayer(notification: SMTNotificationData) {
        mediaPlayer = MediaPlayer()
        //Set up MediaPlayer event listeners
        mediaPlayer!!.setOnCompletionListener(this)
        mediaPlayer!!.setOnErrorListener(this)
        mediaPlayer!!.setOnPreparedListener(this)
        mediaPlayer!!.setOnSeekCompleteListener(this)
        //Reset so that the MediaPlayer is not pointing to another data source
        mediaPlayer!!.reset()
        val attributes = AudioAttributes.Builder().setLegacyStreamType(AudioManager.STREAM_MUSIC).build()
        mediaPlayer!!.setAudioAttributes(attributes)
        try {
            // Set the data source to the mediaFile location
            mediaPlayer!!.setDataSource(notification.mMediaLocalPath)

            mediaPlayer!!.prepareAsync()

            seekHandler.postDelayed(moveSeekBarThread, 100)
        } catch (e: IOException) {
            SMTLogger.e(TAG, e.message.toString())
            //notify service to stop there is player error
            notifyPlayerError(notification)
        }
    }


    @Suppress("DEPRECATION")
    private fun requestAudioFocus() {
        audioManager = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val mAudioAttributes = AudioAttributes.Builder()
                    .setUsage(AudioAttributes.USAGE_MEDIA)
                    .setContentType(AudioAttributes.CONTENT_TYPE_MUSIC)
                    .build()
//            audioFocusRequest =
//             AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN)
//                    .setAudioAttributes(mAudioAttributes)
//                    .setAcceptsDelayedFocusGain(true)
//                    .setOnAudioFocusChangeListener(...) // Need to implement listener
//            .build();

            audioFocusRequest = AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN).run {
                setAudioAttributes(mAudioAttributes)
                setAcceptsDelayedFocusGain(true)
                setOnAudioFocusChangeListener(this@SMTMediaPlayer, Handler())
                build()

            }

            val res = audioManager.requestAudioFocus(audioFocusRequest)
            synchronized(focusLock) {
                playbackNowAuthorized = when (res) {
                    AudioManager.AUDIOFOCUS_REQUEST_FAILED -> false
                    AudioManager.AUDIOFOCUS_REQUEST_GRANTED -> {
                        true
                    }
                    AudioManager.AUDIOFOCUS_REQUEST_DELAYED -> {
                        playbackDelayed = true
                        false
                    }
                    else -> false
                }
            }
            initSetup(notification!!)

        } else {
            val am = context
                    .getSystemService(Context.AUDIO_SERVICE) as AudioManager
            // Request audio focus for play back
            val result = am.requestAudioFocus(this,
                    // Use the music stream.
                    AudioManager.STREAM_MUSIC,
                    // Request permanent focus.
                    AudioManager.AUDIOFOCUS_GAIN)

            playbackNowAuthorized = result == AudioManager.AUDIOFOCUS_REQUEST_GRANTED
            if (playbackNowAuthorized) {
                initSetup(notification!!)
            } else {
                notifyPlayerError(notification)
            }
        }
    }


    override fun onAudioFocusChange(focusState: Int) {

        when (focusState) {
            AudioManager.AUDIOFOCUS_GAIN -> {

                if (playbackDelayed) {
                    synchronized(focusLock) {
                        playbackDelayed = false
                    }
                    initMediaPlayer(notification!!)
                } else {
                    if (mIsPrepared && mediaPlayer != null && !mediaPlayer!!.isPlaying) {
                        mediaPlayer!!.start()
                        mediaPlayer!!.setVolume(1.0f, 1.0f)
                    }
                }
            }

            AudioManager.AUDIOFOCUS_LOSS -> {
                // Lost focus for an unbounded amount of time: stop playback and release media player
                playbackDelayed = false
                stop(notification!!)
                playbackNowAuthorized = false
                playbackDelayed = false
                notifyAudioStop()

            }
            AudioManager.AUDIOFOCUS_LOSS_TRANSIENT -> {
                // Lost focus for a short time, but we have to stop
                // playback. We don't release the media player because playback
                // is likely to resume
                playbackDelayed = false
                if (mediaPlayer != null && mediaPlayer!!.isPlaying) pause(notification!!)
            }

            AudioManager.AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK -> {
                // Lost focus for a short time, but it's ok to keep playing
                // at an attenuated level
                if (mediaPlayer != null && mediaPlayer!!.isPlaying) mediaPlayer!!.setVolume(0.1f, 0.1f)
            }
        }

    }

    private fun removeAudioFocus() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {

            audioManager.abandonAudioFocusRequest(audioFocusRequest)

        } else {
            audioManager.abandonAudioFocus(this)
        }
        playbackNowAuthorized = false
        playbackDelayed = false
    }


    private val moveSeekBarThread = object : Runnable {
        override fun run() {
            if (mediaPlayer != null) {
                if (mediaPlayer!!.isPlaying) {
                    val mediaPos_new = mediaPlayer!!.currentPosition
                    val mediaMax_new = mediaPlayer!!.duration
                    notifyPlayingProgress(mediaPos_new, mediaMax_new)
                    seekHandler.postDelayed(this, 100) //Looping the thread after 0.1 second
                } else {
                    seekHandler.postDelayed(this, 100) //Looping the thread after 0.1 second
                }
            }
        }
    }

    private fun notifyPlayingProgress(mediaPos_new: Int, mediaMax_new: Int) {
        if (notification != null) {
            val intent = Intent(notification!!.mTrid)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_ACTION, SMTInboxAudioPlayerService.AUDIO_PROGRESS)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_PROGRESS_RATE, mediaPos_new)
            intent.putExtra(SMTInboxAudioPlayerService.AUDIO_MAX_DURATION, mediaMax_new)
            sendLocalBroadCast(intent)

        }
    }

    private fun sendLocalBroadCast(intent: Intent) {
        LocalBroadcastManager.getInstance(context).sendBroadcast(intent)
    }

    fun seekAudio(progress: Int) {

        if (mediaPlayer != null && mIsPrepared) {
            mediaPlayer!!.seekTo(progress)
        }
    }
}