package com.netcore.android.event

import android.content.Context
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.inapp.SMTInAppConstants
import com.netcore.android.inapp.model.SMTInAppRule
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTInfo
import org.json.JSONObject
import java.util.*
import kotlin.collections.HashMap

/**
 * @Description
 *
 * Creates event payload as per event id
 *
 * @author Netcore
 * @version 1.0
 * @since 25-03-2019
 */
internal object SMTEventPayloadCreator {

    /**
     * Creates event payload wrt eventId and returns the payload as HashMap
     *
     * @param context app context
     * @param eventid event id
     * @param eventName event name
     * @param smtInfo app settings info reference
     * @param customPayload if any payload has to be track along with event
     */
    internal fun createEventPayload(context: Context, eventid: Int, eventName: String?,
                                    smtInfo: SMTInfo, customPayload: HashMap<String, Any>?): HashMap<String, Any> {

        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        // for custom event use existing payloda map received
        val eventMap = HashMap<String, Any>()

        // fill map with common data
        eventMap[SMTEventParamKeys.SMT_NETWORK_MODE] = smtInfo.mNetworkInfo?.networkMode ?: ""

        // for PN events session id is not required
        if (eventid != SMTEventId.EVENT_PN_CLICKED && eventid != SMTEventId.EVENT_PN_DISMISSED && eventid != SMTEventId.EVENT_PN_DELIVERED && eventid != SMTEventId.EVENT_PN_RENDERED) {
            val storedTimestamp = preference.getLong(SMTPreferenceConstants.CURRENT_SESSION_ID)
            val sessionId = if (storedTimestamp < 0) {
                0
            } else {
                storedTimestamp
            }
            eventMap[SMTEventParamKeys.SMT_SESSION_ID] = sessionId.toString()
        }

        eventMap[SMTEventParamKeys.SMT_EVENT_ID] = eventid

        eventMap[SMTEventParamKeys.SMT_EVENT_NAME] = eventName ?: ""
        eventMap[SMTEventParamKeys.SMT_EVENT_TIME] = System.currentTimeMillis().toString()
        eventMap[SMTEventParamKeys.SMT_TIME_ZONE] = smtInfo.mDeviceInfo?.timeZone ?: ""
        eventMap[SMTEventParamKeys.SMT_SCREEN_ORIENTATION] = smtInfo.mDeviceInfo?.getUpdatedDeviceOrientation()
                ?: ""
        eventMap[SMTEventParamKeys.SMT_LATITUDE] = smtInfo.mDeviceInfo?.latitude ?: "0.0"
        eventMap[SMTEventParamKeys.SMT_LONGITUDE] = smtInfo.mDeviceInfo?.longitude ?: "0.0"
        eventMap[SMTEventParamKeys.SMT_RETRY] = 0

        // Add user identity if any
        eventMap[SMTEventParamKeys.SMT_IDENTITY] = if (preference.getString(SMTPreferenceConstants.SMT_USER_IDENTITY).isNotEmpty()) {
            preference.getString(SMTPreferenceConstants.SMT_USER_IDENTITY)
        } else {
            ""
        }


        // as per eventId fill event specific params
        when (eventid) {
            // BASIC APP EVENTS
            SMTEventId.EVENT_APP_INSTALLED -> {

            }
            SMTEventId.EVENT_APP_UPDATED,
            SMTEventId.EVENT_FIRST_APP_LAUNCHED,
            SMTEventId.EVENT_APP_REINSTALLED,
            SMTEventId.EVENT_APP_CRASHED,
            SMTEventId.EVENT_LOCATION_FETCH_ENABLED,
            SMTEventId.EVENT_LOCATION_FETCH_DISABLED -> {
            }

            SMTEventId.EVENT_APP_LAUNCHED -> {
                addStoredAttributedParamsToPayload(context, eventMap)
                checkLateBind(context, eventMap)
            }
            SMTEventId.EVENT_DEVICE_DETAILS_UPDATED -> {
//                addDeviceDetailsupdatedParams(context, eventMap, smtInfo, preference)
            }

            // PUSH TOKEN EVENTS
            SMTEventId.EVENT_PN_TOKEN_GENERATED,
            SMTEventId.EVENT_PN_TOKEN_FAILED,
            SMTEventId.EVENT_PN_TOKEN_REFRESHED -> {
                eventMap[SMTEventParamKeys.SMT_PUSH_TOKEN_OLD] = customPayload?.get(SMTEventParamKeys.SMT_PUSH_TOKEN_OLD)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_PUSH_TOKEN_CURRENT] = customPayload?.get(SMTEventParamKeys.SMT_PUSH_TOKEN_CURRENT)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_GWSOURCE] = customPayload?.get(SMTEventParamKeys.SMT_GWSOURCE)
                        ?: ""
            }

            //PUSH NOTIFICATION EVENTS
            SMTEventId.EVENT_PN_DISMISSED,
            SMTEventId.EVENT_PN_RENDERED -> {
                eventMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = customPayload?.get(SMTEventParamKeys.SMT_IS_AMPLIFIED)
                        ?: 0
                eventMap[SMTEventParamKeys.SMT_TRID] = customPayload?.get(SMTEventParamKeys.SMT_TRID)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_GWSOURCE] = customPayload?.get(SMTEventParamKeys.SMT_GWSOURCE)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_CT] = customPayload?.get(SMTEventParamKeys.SMT_CT)
                        ?: 0
                val pnMeta = customPayload?.get(SMTEventParamKeys.SMT_PNMETA)
                        ?: JSONObject().toString()
                eventMap[SMTEventParamKeys.SMT_PNMETA] = JSONObject(pnMeta as String)
            }
            SMTEventId.EVENT_PN_DELIVERED -> {
                eventMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = customPayload?.get(SMTEventParamKeys.SMT_IS_AMPLIFIED)
                        ?: 0
                eventMap[SMTEventParamKeys.SMT_TRID] = customPayload?.get(SMTEventParamKeys.SMT_TRID)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_GWSOURCE] = customPayload?.get(SMTEventParamKeys.SMT_GWSOURCE)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_CT] = customPayload?.get(SMTEventParamKeys.SMT_CT)
                        ?: 0
                val pnMeta = customPayload?.get(SMTEventParamKeys.SMT_PNMETA)
                        ?: JSONObject().toString()
                eventMap[SMTEventParamKeys.SMT_PNMETA] = JSONObject(pnMeta as String)
            }

            SMTEventId.EVENT_PN_CLICKED -> {
                addStoredAttributedParamsToPayload(context, eventMap)
                eventMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = customPayload?.get(SMTEventParamKeys.SMT_IS_AMPLIFIED)
                        ?: 0
                eventMap[SMTEventParamKeys.SMT_TRID] = customPayload?.get(SMTEventParamKeys.SMT_TRID)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_APN_CLICK_LINK] = customPayload?.get(SMTEventParamKeys.SMT_APN_CLICK_LINK)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_CT] = customPayload?.get(SMTEventParamKeys.SMT_CT)
                        ?: 0
                eventMap[SMTEventParamKeys.SMT_GWSOURCE] = customPayload?.get(SMTEventParamKeys.SMT_GWSOURCE)
                        ?: ""
                val pnMeta = customPayload?.get(SMTEventParamKeys.SMT_PNMETA)
                        ?: JSONObject().toString()
                eventMap[SMTEventParamKeys.SMT_PNMETA] = JSONObject(pnMeta as String)
            }
            //IN APP EVENTS
            SMTEventId.EVENT_INAPP_VIEWED -> {
                addStoredAttributedParamsToPayload(context, eventMap)
                addInAppRuleParametersToPayload(eventMap, customPayload)

            }
            SMTEventId.EVENT_INAPP_DISMISSED -> {
                addStoredAttributedParamsToPayload(context, eventMap)
                addInAppRuleParametersToPayload(eventMap, customPayload)
                eventMap.remove(SMTEventParamKeys.SMT_CG_REPEAT)
                eventMap.remove(SMTEventParamKeys.SMT_CG_CONTROL_GROUP)
                eventMap.remove(SMTEventParamKeys.SMT_CG_RANDOM_NO)
            }
            SMTEventId.EVENT_INAPP_CLICKED -> {
                addInAppRuleParametersToPayload(eventMap, customPayload)
                addAttributedParamsToPayload(eventMap, customPayload)
                addStoredAttributedParamsToPayload(context, eventMap)
                eventMap.remove(SMTEventParamKeys.SMT_CG_REPEAT)
                eventMap.remove(SMTEventParamKeys.SMT_CG_CONTROL_GROUP)
                eventMap.remove(SMTEventParamKeys.SMT_CG_RANDOM_NO)
                eventMap[SMTEventParamKeys.SMT_IN_APP_CLICK_LINK] = customPayload?.get(SMTEventParamKeys.SMT_IN_APP_CLICK_LINK)
                        ?: ""
            }
            // APP INBOX EVENTS
            SMTEventId.EVENT_INBOX_DELIVERED,
            SMTEventId.EVENT_INBOX_VIEWED,
            SMTEventId.EVENT_INBOX_DISMISSED -> {
                eventMap[SMTEventParamKeys.SMT_TRID] = customPayload?.get(SMTEventParamKeys.SMT_TRID)
                        ?: ""
            }
            SMTEventId.EVENT_INBOX_CLICKED -> {
                eventMap[SMTEventParamKeys.SMT_TRID] = customPayload?.get(SMTEventParamKeys.SMT_TRID)
                        ?: ""
                eventMap[SMTEventParamKeys.SMT_INBOX_CLICK_LINK] = customPayload?.get(SMTEventParamKeys.SMT_INBOX_CLICK_LINK)
                        ?: ""
                addAttributedParamsToPayload(eventMap, customPayload)
            }
            // USER LOGIN/OUT EVENTS
            SMTEventId.EVENT_USER_LOGGED_IN -> {
                addStoredAttributedParamsToPayload(context, eventMap)
            }
            SMTEventId.EVENT_USER_LOGGED_OUT -> {
                addStoredAttributedParamsToPayload(context, eventMap)
                eventMap[SMTEventParamKeys.SMT_CLEAR_IDENTITY] = customPayload?.get(SMTEventParamKeys.SMT_CLEAR_IDENTITY)
                        ?: false
            }
            // OPT IN OUT EVENTS
            SMTEventId.EVENT_USER_DISABLED_PN,
            SMTEventId.EVENT_USER_ENABLED_PN,
            SMTEventId.EVENT_USER_TRACKING_OPT_IN,
            SMTEventId.EVENT_USER_TRACKING_OPT_OUT,
            SMTEventId.EVENT_USER_PN_OPT_IN,
            SMTEventId.EVENT_USER_PN_OPT_OUT,
            SMTEventId.EVENT_USER_INAPP_OPT_IN,
            SMTEventId.EVENT_USER_INAPP_OPT_OUT -> {
            }

            // Profile push
            SMTEventId.EVENT_USER_PROFILE_PUSH -> {
                eventMap[SMTEventParamKeys.SMT_PAYLOAD] = if (customPayload != null) {
                    JSONObject(customPayload)
                } else {
                    ""
                }
            }

            //Custom event
            SMTEventId.EVENT_CUSTOM -> {
                addStoredAttributedParamsToPayload(context, eventMap)
                addInAppParametersToPayloadCustom(context, eventMap)
                eventMap[SMTEventParamKeys.SMT_PAYLOAD] = if (customPayload != null) {
                    JSONObject(customPayload)
                } else {
                    ""
                }
            }

            //Hansel events
            SMTEventId.EVENT_NH_PROMPT_DISMISS_ID -> {
                eventMap[SMTEventParamKeys.SMT_PAYLOAD] = if (customPayload != null) {
                    JSONObject(customPayload)
                } else {
                    ""
                }
            }
            SMTEventId.EVENT_NH_PROMPT_SHOW_ID -> {
                eventMap[SMTEventParamKeys.SMT_PAYLOAD] = if (customPayload != null) {
                    JSONObject(customPayload)
                } else {
                    ""
                }
            }
            SMTEventId.EVENT_NH_BRANCH_TRACKER_ID -> {
                eventMap[SMTEventParamKeys.SMT_PAYLOAD] = if (customPayload != null) {
                    JSONObject(customPayload)
                } else {
                    ""
                }
            }

        }

        return eventMap
    }

    private fun addInAppParametersToPayloadCustom(context: Context, eventMap: HashMap<String, Any>) {

        // CG value
        val cgValue: String = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_CG, "")
        if (cgValue.isNotEmpty()) {
            eventMap[SMTEventParamKeys.SMT_CG] = cgValue
        }

        val midValue: String = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_MID, "")
        if (midValue.isNotEmpty()) {
            eventMap[SMTEventParamKeys.SMT_MID] = midValue.toInt()
        }
    }

    private fun addInAppRuleParametersToPayload(eventMap: HashMap<String, Any>, customPayload: HashMap<String, Any>?) {

        eventMap[SMTEventParamKeys.SMT_MID] = customPayload?.get(SMTEventParamKeys.SMT_MID)
                ?: ""
        eventMap[SMTEventParamKeys.SMT_CG_REPEAT] = customPayload?.get(SMTEventParamKeys.SMT_CG_REPEAT)
                ?: 0
        eventMap[SMTEventParamKeys.SMT_CG] = customPayload?.get(SMTEventParamKeys.SMT_CG).toString()
        eventMap[SMTEventParamKeys.SMT_CG_RANDOM_NO] = customPayload?.get(SMTEventParamKeys.SMT_CG_RANDOM_NO).toString()
        eventMap[SMTEventParamKeys.SMT_CG_CONTROL_GROUP] = customPayload?.get(SMTEventParamKeys.SMT_CG_CONTROL_GROUP).toString()
    }

    /**
     * retrieves attributed params from custompayload and inserts into eventmap
     */
    private fun addAttributedParamsToPayload(eventMap: HashMap<String, Any>, customPayload: HashMap<String, Any>?) {


        customPayload?.let {
            if (it.isNotEmpty()) {

                if (customPayload.containsKey(SMTNotificationConstants.NOTIF_ATTRIBUTION_ID)) {
                    try {
                        eventMap[SMTEventParamKeys.SMT_ATCI] = (customPayload[SMTNotificationConstants.NOTIF_ATTRIBUTION_ID].toString()).toInt()
                    } catch (e: Exception) {
                    }
                }

                if (customPayload.containsKey(SMTNotificationConstants.NOTIF_ATTRIBUTION_SOURCE)) {
                    eventMap[SMTEventParamKeys.SMT_ATCS] = customPayload[SMTNotificationConstants.NOTIF_ATTRIBUTION_SOURCE]
                            ?: ""
                }

                if (customPayload.containsKey(SMTNotificationConstants.NOTIF_ATTRIBUTION_MEDIUM)) {
                    eventMap[SMTEventParamKeys.SMT_ATCM] = customPayload[SMTNotificationConstants.NOTIF_ATTRIBUTION_MEDIUM]
                            ?: ""
                }

                //eventMap[SMTEventParamKeys.SMT_ATCA] = customPayload?.get(SMTEventParamKeys.SMT_ATCA) ?: ""
            }
            // As per discussion with product team, we don't need to send utm parameters in
            // other parameter node
            //eventMap[SMTEventParamKeys.SMT_AT_OTHER] = customPayload[SMTEventParamKeys.SMT_AT_OTHER]
            //        ?: ""
        }


    }

    /**
     * retrieves attributed params from shared preference and inserts into eventmap
     */
    private fun addStoredAttributedParamsToPayload(context: Context, eventMap: HashMap<String, Any>) {

        val attrParams = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getString(SMTPreferenceConstants.SMT_ATTRIBUTION_PARAMS)

        // Pass attribute params to payload
        eventMap[SMTEventParamKeys.SMT_ATTRIBUTE_PARAMS] = when (attrParams.isNotEmpty()) {
            true -> JSONObject(attrParams)
            false -> JSONObject()
        }
    }

    /**
     * This function will check weather its a late bind or not based on the status of flag
     * @param context app contex
     * @param eventMap Hashmap to be filled
     */
    private fun checkLateBind(context: Context, eventMap: HashMap<String, Any>) {

        val prefrence = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val oldIdentity = prefrence.getString(SMTPreferenceConstants.SMT_USER_OLD_IDENTITY, "")
        val currentIdentity = prefrence.getString(SMTPreferenceConstants.SMT_USER_IDENTITY, "")

        eventMap[SMTEventParamKeys.SMT_LATE_BIND] = when (oldIdentity != currentIdentity) {
            true -> {
                prefrence.setString(SMTPreferenceConstants.SMT_USER_OLD_IDENTITY, currentIdentity)
                true
            }
            false -> false
        }
        SMTLogger.v("SMTEventPayload", "APP Launch status  ${eventMap[SMTEventParamKeys.SMT_LATE_BIND]}")
    }

    fun createMultiEventPayload(multiEventsRules: SMTInAppRule.Rules): HashMap<String, Any> {
        val eventMap = HashMap<String, Any>()
        val eventName = multiEventsRules.eventName
        eventMap[SMTEventParamKeys.SMT_EVENT_ID] = when {
            eventName.equals(SMTInAppConstants.SystemEvents.APP_LAUNCH.value, ignoreCase = true) -> {
                SMTEventId.EVENT_APP_LAUNCHED
            }
            eventName.equals(SMTInAppConstants.SystemEvents.FIRST_APP_LAUNCH.value, ignoreCase = true) -> {
                SMTEventId.EVENT_FIRST_APP_LAUNCHED
            }
            else -> {
                0
            }
        }
        // fill map with common data
        eventMap[SMTEventParamKeys.SMT_EVENT_NAME] = eventName.toLowerCase(Locale.getDefault())
        eventMap[SMTEventParamKeys.SMT_EVENT_TIME] = multiEventsRules.eventTime
        return eventMap
    }

    /**
     * adds device detail updated params
    //     * @param context app context
    //     * @param eventMap hashmap to be filled
    //     * @param smtInfo app settings info reference
    //     * @param preference Shared preference reference
     */
    /*private fun addDeviceDetailsupdatedParams(context: Context, eventMap: HashMap<String, Any>, smtInfo: SMTInfo, preference: SMTPreferenceHelper) {
        eventMap[SMTEventParamKeys.SMT_CARRIER] = smtInfo.mNetworkInfo?.carrier ?: ""
        eventMap[SMTEventParamKeys.SMT_COUNTRY_CODE] = smtInfo.mNetworkInfo?.countryCode ?: ""
        eventMap[SMTEventParamKeys.SMT_RADIO] = smtInfo.mNetworkInfo?.radio ?: ""
        eventMap[SMTEventParamKeys.SMT_VENDOR_ID] = ""
        eventMap[SMTEventParamKeys.SMT_DEVICE_WIDTH] = smtInfo.mDeviceInfo?.deviceWidth ?: ""
        eventMap[SMTEventParamKeys.SMT_DEVICE_HEIGHT] = smtInfo.mDeviceInfo?.deviceHeight ?: ""
        eventMap[SMTEventParamKeys.SMT_USE_ADV_ID] = preference.getInt(SMTPreferenceConstants.SMT_MF_IS_USE_AD_ID)
        eventMap[SMTEventParamKeys.SMT_DEVICE_UID] = smtInfo.mDeviceInfo?.guid ?: ""
        eventMap[SMTEventParamKeys.SMT_BOD] = SMTCommonUtility.getBOD(context)
        eventMap[SMTEventParamKeys.SMT_NETWORK_MODE] = smtInfo.mNetworkInfo?.networkMode ?: ""
    }*/
}