package com.netcore.android.db

import android.content.Context
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.inapp.model.SMTEvents
import com.netcore.android.inapp.model.SMTInAppRule
import com.netcore.android.inbox.utility.SMTInboxMessageStatus
import com.netcore.android.inbox.utility.SMTInboxMessageType
import com.netcore.android.network.models.SMTInboxResponse
import com.netcore.android.notification.SMTSchedulePNData
import com.netcore.android.notification.models.SMTNotificationData
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * SingleTon class Extends SQLiteOpenHelper class
 *
 *@param context - Weakrefernce of the context
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTDataBaseService private constructor(val context: WeakReference<Context>) {

    companion object {
        private var mSMTDatabase: SMTDatabase? = null

        @Volatile
        private var INSTANCE: SMTDataBaseService? = null

        /**
         * Getting instance of the class
         */
        fun getInstance(context: WeakReference<Context>): SMTDataBaseService =
                INSTANCE ?: synchronized(SMTDataBaseService::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: WeakReference<Context>): SMTDataBaseService {
            context.get()?.let {
                mSMTDatabase = SMTDatabase.getInstance(it)
            }
            return SMTDataBaseService(context)
        }

    }


    /**
     * Insert a single notification to the notification table
     * @param trid notification transaction id
     * @param payload notification payload
     * @param source notification source like PN or PushAmp
     */
    fun insertPNToNotificationTable(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int) {
        mSMTDatabase?.getNotificationTable()?.insertNotification(trid, payload, source)
    }


    fun insertPNToNotificationTable(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int, smtNotificationData: SMTNotificationData) {
        mSMTDatabase?.getNotificationTable()?.insertNotification(trid, payload, source, smtNotificationData)
    }

    /**
     * Insert a single notification to the notification table
     * @param trid notification transaction id
     * @param payload notification payload
     * @param source notification source like PN or PushAmp
     */
    fun insertSchedulePNToTable(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int, scheduledDate: String, scheduledPNStatus: String, ttl: String) {
        mSMTDatabase?.getNotificationTable()?.insertScheduledPN(trid, payload, source, scheduledDate, scheduledPNStatus, ttl)
    }

    fun updateSchedulePNToTable(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int, modifiedDate: String, scheduledPNStatus: String) {
        mSMTDatabase?.getNotificationTable()?.updateScheduledPN(trid, payload, source, modifiedDate, scheduledPNStatus)
    }

    fun getScheduledPNPayload(trid: String) {
        mSMTDatabase?.getNotificationTable()?.getScheduledPNPayload(trid)
    }

    /**
     * Update notification status in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnName - columname which to be updated
     * @param columnValue - value to be updated
     */
    fun updateNotification(trid: String, columnName: String, columnValue: Boolean) {
        mSMTDatabase?.getNotificationTable()?.updateNotification(trid, columnName, columnValue)
    }

    /**
     * Update notification Id in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnValue - notification Id
     */
    fun updateNotificationId(trid: String, columnValue: Int) {
        mSMTDatabase?.getNotificationTable()?.updateNotificationId(trid, columnValue)
    }

    /**
     * Update notification Id in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnValue - notification Id
     */
    fun getNotificationById(trid: String): SMTNotificationData? {
        return mSMTDatabase?.getNotificationTable()?.getNotificationById(trid)
    }

    /**
     * Get notification clicked status from Id in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnValue - notification Id
     * @return Boolean Notification clicked status
     */
    fun getNotificationClickedStatusById(trid: String): Boolean? {
        return mSMTDatabase?.getNotificationTable()?.getNotificationClickedStatus(trid)
    }

    /**
     * Delete the notifications after certain time interval
     * currently its 30 days of notification expire time
     * @param timeStamp - The timestamp to be checked after which notifications to be deleted
     */
    fun checkAndDeleteExpiredNotifcation(timeStamp: Long) {
        mSMTDatabase?.getNotificationTable()?.deleteNotifications(timeStamp)
    }

/*    */
    /**
     * Insert a single notification to the Event table
     * //@param eventId event Id
    //     * @param eventName Event name
    //     * @param payload event payload
    //     * @param type event type
    //     * @param source notification source like PN or PushAmp
     *//*
    fun insertPNToEventTable(eventId: Int, eventName: String, payload: String, @SMTEventType.Type type: String) {
        mSMTDatabase?.getEventTable()?.insertEvent(eventId, eventName, payload, type)
    }*/

    /*
    */
    /**
     * Retrieves Array of events from DB using batch size
     * //@param batchSize size of each batch
     * @return returns arraylist of arraylist of hashmap of events
     *//*
    fun getEventsArrayForBatchProcessing(batchSize: Int): ArrayList<ArrayList<HashMap<String, String>>> {
        return mSMTDatabase?.getEventTable()?.getEventsArrayForBatchProcessing(batchSize)
                ?: arrayListOf()
    }
    */

    /**
     * Retrieves InProgress HaspMap of events from DB using batch size
     * @param batchSize size of each batch
     * @return returns hashmap of events
     */

    fun getInProgressEvents(batchSize: Int): HashMap<String, String> {
        return mSMTDatabase?.getEventTable()?.getInProgressEventsMapWithSize(batchSize)
                ?: hashMapOf()
    }

    /**
     * Retrieves HaspMap of events from DB using batch size
     * @param batchSize size of each batch
     * @param startIndex startindex of the row
     * @return returns hashmap of events
     */

    fun getEventsMapForBatchProcessing(batchSize: Int, startIndex: Int): HashMap<String, String> {
        return mSMTDatabase?.getEventTable()?.getEventsMapWithSize(batchSize, startIndex)
                ?: hashMapOf()
    }

    /**
     * Retrieves HaspMap of events from DB using batch size
     * @param batchSize size of each batch
     * @param startIndex startindex of the row
     * @return returns hashmap of events
     */

    fun getPendingEventsMapForBatchProcessing(batchSize: Int, startIndex: Int): HashMap<String, String> {
        return mSMTDatabase?.getEventTable()?.getPendingEventsMapWithSize(batchSize, startIndex)
                ?: hashMapOf()
    }

    /**
     * Retrieves HaspMap of events from DB using batch size
     * @param batchSize size of each batch
     * @param startIndex startindex of the row
     * @return returns hashmap of events
     */

    fun getPendingFailedEventsMapForBatchProcessing(batchSize: Int, startIndex: Int): HashMap<String, String> {
        return mSMTDatabase?.getEventTable()?.getPendingAndFailedEventsMapWithSize(batchSize, startIndex)
                ?: hashMapOf()
    }


    /**
     * Updated events sync status
     * @param ids event row ids which will be updated
     * @param columneName the column which needs to be updated
     * @param columnValue the value with which the column to be updated
     */
    fun updateEventProcessingStatus(ids: Array<Int>, columneName: String, columnValue: Int) {
        mSMTDatabase?.getEventTable()?.updateMultipleRowsWithSameColumnValue(ids, columneName, columnValue)
    }

    /**
     * Delete events from event table
     */
    fun deleteEventsOnBatchRequestSuccess() {
        mSMTDatabase?.getEventTable()?.deleteEventsWithInProgress()
    }

    /**
     *  Delete events based on id list
     */
    fun deleteEventsMultipleRows(ids: Array<Int>) {
        mSMTDatabase?.getEventTable()?.deleteMultipleEventRows(ids)
    }

    fun updateMultipleRowsWithSameColumnValue(columnValue: Int) {
        mSMTDatabase?.getEventTable()?.updateMultipleRowsWithSameColumnValue(columnValue)
    }

    /**
     * TO update failed events payload with retry = 1
     */
    fun updateFailedBatchPayload() {
        mSMTDatabase?.getEventTable()?.updateFailedBatchPayload()
    }

    /**
     * Checks if there is more events which needs to be processed
     * if number of events less that batchSize then dont process
     * @param batchSize size of batch
     */
    fun checkIfMoreEventsPresentForBatchRequest(batchSize: Int): Boolean {
        return mSMTDatabase?.getEventTable()?.isMoreEventsPresentForProcessing(batchSize) ?: false
    }

    /**
     *  Check if Pending and Failed events are present for
     *  processing
     *  @param batchSize size of batch
     *  @return boolean
     */
    fun checkIfPendingFailedEventsBatchRequest(batchSize: Int): Boolean {
        return mSMTDatabase?.getEventTable()?.isMoreEventsPendingFailedPresentForProcessing(batchSize)
                ?: false
    }

    /**
     * Insert a single notification to the Event table
     * @param eventId event Id
     * @param eventName Event name
     * @param payload event payload
     * @param eventType event type
     */
    fun insertEventToDb(eventId: Int, eventName: String?, payload: String, eventType: String) {
        mSMTDatabase?.getEventTable()?.insertEvent(eventId, eventName, payload, eventType)
    }

    fun getInAppRules(payloadMap: HashMap<String, Any>): MutableList<SMTInAppRule>? {

        return mSMTDatabase?.getInAppRuleTable()?.getMatchingRules(payloadMap)
    }

    fun findNotificationWithId(mTrid: String, @SMTNotificationSourceType.Source sourceType: Int): Boolean {
        return mSMTDatabase?.getNotificationTable()?.findNotificationById(mTrid, sourceType)
                ?: false
    }

    fun findRowIdByTrid(mTrid: String): Int {
        return mSMTDatabase?.getNotificationTable()?.getRowIdByTrid(mTrid) ?: 0
    }

    fun findNotificationReadStatusWithId(mTrid: String): Boolean {
        return mSMTDatabase?.getNotificationTable()?.findNotificationReadStatusById(mTrid) ?: false
    }

    fun updateInAppRuleUsage(inAppRule: SMTInAppRule, todayInMilliSec: Long) {
        mSMTDatabase?.getInAppRuleTable()?.updateInAppUsage(inAppRule, todayInMilliSec)
    }

    fun deleteOtherInAppRules(ruleIds: String?) {
        mSMTDatabase?.getInAppRuleTable()?.deleteOtherInAppRules(ruleIds)
    }

    fun insertInAppRules(inAppRules: ArrayList<SMTInAppRule>) {
        mSMTDatabase?.getInAppRuleTable()?.insertInAppRules(inAppRules)

    }

    fun resetUsageForSessionTypeRule() {
        mSMTDatabase?.getInAppRuleTable()?.resetUsageForSessionTypeRule()
    }

    /**
     * Used for test case classes
     */
    internal fun resetInAppTableData(tableName: String) {
        mSMTDatabase?.getInAppRuleTable()?.deleteTableData(tableName)
    }

    /**
     * Update notification status in notification table
     * @param ruleId - Rule ID, which to be updated
     * @param columnName - columname which to be updated
     * @param columnValue - value to be updated
     */
    internal fun updateInAppTable(ruleId: String, columnName: String, columnValue: Int) {
        mSMTDatabase?.getInAppRuleTable()?.updateInApp(ruleId, columnName, columnValue)
    }

    /**
     * Used for test case classes
     */
    internal fun closeDB() {
        mSMTDatabase?.close()
        mSMTDatabase = null
        INSTANCE = null
    }

    fun storeInboxNotificaiton(item: SMTInboxResponse.InboxResponse) {
        mSMTDatabase?.getInboxTable()?.insertAppInboxMessage(item.trid, item.payload, item.status, item.timeStamp)
    }

    fun clearInboxTable() {
        mSMTDatabase?.getInboxTable()?.clearInboxTable()
    }

    internal fun getInboxMessages(@SMTInboxMessageType.Type type: Int): MutableList<SMTNotificationData>? {
        return mSMTDatabase?.getInboxTable()?.getInboxMessages(type)
    }

    internal fun getInboxMessageCount(@SMTInboxMessageType.Type messageType: Int): Int {
        return mSMTDatabase?.getInboxTable()?.getInboxMessageCount(messageType) ?: 0
    }


    internal fun getInboxMessageById(trid: String): SMTNotificationData? {
        return mSMTDatabase?.getInboxTable()?.getInboxMessageById(trid)
    }

    internal fun updateInboxMessageStatus(trid: String, @SMTInboxMessageStatus.Status status: Int): Boolean {
        return mSMTDatabase?.getInboxTable()?.updateInboxMessageStatus(trid, status) ?: false
    }

    fun updateInboxMessagePayload(notification: SMTNotificationData) {
        mSMTDatabase?.getInboxTable()?.updateInboxMessagePayload(notification)
    }

    fun getScheduledNotification(): ArrayList<SMTSchedulePNData>? {
        return mSMTDatabase?.getNotificationTable()?.getScheduledNotification()
    }

    fun updateScheduledStatus(trid: String, status: String) {
        mSMTDatabase?.getNotificationTable()?.updateScheduledPNStatus(trid, status)
    }

    fun getScheduledNotificationForDemoApp(): ArrayList<SMTSchedulePNData>? {
        return mSMTDatabase?.getNotificationTable()?.getScheduledNotificationForDemoApp()
    }

    //Multievents methods
    fun getStoredEventData(payloadMap: HashMap<String, Any>): MutableList<SMTEvents>? {
        return mSMTDatabase?.getEventTable()?.getStoredEventData(payloadMap)
    }

    fun deleteEvents(limit: Int) {
        mSMTDatabase?.getEventTable()?.deleteMultipleEvents(limit)
    }

    fun findByCollapseKey(collapseKey: String): Int {
        return mSMTDatabase?.getNotificationTable()?.getNotifierIdByCollapseKey(collapseKey)!!
    }

    fun getUpdatedPNGroupByCollapseKey(mStrCollapse: String): ArrayList<SMTNotificationData>? {
        return mSMTDatabase?.getNotificationTable()?.getUpdatedPN(mStrCollapse)
    }

    fun getScheduledNotificationWithNotCurrentTRID(trid: String, collapse: String?): ArrayList<SMTSchedulePNData>? {
        return mSMTDatabase?.getNotificationTable()?.getScheduledNotificationWithNotCurrentTRID(trid, collapse)
    }

    fun updateSchedulePNByCollapseKey(trid: String, collapse: String) {
        mSMTDatabase?.getNotificationTable()?.updateSchedulePNByCollapseKey(trid, collapse)
    }


}