package com.netcore.android.notification.services

import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.core.app.JobIntentService
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTScheduleNotification
import com.netcore.android.notification.SMTScheduledPNReceiver
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.workmgr.SMTScheduledPNWorker
import com.netcore.android.workmgr.SMTWorkerScheduler
import java.lang.ref.WeakReference
import java.text.SimpleDateFormat
import java.util.*

class SMTAlarmService : JobIntentService() {
    override fun onHandleWork(intent: Intent) {
        val TAG = SMTAlarmService::class.java.simpleName
        lateinit var notifData: SMTNotificationData
        lateinit var payload: String
        var isFromBootReciever = false

        try {
            intent.let {
                notifData = it.getParcelableExtra(SMTScheduleNotification.PUT_EXTRA_NOTIF_DATA)
                payload = it.getStringExtra(SMTScheduleNotification.PUT_EXTRA_PAYLOAD)
                isFromBootReciever = it.getBooleanExtra(SMTScheduleNotification.PUT_EXTRA_IS_FROM_BOOT, false)
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }

        try {
            val trId = notifData.mTrid
            val scheduledTimePN = notifData.mScheduledDate

            if (trId.isNotEmpty()) {
                scheduledTimePN?.let {
                    // rowId of table is use to set unique request code for broad cast notification
                    val rowId = SMTDataBaseService.getInstance(WeakReference(this)).findRowIdByTrid(trId)
                    val alarmMgr = this.getSystemService(Context.ALARM_SERVICE) as AlarmManager
                    if (rowId > 0) {
                        val mIntent = Intent(this, SMTScheduledPNReceiver::class.java).apply {
                            putExtra(SMTScheduledPNWorker.NOTIFICATION_DATA, payload)
                            putExtra(SMTScheduledPNWorker.SOURCE_TYPE, notifData.mSourceType)
                        }
                        val mPendingIntent = PendingIntent.getBroadcast(this, rowId, mIntent, 0)
                        val scheduledTime = getUtcDateTime(it)

                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                            alarmMgr.setExact(AlarmManager.RTC_WAKEUP, scheduledTime.time, mPendingIntent)
                        } else {
                            alarmMgr.set(
                                    AlarmManager.RTC_WAKEUP,
                                    scheduledTime.time,
                                    mPendingIntent
                            )
                        }
                    }
                }
            }
        } catch (e: java.lang.Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
        if (!isFromBootReciever) {
            notifData.mTrid.let {
                SMTEventRecorder.getInstance(this).recordNotificationDelivery(it, payload, notifData.mSourceType, notifData)
            }
            SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(this)
        }
    }

    companion object {
        val jobId = 1001

        fun enqueueWork(context: Context, intent: Intent) {
            enqueueWork(context, SMTAlarmService::class.java, jobId, intent)
        }
    }

    private fun getUtcDateTime(scheduledTimePN: String): Date {
        var simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
        simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return simpleDateFormat.parse(scheduledTimePN)
    }

}