package com.netcore.android.db

import android.content.ContentValues
import android.content.Context
import android.database.Cursor
import android.database.sqlite.SQLiteDatabase
import com.netcore.android.logger.SMTLogger
import java.lang.ref.WeakReference

@Suppress("PrivatePropertyName")
/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * SingleTon class Database wrapper class
 * All database operations to be called through this wrapper
 *
 *@param context - Weakrefernce of the context
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTDataBaseWrapper private constructor(private val context: WeakReference<Context>) {

    private val TAG = SMTDataBaseWrapper::class.java.simpleName

    companion object {
        private var mDatabase: SQLiteDatabase? = null

        @Volatile
        private var INSTANCE: SMTDataBaseWrapper? = null

        /**
         * Get method for creating singleton instance of the class
         * @param context - App context
         * @return SMTDatabase - Returns Database wrapper instance
         */
        fun getInstance(context: Context): SMTDataBaseWrapper =
                INSTANCE ?: synchronized(SMTDataBaseWrapper::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        /**
         * Buils the instance of SMTDatabase
         *
         * @param context - App context
         * @return SMTDatabase - Returns Database wrapper instance
         */
        private fun buildInstance(context: Context): SMTDataBaseWrapper {
            return SMTDataBaseWrapper(WeakReference(context))
        }
    }

    fun setDatabaseInstance(mDb: SQLiteDatabase) {
        mDatabase = mDb
    }

    /**
     * Begins a transaction in EXCLUSIVE mode.
     */
    fun beginTransaction() {
        mDatabase?.beginTransaction()
    }

    /**
     * Marks the current transaction as successful.
     */
    fun setTransactionSuccessful() {
        mDatabase?.setTransactionSuccessful()
    }

    /**
     * End a transaction.
     */
    fun endTransaction() {
        try {
            mDatabase?.endTransaction()
        } catch (ex: Exception) {
            SMTLogger.e(TAG, "Database full, unable to endTransaction, error - $ex")
        }
    }

    /**
     * Runs the provided SQL and returns a Cursor over the result set.
     * @param sql The SQL query. The SQL string must not be ; terminated
     * @param args You may include ?s in where clause in the query, which will be replaced by the values from selectionArgs
     *
     * @return Cursor - Result set will be returned through cursor
     */
    fun rawQuery(sql: String, args: Array<String>?): Cursor? {
        return mDatabase?.rawQuery(sql, args)
    }

    /**
     * Convenience method for updating rows in the database
     *
     * @param table The table to update in
     * @param values A map from column names to new column values
     * @param selection The optional WHERE clause to apply when updating. Passing null will update all rows.
     * @param selectionArgs You may include ?s in the where clause, which will be replaced by the values from whereArgs.
     *
     * @return Int - the number of rows affected
     */
    fun update(table: String, values: ContentValues,
               selection: String, selectionArgs: Array<String>?): Int {

        var count = 0
        try {
            count = mDatabase?.update(table, values, selection, selectionArgs) ?: 0
        } catch (ex: Exception) {
            SMTLogger.e(TAG, "Database full, unable to update, error - $ex")
        }
        return count
    }

    /**
     * Method for deleting rows in the database.
     *
     * @param table The table to delete from
     * @param whereClause The optional WHERE clause to apply when deleting. Passing null will delete all rows.
     * @param whereArgs You may include ?s in the where clause, which will be replaced by the values from whereArgs.
     *
     * @return Int the number of rows affected
     */
    fun delete(table: String, whereClause: String?, whereArgs: Array<String>?): Int {

        var count = 0
        try {
            count = mDatabase?.delete(table, whereClause, whereArgs) ?: 0
        } catch (ex: Exception) {
            SMTLogger.e(TAG, "Database full, unable to delete, error - $ex")
        }
        return count
    }

    /**
     * Mmethod for inserting a row into the database.
     *
     * @param table The table to insert the row into
     * @param nullColumnHack parameter provides the name of nullable column name to explicitly insert a NULL into in the case where your values is empty.
     * @param values  this map contains the initial column values for the row. The keys should be the column names and the values the column values.
     *
     * @return Long the row ID of the newly inserted row, or -1 if an error occurred
     */
    fun insert(table: String, nullColumnHack: String?,
               values: ContentValues): Long {

        var rowId: Long = -1
        try {
            rowId = mDatabase?.insert(table, nullColumnHack, values) ?: -1
        } catch (ex: Exception) {
            SMTLogger.e(TAG, "Database full, unable to insert, error $ex")
        }
        return rowId
    }

    /**
     * Executes SQL statement
     * @param sql SQL statement to be executed. Multiple statements separated by semicolons are not supported.
     */
    fun execSQL(sql: String) {

        try {
            mDatabase?.execSQL(sql)
        } catch (ex: Exception) {
            SMTLogger.e(TAG, "Database full, unable to execSQL, error $ex")
        }
    }

    /**
     * Return the Database reference
     *
     * @return SQLiteDatabse refernce returned
     */
    fun getDatabase(): SQLiteDatabase? {
        return mDatabase
    }
}