package com.netcore.android.inbox.views

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.ProgressBar
import android.widget.TextView
import com.netcore.android.R
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.SMTNotificationType
import com.netcore.android.notification.models.SMTCarouselItemData
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility

internal class SMTInboxCarouselView(context: Context, attrs: AttributeSet?) : SMTBaseView(context, attrs) {

    private var mCurrentIndex = 0
    private lateinit var landscapeView: View
    private lateinit var portraitView: View
    private var isPortrait: Boolean = false

    override fun setNotificationData(notification: SMTNotificationData) {
        mCurrentIndex = 0
        isPortrait = false
        setData(notification)
        if (mNotificationData.mCarouselList != null && mNotificationData.mCarouselList!!.size > 0) {

            if (notification.mNotificationType == SMTNotificationType.CAROUSEL_PORTRAIT.type) {
                isPortrait = true
                setPortraitView()
            } else {
                isPortrait = false
                setLandscapeView()
            }
        } else {
            SMTLogger.e("SMTInboxCarouselView", "No carousel item to display")
        }

        fetchImage()
    }

    private fun setLandscapeView() {
        landscapeView = LayoutInflater.from(context).inflate(R.layout.inbox_carousel_landscape_layout, this)
        setBackgroundShape()

        setLandscapeLayoutAction()

        landscapeView.findViewById<TextView>(R.id.tv_landscape_carousel_title).text = mNotificationData.mTitle
        landscapeView.findViewById<TextView>(R.id.tv_landscape_carousel_content).text = mNotificationData.mMessage
        landscapeView.findViewById<TextView>(R.id.tv_landscape_carousel_time).text = SMTCommonUtility.getFormattedTimeDifference(mNotificationData.mPublishedTimeStamp)

        setLandscapeViewDetails(mNotificationData.mCarouselList!![mCurrentIndex])

        if (mNotificationData.mCarouselList!!.size > 1) {
            landscapeView.findViewById<ImageView>(R.id.iv_landscape_carousel_left_arrow).setOnClickListener {
                landscapeLeftArrowClicked()
            }
            landscapeView.findViewById<ImageView>(R.id.iv_landscape_carousel_right_arrow).setOnClickListener {
                landscapeRightArrowClicked()
            }
        } else {
            landscapeView.findViewById<ImageView>(R.id.iv_landscape_carousel_left_arrow).isEnabled = false
            landscapeView.findViewById<ImageView>(R.id.iv_landscape_carousel_right_arrow).isEnabled = false

        }
    }

    private fun setLandscapeLayoutAction() {
        landscapeView.setOnClickListener {
            handleClick(mNotificationData.mDeepLinkPath)
        }
    }

    private fun landscapeRightArrowClicked() {

        handleLeftRightClickEvent()

        when (mCurrentIndex) {
            mNotificationData.mCarouselList!!.size - 1 -> mCurrentIndex = 0
            else -> mCurrentIndex += 1
        }
        setLandscapeViewDetails(mNotificationData.mCarouselList!![mCurrentIndex])
    }

    private fun handleLeftRightClickEvent() {
        handleClick(null)
    }

    private fun landscapeLeftArrowClicked() {

        handleLeftRightClickEvent()

        when (mCurrentIndex) {
            0 -> mCurrentIndex = mNotificationData.mCarouselList!!.size - 1
            else -> mCurrentIndex -= 1
        }
        setLandscapeViewDetails(mNotificationData.mCarouselList!![mCurrentIndex])


    }

    private fun setLandscapeViewDetails(carouselItemdata: SMTCarouselItemData) {

        setBitmapImage(true, carouselItemdata, null)

        landscapeView.findViewById<TextView>(R.id.tv_landscape_carousel_image_title).text = carouselItemdata.imgTitle
        landscapeView.findViewById<TextView>(R.id.tv_landscape_carousel_image_description).text = carouselItemdata.imgMsg
    }

    private fun setPortraitView() {

        portraitView = LayoutInflater.from(context).inflate(R.layout.inbox_carousel_portrait_layout, this)

        setBackgroundShape()

        portraitView.findViewById<TextView>(R.id.tv_portrait_carousel_title).text = mNotificationData.mTitle
        portraitView.findViewById<TextView>(R.id.tv_portrait_carousel_content).text = mNotificationData.mMessage
        portraitView.findViewById<TextView>(R.id.tv_portrait_carousel_time).text = SMTCommonUtility.getFormattedTimeDifference(mNotificationData.mPublishedTimeStamp)

        var leftItem: SMTCarouselItemData? = null
        var rightItem: SMTCarouselItemData? = null

        if (mCurrentIndex < (mNotificationData.mCarouselList?.size?.minus(1)!!)) {
            leftItem = mNotificationData.mCarouselList!![mCurrentIndex]
        }
        if ((mCurrentIndex + 1) < (mNotificationData.mCarouselList?.size?.minus(1)!!)) {
            rightItem = mNotificationData.mCarouselList!![mCurrentIndex + 1]
        }

        setPortraitViewDetails(leftItem, rightItem)

        if (mNotificationData.mCarouselList!!.size > 2) {
            portraitView.findViewById<ImageView>(R.id.iv_portrait_left_carousel_arrow).setOnClickListener {
                portraitLeftArrowClicked()
            }
            portraitView.findViewById<ImageView>(R.id.iv_portrait_right_carousel_arrow).setOnClickListener {
                portraitRightArrowClicked()
            }
        } else {
            portraitView.findViewById<ImageView>(R.id.iv_portrait_left_carousel_arrow).isEnabled = false
            portraitView.findViewById<ImageView>(R.id.iv_portrait_right_carousel_arrow).isEnabled = false

        }

    }

    private fun setPortraitLayoutAction(leftItem: SMTCarouselItemData?, rightItem: SMTCarouselItemData?) {
        portraitView.setOnClickListener {
            handleClick(mNotificationData.mDeepLinkPath)
        }

        leftItem?.let { left ->
            portraitView.findViewById<LinearLayout>(R.id.ll_portrait_left_item_layout).setOnClickListener {
                handleClick(left.imgDeeplink)
            }
        }

        rightItem?.let { right ->
            portraitView.findViewById<LinearLayout>(R.id.ll_portrait_right_carousel_layout).setOnClickListener {
                handleClick(right.imgDeeplink)
            }
        }
    }

    private fun fetchImage() {
        val pair = getCarouselItems(mNotificationData)
        pair?.let {
            setBitmapImage(false, it.first, it.second)
        }

        if (!hasDownloadedImages()) {
            showProgressBar()
            downloadImage()
        } else {
            pair?.let {
                setBitmapImage(true, it.first, it.second)
            }
            hideProgressBar()
        }
    }

    private fun hasDownloadedImages(): Boolean {

        var hasImages = true
        if (mNotificationData.mCarouselList != null) {
            for (item in mNotificationData.mCarouselList!!) {
                if (item.mMediaLocalPath.isNullOrEmpty()) {
                    hasImages = false
                    break
                }
            }
        }
        return hasImages
    }

    private fun getCarouselItems(data: SMTNotificationData): Pair<SMTCarouselItemData?, SMTCarouselItemData?>? {
        if (data.mTrid == mNotificationData.mTrid) {
            val leftitem = this.mNotificationData.mCarouselList?.get(mCurrentIndex)
            var rightItem: SMTCarouselItemData? = null

            if (isPortrait && ((mCurrentIndex + 1) < (mNotificationData.mCarouselList?.size?.minus(1)!!))) {
                rightItem = this.mNotificationData.mCarouselList?.get(mCurrentIndex + 1)
            }
            return Pair(leftitem, rightItem)
        }
        return null
    }

    /**
     * Download image
     */
    private fun downloadImage() {
        mNotificationData.mIsForInbox = true

        if (!mNotificationData.mIsDownloadInProgress) {
            mNotificationData.mIsDownloadInProgress = true

            // Download media
            SMTMediaDownloadManager().downloadMedia(context, mNotificationData, object : SMTMediaDownloadManager.MediaDownloadListener {

                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {

                        mNotificationData.mIsDownloadInProgress = false
                        mNotificationData = notification
                        val pair = getCarouselItems(notification)
                        pair?.let {
                            setBitmapImage(true, pair.first, pair.second)
                        }
                        hideProgressBar()
                    }
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        // disable progress bar
                        hideProgressBar()
                        mNotificationData.mIsDownloadInProgress = false
                    }
                }
            })
        }
    }

    private fun hideProgressBar() {
        if (isPortrait) {
            portraitView.findViewById<ProgressBar>(R.id.left_loading_progressBar).visibility = View.GONE
            portraitView.findViewById<ProgressBar>(R.id.right_loading_progressBar).visibility = View.GONE
        } else {
            landscapeView.findViewById<ProgressBar>(R.id.landscape_carouel_loading_progressBar).visibility = View.GONE
        }
    }

    private fun showProgressBar() {
        if (isPortrait) {
            portraitView.findViewById<ProgressBar>(R.id.left_loading_progressBar).visibility = View.VISIBLE
            portraitView.findViewById<ProgressBar>(R.id.right_loading_progressBar).visibility = View.VISIBLE
        } else {
            landscapeView.findViewById<ProgressBar>(R.id.landscape_carouel_loading_progressBar).visibility = View.VISIBLE
        }
    }

    /**
     * Set Bitmap Image
     * If download success then set the downloaded image
     * else show placeholder image
     */
    private fun setBitmapImage(isDownloadSuccess: Boolean, leftItem: SMTCarouselItemData?, rightItem: SMTCarouselItemData?) {
        if (isDownloadSuccess) {
            if (isPortrait) {
                leftItem?.let {
                    portraitView.findViewById<ImageView>(R.id.iv_portrait_left_carousel_image)
                            .setImageBitmap(SMTCommonUtility.loadImageFromLocalStorage(it.mMediaLocalPath!!))
                }
                rightItem?.let {
                    portraitView.findViewById<ImageView>(R.id.iv_portrait_right_carousel_image)
                            .setImageBitmap(SMTCommonUtility.loadImageFromLocalStorage(it.mMediaLocalPath!!))
                }
            } else {
                leftItem?.let {
                    landscapeView.findViewById<ImageView>(R.id.iv_landscape_carousel_image)
                            .setImageBitmap(SMTCommonUtility.loadImageFromLocalStorage(it.mMediaLocalPath!!))
                }
            }
        } else {
            val notifOptions = SMTCommonUtility.getNotificationOptions(context)

            if (isPortrait) {
                portraitView.findViewById<ImageView>(R.id.iv_portrait_left_carousel_image)
                        .setImageBitmap(SMTCommonUtility.getBitmapFromResId(context, notifOptions.placeHolderIcon))
                portraitView.findViewById<ImageView>(R.id.iv_portrait_right_carousel_image)
                        .setImageBitmap(SMTCommonUtility.getBitmapFromResId(context, notifOptions.placeHolderIcon))

            } else {
                landscapeView.findViewById<ImageView>(R.id.iv_landscape_carousel_image)
                        .setImageBitmap(SMTCommonUtility.getBitmapFromResId(context, notifOptions.placeHolderIcon))
            }
        }
    }

    private fun setPortraitViewDetails(leftItem: SMTCarouselItemData?, rightItem: SMTCarouselItemData?) {

        setPortraitLayoutAction(leftItem, rightItem)

        setBitmapImage(true, leftItem, rightItem)

        leftItem?.let {
            portraitView.findViewById<TextView>(R.id.tv_portrait_left_carousel_title).text = it.imgTitle
            portraitView.findViewById<TextView>(R.id.tv_portrait_left_carousel_content).text = it.imgMsg
        }

        rightItem?.let {
            portraitView.findViewById<TextView>(R.id.tv_portrait_right_carousel_title).text = it.imgTitle
            portraitView.findViewById<TextView>(R.id.tv_portrait_right_carousel_content).text = it.imgMsg
        }

        if (rightItem == null) {
            portraitView.findViewById<LinearLayout>(R.id.ll_portrait_right_carousel_layout).visibility = View.INVISIBLE
        } else {
            portraitView.findViewById<LinearLayout>(R.id.ll_portrait_right_carousel_layout).visibility = View.VISIBLE
        }
    }

    private fun portraitRightArrowClicked() {

        handleLeftRightClickEvent()

        val leftItemIndex: Int = (mCurrentIndex + 2) % mNotificationData.mCarouselList!!.size
        val rightItemIndex: Int

        rightItemIndex = if (leftItemIndex + 1 > mNotificationData.mCarouselList!!.size - 1) {
            0
        } else {
            leftItemIndex + 1
        }
        mCurrentIndex = leftItemIndex
        setPortraitViewDetails(mNotificationData.mCarouselList!![leftItemIndex], mNotificationData.mCarouselList!![rightItemIndex])


    }

    private fun portraitLeftArrowClicked() {
        var leftItemIndex = (mCurrentIndex - 2) % mNotificationData.mCarouselList!!.size
        if (leftItemIndex < 0) {
            leftItemIndex += mNotificationData.mCarouselList!!.size
        }
        var rightItemIndex: Int

        handleLeftRightClickEvent()

        if (leftItemIndex + 1 >= mNotificationData.mCarouselList!!.size) {
            if (leftItemIndex + 1 == mNotificationData.mCarouselList!!.size) {
                rightItemIndex = 0
            } else {
                rightItemIndex = leftItemIndex + 1
            }
        } else {
            rightItemIndex = leftItemIndex + 1
        }
        mCurrentIndex = leftItemIndex
        setPortraitViewDetails(mNotificationData.mCarouselList!![leftItemIndex], mNotificationData.mCarouselList!![rightItemIndex])
    }
}