package com.netcore.android.inbox.views

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.ProgressBar
import android.widget.SeekBar
import androidx.appcompat.widget.AppCompatTextView
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.netcore.android.R
import com.netcore.android.inbox.helpers.SMTInboxAudioPlayerService
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility


internal class SMTInboxAudioView(context: Context, attrs: AttributeSet?) : SMTBaseView(context, attrs) {

    lateinit var mView: View
    var mIsAudioPlaying = false
    var mAudioSeekBar: SeekBar? = null

    init {

        initLayout()
    }

    private fun initLayout() {
        mView = LayoutInflater.from(context).inflate(R.layout.inbox_audio_message_layout, this)
        mAudioSeekBar = mView.findViewById(R.id.seek_bar)
        val lp = LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)

        this.layoutParams = lp

        this.orientation = LinearLayout.VERTICAL
    }

    override fun setNotificationData(notification: SMTNotificationData) {
        setData(notification)
        setViewDetail()
        setLayoutAction()
        setBackgroundShape()
    }


    private fun downloadAudio() {
        if (!mNotificationData.mMediaUrl.isNullOrEmpty()) {
            showProgressBar()
            download()
        } else {
            hideProgressBar()
            showError(context.getString(R.string.error_in_media_url))
        }

    }

    private fun download() {
        mNotificationData.mIsForInbox = true
        if (!mNotificationData.mIsDownloadInProgress) {
            mNotificationData.mIsDownloadInProgress = true
            // Download media
            SMTMediaDownloadManager().downloadMedia(context, mNotificationData, object : SMTMediaDownloadManager.MediaDownloadListener {

                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        mNotificationData.mIsDownloadInProgress = false
                        hideProgressBar()
                        mNotificationData = notification
                        if (this@SMTInboxAudioView.isShown && this@SMTInboxAudioView.isAttachedToWindow) {
                            playAudio()
                        } else {
                            SMTLogger.e("InboxAudioView", "view isn't in focus so ingnoring playback isShown " + this@SMTInboxAudioView.isShown + " is attached " + this@SMTInboxAudioView.isAttachedToWindow)
                        }
                    }
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        // disable progress bar
                        hideProgressBar()
                        mNotificationData.mIsDownloadInProgress = false
                        hideProgressBar()
                    }
                }
            })
        } else {
            showProgressBar()
        }
    }

    private fun playAudio() {
        LocalBroadcastManager.getInstance(context).registerReceiver(playerServiceBroadCastReceiver, IntentFilter(mNotificationData.mTrid))
        val playerIntent = Intent(context, SMTInboxAudioPlayerService::class.java)
        playerIntent.putExtra("Notification", mNotificationData)
        playerIntent.putExtra("Action", SMTInboxAudioPlayerService.PLAY)
        context.startService(playerIntent)

    }

    private fun onPauseAudio() {
        LocalBroadcastManager.getInstance(context).unregisterReceiver(playerServiceBroadCastReceiver)
        setPlayButton()
    }

    private fun onStopAudio() {

        if (mIsAudioPlaying) {
            mIsAudioPlaying = false
            stopPlaying()
        }
        LocalBroadcastManager.getInstance(context).unregisterReceiver(playerServiceBroadCastReceiver)
        mIsAudioPlaying = false
        setCurrentProgress(0, 0)
        setPlayButton()

    }

    private fun showProgressBar() {
        mView.findViewById<ProgressBar>(R.id.loadingProgressBar).visibility = View.VISIBLE
        mView.findViewById<ImageView>(R.id.play_pause).visibility = View.GONE
    }

    private fun hideProgressBar() {
        mView.findViewById<ProgressBar>(R.id.loadingProgressBar).visibility = View.GONE
        mView.findViewById<ImageView>(R.id.play_pause).visibility = View.VISIBLE
    }

    private fun setViewDetail() {

        mView.findViewById<AppCompatTextView>(R.id.tv_audio_title).text = mNotificationData.mTitle
        mView.findViewById<AppCompatTextView>(R.id.tv_audio_timestamp).text = SMTCommonUtility.getFormattedTimeDifference(mNotificationData.mPublishedTimeStamp)
        mView.findViewById<AppCompatTextView>(R.id.tv_audio_content).text = mNotificationData.mMessage
        mView.findViewById<ImageView>(R.id.play_pause).setOnClickListener {
            handleAudioPlayClick()
        }
        setSeekBarListener()

        if (mNotificationData.mIsDownloadInProgress) {
            showProgressBar()
        } else {
            hideProgressBar()
        }
    }

    private fun setSeekBarListener() {

        mAudioSeekBar!!.setOnSeekBarChangeListener(object : SeekBar.OnSeekBarChangeListener {
            override fun onProgressChanged(p0: SeekBar?, p1: Int, p2: Boolean) {

            }

            override fun onStartTrackingTouch(p0: SeekBar?) {
            }

            override fun onStopTrackingTouch(seekBar: SeekBar?) {
                seekAudio(seekBar?.progress)
            }
        })
    }

    private fun seekAudio(progress: Int?) {
        if (mIsAudioPlaying) {
            val playerIntent = Intent(context, SMTInboxAudioPlayerService::class.java)
            playerIntent.putExtra("Action", SMTInboxAudioPlayerService.SEEK)
            playerIntent.putExtra("Notification", mNotificationData)
            playerIntent.putExtra("Progress", progress)
            context.startService(playerIntent)
        }
    }

    private fun handleAudioPlayClick() {

        handleClick(null)

        if (!mNotificationData.mMediaLocalPath.isNullOrEmpty()) {
            // we have the media file downloaded play or pause it
            if (mIsAudioPlaying) {
                // audio is playing pause it
                pauseAudio()
            } else {
                // play it
                playAudio()
            }
        } else {
            downloadAudio()
        }
    }

    private fun pauseAudio() {
        if (mIsAudioPlaying) {
            val playerIntent = Intent(context, SMTInboxAudioPlayerService::class.java)
            playerIntent.putExtra("Action", SMTInboxAudioPlayerService.PAUSE)
            playerIntent.putExtra("Notification", mNotificationData)
            context.startService(playerIntent)
            onPauseAudio()
        }
    }

    /**
     * Set action listener to the layout
     */
    private fun setLayoutAction() {
        mView.setOnClickListener {
            handleClick(mNotificationData.mDeepLinkPath)
        }
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        LocalBroadcastManager.getInstance(context).unregisterReceiver(playerServiceBroadCastReceiver)
        stopPlaying()

    }

    private fun stopPlaying() {
        if (mIsAudioPlaying) {
            val playerIntent = Intent(context, SMTInboxAudioPlayerService::class.java)
            playerIntent.putExtra("Action", SMTInboxAudioPlayerService.STOP)
            playerIntent.putExtra("Notification", mNotificationData)
            context.startService(playerIntent)
            onStopAudio()
        }
    }

    override fun onWindowVisibilityChanged(visibility: Int) {
        super.onWindowVisibilityChanged(visibility)

        if (visibility != View.VISIBLE) {
            stopPlaying()
        }
    }

    private val playerServiceBroadCastReceiver = object : BroadcastReceiver() {
        override fun onReceive(p0: Context?, intent: Intent?) {

            if (intent?.action == mNotificationData.mTrid) {
                when (intent.getIntExtra(SMTInboxAudioPlayerService.AUDIO_ACTION, 0)) {
                    SMTInboxAudioPlayerService.AUDIO_STOPPED -> {
                        onStopAudio()
                    }
                    SMTInboxAudioPlayerService.AUDIO_PLAYING -> {
                        setPauseButton()
                        setCurrentProgress(intent.getIntExtra(SMTInboxAudioPlayerService.AUDIO_MAX_DURATION, 0), intent.getIntExtra(SMTInboxAudioPlayerService.AUDIO_PROGRESS_RATE, 0))
                    }
                    SMTInboxAudioPlayerService.AUDIO_ERROR -> {
                        showError(context.getString(R.string.audio_play_error))
                        setPlayButton()
                    }
                    SMTInboxAudioPlayerService.AUDIO_PROGRESS -> {
                        setCurrentProgress(intent.getIntExtra(SMTInboxAudioPlayerService.AUDIO_MAX_DURATION, 0), intent.getIntExtra(SMTInboxAudioPlayerService.AUDIO_PROGRESS_RATE, 0))
                    }
                }
            }
        }
    }

    private fun setCurrentProgress(maxDuration: Int, currentProgress: Int) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mAudioSeekBar?.setProgress(currentProgress, true)
        } else {
            mAudioSeekBar?.progress = currentProgress
        }

        mAudioSeekBar?.max = maxDuration
    }

    private fun setPauseButton() {
        mIsAudioPlaying = true
        mView.findViewById<ImageView>(R.id.play_pause).setImageResource(R.drawable.ic_pause_black_24dp)
    }

    private fun setPlayButton() {
        mIsAudioPlaying = false
        mView.findViewById<ImageView>(R.id.play_pause).setImageResource(R.drawable.ic_play_arrow_black_24dp)
    }
}