package com.netcore.android.workmgr

import android.content.Context
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.db.SMTEventTable
import com.netcore.android.event.SMTEventProcessor
import com.netcore.android.event.SMTEventSyncStatus
import com.netcore.android.event.SMTEventsBatchProcessor
import com.netcore.android.event.SMTWorkerType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTNetowrkUtil
import com.netcore.android.notification.models.SMTEventPayload
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference

/**
 *  This worker is created to sync events in Foreground.
 *
 *  @author Netcore
 *
 *  @Description This worker will run in foreground and It will sync only Pending events from the Database. Once its started, it will
 *  sync all pending events present inside the Database. There ia an handler is running to check counts of events presents in Database so
 *  if any new events are inserted in Database then this worker will be automatically started to sync events.
 *
 *  This worker is also used to sync Notification events from the database beacuse we need to sync those events realtime. Never use this
 *  worker to sync failed events.
 *
 *  By default, It's a OneTimeWork worker which will after invoked by handler.
 *
 */

internal class EventSyncWorker(context: Context, param: WorkerParameters) : Worker(context, param) {

    val TAG: String = EventSyncWorker::class.java.simpleName
    lateinit var context: Context
    private lateinit var eventPayload: SMTEventPayload
    private lateinit var smtEventsBatchProcessor: SMTEventsBatchProcessor

    override fun doWork(): Result {
        try {
            context = applicationContext
            smtEventsBatchProcessor = SMTEventsBatchProcessor.getInstance(context)

            initiateBatchExecution()

        } catch (e: Exception) {
            SMTLogger.e(TAG, "${e.printStackTrace()}")
            handleError()
            return Result.failure()
        }

        return Result.success()
    }

    override fun onStopped() {
        super.onStopped()
        SMTLogger.v(TAG, "On stopped called ")
        handleError()
    }

    private fun handleError() {
        SMTLogger.i(TAG, "Event sync worker stopped")
        if (::eventPayload.isInitialized) {
            SMTDataBaseService.getInstance(WeakReference(context)).updateEventProcessingStatus(eventPayload.idArray, SMTEventTable.KEY_EVENT_SYNC_STATUS, SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED)
        } else {
            SMTLogger.i(TAG, "EventPayload is not initialised.")
        }
    }

    private fun initiateBatchExecution() {

        // Check SDK status before processing event
        processEvents()

    }

    private tailrec fun processEvents() {

        eventPayload = smtEventsBatchProcessor.prepareBatchRequest(WeakReference(applicationContext), SMTWorkerType.EventWorker)
        if (eventPayload.eventArray.length() <= 0) {
            SMTLogger.v(TAG, "EventsArray size is 0")
            return
        }
        val smtRequest = smtEventsBatchProcessor.getBatchProcessRequest(eventPayload.eventArray)
        val smtResponse = SMTEventProcessor.getInstance().processEvents(smtRequest)
        SMTLogger.v(TAG, "Request code is :${smtResponse?.httpCode}")

        smtResponse?.let {
            if (it.isSuccess) {
                onSuccess(eventPayload.idArray)
            } else onFailure(eventPayload.idArray)
            if (isEventsPresent()) {
                SMTLogger.internal(TAG, "Still events are present in DB to be processed.")
                processEvents()
            } else {
                SMTLogger.internal(TAG, "No events are present in DB to be processed.")
            }
        } ?: onFailure(eventPayload.idArray)
    }


    private fun isEventsPresent(): Boolean {
        return smtEventsBatchProcessor.checkIfMoreEventsPresentForBatchRequest(WeakReference(context), SMTWorkerType.EventWorker)
    }

    private fun onSuccess(idArray: Array<Int>) {
        SMTDataBaseService.getInstance(WeakReference(context)).deleteEventsMultipleRows(idArray)
    }

    private fun onFailure(idArray: Array<Int>) {
        SMTDataBaseService.getInstance(WeakReference(context)).updateEventProcessingStatus(idArray, SMTEventTable.KEY_EVENT_SYNC_STATUS, SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED)
        SMTDataBaseService.getInstance(WeakReference(context)).updateFailedBatchPayload()
        SMTLogger.v(TAG, "Events failed.")
    }
}