package com.netcore.android.notification

import android.util.Log
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.models.SMTActionButtonData
import com.netcore.android.notification.models.SMTCarouselItemData
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONArray
import org.json.JSONObject
import java.net.URLDecoder
import java.util.*
import kotlin.collections.ArrayList
import kotlin.collections.HashMap

/**
 * Class useful for parsing notification payload
 * It parses the notification payload and populates
 * the data into SMTNotificationData class
 *
 *  Copyright © 2019 Netcore. All rights reserved.
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */

/*
here is sample notification payload
{
    "data": {
    "source": "Smartech",
    "smtPayload": {
        "type": "Carousel",
        "trid": "99999-111-0-0-19021511111",
        "title": "Notification Title",
        "body": "Notification Body",
        "mediaurl": "",
        "deeplink": "",
        "actionButton": [
            {
            "actionName": "Yes",
            "actionDeeplink": "https://www.google.com"
            }
        ],
        "carousel": [
        {
            "imgUrl": "https://pnstage.netcore.co.in/netcoresdk-demo-files/image/demo-image-1.jpg",
            "imgTitle": "Carousel Image 1 Title",
            "imgMsg": "Carousel Image 1 Message",
            "imgDeeplink": "https://www.google.com"
        }
        ]
    },
    "smtCustomPayload": {
        "city": "Mumbai",
        "state": "Maharashtra",
        "country": "India"
    }
}
}*/
internal class SMTNotificationParser {

    private val TAG = SMTNotificationParser::class.java.simpleName

    /**
     * Parses the notificaiton payload
     * @param notifData - Notification Payload
     * @return SMTNotificationData payload object
     */


    fun parse(notifData: String, @SMTNotificationSourceType.Source notifSource: Int): SMTNotificationData? {

        try {
            val jsonObject = JSONObject(notifData)

            val isNewPayload = when (notifSource == SMTNotificationSourceType.NOTIFICATION_SOURCE_PN) {
                false -> (jsonObject.has(SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY))

                true -> {
                    //val objData = jsonObject.getJSONObject(SMTNotificationConstants.NOTIF_DATA_KEY)
                    jsonObject.has(SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY)
                }
            }

            return if (isNewPayload) parseNewJson(notifData, notifSource) else parseMixedJson(notifData, notifSource)

        } catch (e: Exception) {
            e.printStackTrace()
            return null
        }
    }

    private fun parseNewJson(notifData: String, @SMTNotificationSourceType.Source notifSource: Int): SMTNotificationData? {

        val notificationBuilder = SMTNotificationData.NotificationBuilder()
        try {
            val jsonObject = JSONObject(notifData)
            try {
                val data: JSONObject = if (notifSource == SMTNotificationSourceType.NOTIFICATION_SOURCE_PN) {
                    jsonObject
                } else {
                    JSONObject(notifData)
                }

                val source = jsonObject.optString(SMTNotificationConstants.NOTIF_SOURCE_KEY)
                val isSourceMatched = (SMTNotificationConstants.NOTIF_SOURCE_VALUE).equals(source, true)
                if (!isSourceMatched) {
                    return null
                }
                notificationBuilder.setIsFromSmartech(true)
                // Parse Smartech Payload
                try {
                    val smtPayload: JSONObject = jsonObject.getJSONObject(SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY)
                    notificationBuilder.setSource(notifSource)
                    try {
                        notificationBuilder.setTrId(smtPayload.optString(SMTNotificationConstants.NOTIF_TR_ID_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setTitle(smtPayload.optString(SMTNotificationConstants.NOTIF_TITLE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setSubtitle(smtPayload.optString(SMTNotificationConstants.NOTIF_SUBTITLE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setMessage(smtPayload.optString(SMTNotificationConstants.NOTIF_BODY_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setMediaUrl(smtPayload.optString(SMTNotificationConstants.NOTIF_MEDIA_URL_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setMediaLocalPath(smtPayload.optString(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setNotificationType(getNotificationType(smtPayload.optString(SMTNotificationConstants.NOTIF_TYPE_KEY)))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setDeepLinkPath(smtPayload.optString(SMTNotificationConstants.NOTIF_DEEPLINK_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setStatus(smtPayload.optString(SMTNotificationConstants.NOTIF_STATUS_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setPublishedDate(smtPayload.optString(SMTNotificationConstants.NOTIF_PUBLISHED_DATE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setSound(smtPayload.optBoolean(SMTNotificationConstants.NOTIF_SOUND_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setSoundFile(smtPayload.optString(SMTNotificationConstants.NOTIF_SOUND_FILE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setChannelId(smtPayload.optString(SMTNotificationConstants.NOTIF_CHANNEL_ID_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    // Parsing of Action Buttons
                    try {
                        val jsonArrayAction: JSONArray = smtPayload.getJSONArray(SMTNotificationConstants.NOTIF_ACTION_BUTTON_KEY)
                        if (jsonArrayAction.length() > 0) {
                            val actionButtonList = ArrayList<SMTActionButtonData>()
                            for (i in 0 until jsonArrayAction.length()) {
                                val item = jsonArrayAction.get(i)
                                try {
                                    if (item is JSONObject) {
                                        val actionName = (item.optString(SMTNotificationConstants.NOTIF_ACTION_NAME_KEY))
                                        val actionDeepLink = (item.optString(SMTNotificationConstants.NOTIF_ACTION_DEEPLINK_KEY))
                                        actionButtonList.add(SMTActionButtonData(actionName, actionDeepLink))
                                    }
                                } catch (e: Exception) {
                                    e.printStackTrace()
                                }
                            }
                            if (actionButtonList.size > 0) {
                                notificationBuilder.setActionButtonList(actionButtonList)
                            }
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    // Parsing Attribute
                    try {
                        val attributionMap = HashMap<String, String>()
                        val objSmtAttribute = smtPayload.optJSONObject(SMTNotificationConstants.NOTIF_SMT_ATTRIBUTE_PAYLOAD_KEY)
                        objSmtAttribute?.let {
                            checkEmptyKeyAttributes(objSmtAttribute, SMTNotificationConstants.NOTIF_ATTRIBUTION_MEDIUM, attributionMap)
                            checkEmptyKeyAttributes(objSmtAttribute, SMTNotificationConstants.NOTIF_ATTRIBUTION_SOURCE, attributionMap)
                            checkEmptyKeyAttributes(objSmtAttribute, SMTNotificationConstants.NOTIF_ATTRIBUTION_STA, attributionMap)
                            val sourceValue = objSmtAttribute.optString(SMTNotificationConstants.NOTIF_ATTRIBUTION_STA)
                            val decodedDataArray = getEncodedValue(sourceValue)
                            attributionMap[SMTNotificationConstants.NOTIF_ATTRIBUTION_ID] = if (decodedDataArray.isNotEmpty()) decodedDataArray[0] else objSmtAttribute.optString(SMTNotificationConstants.NOTIF_ATTRIBUTION_ID)
                            attributionMap[SMTNotificationConstants.NOTIF_ATTRIBUTION_IDENTITY] = if (decodedDataArray.isNotEmpty() && decodedDataArray.size > 1) decodedDataArray[1] else ""
                            notificationBuilder.setSmtAttributePayload(attributionMap)
                        } ?: SMTLogger.i(TAG, "No value for attrParams in payload.")
                    } catch (e: Exception) {
                        SMTLogger.i(TAG, "AttrParams missing")
                    }

                    // Parsing of Carousel Info
                    try {
                        val jsonArrayCarousel: JSONArray? = smtPayload.optJSONArray(SMTNotificationConstants.NOTIF_CAROUSEL_KEY)
                        jsonArrayCarousel?.let {
                            if (it.length() > 0) {
                                val carouselList = ArrayList<SMTCarouselItemData>()
                                for (i in 0 until it.length()) {
                                    val item = it.get(i)
                                    try {
                                        if (item is JSONObject) {
                                            val imgTitle = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_TITLE_KEY))
                                            val imgMsg = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_MSG_KEY))
                                            val imgUrl = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_URL_KEY))
                                            val imgDeeplink = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_DEEPLINK_KEY))
                                            val imgLocalPath = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_LOCAL_PATH_KEY))

                                            carouselList.add(SMTCarouselItemData(i + 1, imgTitle, imgMsg, imgUrl, imgDeeplink, imgLocalPath))
                                        }
                                    } catch (e: Exception) {
                                        e.printStackTrace()
                                    }
                                }
                                if (carouselList.size > 0) {
                                    notificationBuilder.setCarouselList(carouselList)
                                }
                            }
                        } ?: SMTLogger.v(TAG, "No carousel parameter in payload.")

                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                } catch (e: Exception) {
                    e.printStackTrace()
                }


                // Parse Custom Payload
                try {
                    val smtCustomPayload = data.getJSONObject(SMTNotificationConstants.NOTIF_CUSTOM_PAYLOAD_KEY)
                    smtCustomPayload?.let {
                        notificationBuilder.setCustomPayload(SMTCommonUtility.jsonToMap(smtCustomPayload))
                    } ?: SMTLogger.i(TAG, "No value for smtCustomPayload")

                } catch (e: Exception) {
                    e.printStackTrace()
                }

                // Parsing mPNMeta
                try {
                    val objPNMeta = data.optJSONObject(SMTNotificationConstants.NOTIF_PN_META_KEY)
                    notificationBuilder.setPNMeta(objPNMeta?.toString() ?: JSONObject().toString())

                } catch (e: Exception) {
                    e.printStackTrace()
                }

            } catch (e: Exception) {
                e.printStackTrace()
            }

        } catch (e: Exception) {
            e.printStackTrace()
            SMTLogger.e(TAG, "error while parsing notification paylaod : $e")
            return null
        }

        return notificationBuilder.build()

    }


    private fun parseMixedJson(notifData: String, @SMTNotificationSourceType.Source notifSource: Int): SMTNotificationData? {

        val notificationBuilder = SMTNotificationData.NotificationBuilder()
        try {
            val parentObject = JSONObject(notifData)

            try {
                val responseObject: JSONObject = if (notifSource == SMTNotificationSourceType.NOTIFICATION_SOURCE_PN) {
                    parentObject.getJSONObject(SMTNotificationConstants.NOTIF_DATA_KEY)
                } else {
                    parentObject
                }

                val source = parentObject.optString(SMTNotificationConstants.NOTIF_SOURCE_KEY)
                val isSourceMatched = (SMTNotificationConstants.NOTIF_SOURCE_VALUE).equals(source, true)
                if (!isSourceMatched) {
                    return null
                }
                notificationBuilder.setIsFromSmartech(true)

                // Parse Smartech Payload
                try {
                    notificationBuilder.setSource(notifSource)
                    try {
                        notificationBuilder.setTrId(responseObject.optString(SMTNotificationConstants.NOTIF_TR_ID_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setTitle(responseObject.optString(SMTNotificationConstants.NOTIF_TITLE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setSubtitle(responseObject.optString(SMTNotificationConstants.NOTIF_SUBTITLE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setMessage(responseObject.optString(SMTNotificationConstants.NOTIF_MESSAGE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setMediaUrl(responseObject.optString(SMTNotificationConstants.NOTIF_IMAGE_URL_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setMediaLocalPath(responseObject.optString(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setNotificationType(getNotificationType(responseObject.optString(SMTNotificationConstants.NOTIF_TYPE_KEY)))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setDeepLinkPath(responseObject.optString(SMTNotificationConstants.NOTIF_DEEPLINK_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setStatus(responseObject.optString(SMTNotificationConstants.NOTIF_STATUS_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setPublishedDate(responseObject.optString(SMTNotificationConstants.NOTIF_PUBLISHED_DATE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setSound(responseObject.optBoolean(SMTNotificationConstants.NOTIF_SOUND_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setSoundFile(responseObject.optString(SMTNotificationConstants.NOTIF_SOUND_FILE_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    try {
                        notificationBuilder.setChannelId(responseObject.optString(SMTNotificationConstants.NOTIF_CHANNEL_ID_KEY))
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    // Parsing of Action Buttons
                    try {
                        val jsonArrayAction: JSONArray = responseObject.getJSONArray(SMTNotificationConstants.NOTIF_ACTION_BUTTON_KEY)
                        if (jsonArrayAction.length() > 0) {
                            val actionButtonList = ArrayList<SMTActionButtonData>()
                            for (i in 0 until jsonArrayAction.length()) {
                                val item = jsonArrayAction.get(i)
                                try {
                                    if (item is JSONObject) {
                                        val actionName = (item.optString(SMTNotificationConstants.NOTIF_ACTION_NAME_KEY))
                                        val actionDeepLink = (item.optString(SMTNotificationConstants.NOTIF_ACTION_DEEPLINK_KEY))
                                        actionButtonList.add(SMTActionButtonData(actionName, actionDeepLink))
                                    }
                                } catch (e: Exception) {
                                    e.printStackTrace()
                                }
                            }
                            if (actionButtonList.size > 0) {
                                notificationBuilder.setActionButtonList(actionButtonList)
                            }
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    // Parsing Attribute

                    try {
                        val attributionMap = HashMap<String, String>()
                        val objSmtAttribute = responseObject.optJSONObject(SMTNotificationConstants.NOTIF_SMT_ATTRIBUTE_PAYLOAD_KEY)
                        objSmtAttribute?.let {
                            checkEmptyKeyAttributes(objSmtAttribute, SMTNotificationConstants.NOTIF_ATTRIBUTION_MEDIUM, attributionMap)
                            checkEmptyKeyAttributes(objSmtAttribute, SMTNotificationConstants.NOTIF_ATTRIBUTION_SOURCE, attributionMap)
                            checkEmptyKeyAttributes(objSmtAttribute, SMTNotificationConstants.NOTIF_ATTRIBUTION_STA, attributionMap)
                            val sourceValue = objSmtAttribute.optString(SMTNotificationConstants.NOTIF_ATTRIBUTION_STA)
                            val decodedDataArray = getEncodedValue(sourceValue)
                            attributionMap[SMTNotificationConstants.NOTIF_ATTRIBUTION_ID] = if (decodedDataArray.isNotEmpty()) decodedDataArray[0] else objSmtAttribute.optString(SMTNotificationConstants.NOTIF_ATTRIBUTION_ID)
                            attributionMap[SMTNotificationConstants.NOTIF_ATTRIBUTION_IDENTITY] = if (decodedDataArray.isNotEmpty() && decodedDataArray.size > 1) decodedDataArray[1] else ""
                            notificationBuilder.setSmtAttributePayload(attributionMap)
                        } ?: SMTLogger.i(TAG, "No value for attrParams in payload.")
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                    // Parsing of Carousel Info
                    try {
                        val jsonArrayCarousel: JSONArray? = responseObject.optJSONArray(SMTNotificationConstants.NOTIF_CAROUSEL_KEY)
                        jsonArrayCarousel?.let {
                            if (it.length() > 0) {
                                val carouselList = ArrayList<SMTCarouselItemData>()
                                for (i in 0 until it.length()) {
                                    val item = it.get(i)
                                    try {
                                        if (item is JSONObject) {
                                            val imgTitle = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_TITLE_KEY))
                                            val imgMsg = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_MSG_KEY))
                                            val imgUrl = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_URL_KEY))
                                            val imgDeeplink = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_DEEPLINK_KEY))
                                            val imgLocalPath = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_LOCAL_PATH_KEY))

                                            carouselList.add(SMTCarouselItemData(i + 1, imgTitle, imgMsg, imgUrl, imgDeeplink, imgLocalPath))
                                        }
                                    } catch (e: Exception) {
                                        e.printStackTrace()
                                    }
                                }
                                if (carouselList.size > 0) {
                                    notificationBuilder.setCarouselList(carouselList)
                                }
                            }
                        } ?: SMTLogger.i(TAG, "No carousel parameter in payload.")
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }

                } catch (e: Exception) {
                    e.printStackTrace()
                }

                // Parse Custom Payload
                try {
                    val smtCustomPayload = parentObject.optJSONObject(SMTNotificationConstants.NOTIF_CUSTOM_PAYLOAD_KEY)
                    smtCustomPayload?.let {
                        notificationBuilder.setCustomPayload(SMTCommonUtility.jsonToMap(smtCustomPayload))
                    } ?: SMTLogger.i(TAG, "No value for smtCustomPayload")
                } catch (e: Exception) {
                    e.printStackTrace()
                }

                // Parsing mPNMeta
                try {
                    val objPNMeta = responseObject.optJSONObject(SMTNotificationConstants.NOTIF_PN_META_KEY)
                    notificationBuilder.setPNMeta(objPNMeta?.toString() ?: JSONObject().toString())
                } catch (e: Exception) {
                    e.printStackTrace()
                }

            } catch (e: Exception) {
                e.printStackTrace()
            }

        } catch (e: Exception) {
            e.printStackTrace()
            SMTLogger.e(TAG, "error while parsing notification paylaod : $e")
            return null
        }

        return notificationBuilder.build()

    }

    /**
     * To find out the notification type
     * @param type - notification type
     */
    private fun getNotificationType(type: String?): String {
        var notificationType = SMTNotificationType.SIMPLE.type
        type?.let {
            notificationType = when (it.toLowerCase(Locale.getDefault())) {
                SMTNotificationType.AUDIO.type -> SMTNotificationType.AUDIO.type
                SMTNotificationType.GIF.type -> SMTNotificationType.GIF.type
                SMTNotificationType.BIG_IMAGE.type -> SMTNotificationType.BIG_IMAGE.type
                SMTNotificationType.CAROUSEL_PORTRAIT.type -> SMTNotificationType.CAROUSEL_PORTRAIT.type
                SMTNotificationType.CAROUSEL_LANDSCAPE.type -> SMTNotificationType.CAROUSEL_LANDSCAPE.type
                SMTNotificationType.INTERACTIVE.type -> SMTNotificationType.INTERACTIVE.type
                else -> SMTNotificationType.SIMPLE.type
            }
        }
        return notificationType
    }

    fun updatePayload(notification: SMTNotificationData): SMTNotificationData {
        var updatedPayload: String? = notification.mPayload
        try {
            val jsonObject = JSONObject(notification.mPayload)
            val smtPayload = jsonObject.getJSONObject(SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY)
            if (notification.mNotificationType == SMTNotificationType.CAROUSEL_PORTRAIT.type || notification.mNotificationType == SMTNotificationType.CAROUSEL_LANDSCAPE.type) {

                // Parsing of Carousel Info
                try {
                    var jsonArrayCarousel: JSONArray? = smtPayload.getJSONArray(SMTNotificationConstants.NOTIF_CAROUSEL_KEY)
                    jsonArrayCarousel?.let {
                        if (it.length() > 0) {
                            val carouselList = JSONArray()
                            //Iterating through the carousel list
                            for (i in 0 until it.length()) {
                                val item = it.get(i)
                                try {
                                    if (item is JSONObject) {
                                        val imgUrl = (item.optString(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_URL_KEY))

                                        if (notification.mCarouselList != null && !imgUrl.isNullOrEmpty()) {
                                            //Looping through the carousel list in the notification object to find the exact match
                                            for (carousel in notification.mCarouselList!!) {
                                                if (carousel.imgUrl == imgUrl) {
                                                    item.put(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY, carousel.mMediaLocalPath)
                                                    break
                                                }
                                            }
                                        }
                                    }
                                } catch (e: Exception) {
                                    e.printStackTrace()
                                }
                                carouselList.put(item)
                            }
                            jsonArrayCarousel = carouselList
                        }
                    } ?: SMTLogger.v(TAG, "No carousel parameter in payload.")

                    smtPayload.put(SMTNotificationConstants.NOTIF_CAROUSEL_KEY, jsonArrayCarousel)
                } catch (e: Exception) {
                    e.printStackTrace()
                }

            } else {

                try {

                    smtPayload.put(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY, notification.mMediaLocalPath)
                } catch (e: Exception) {
                    e.printStackTrace()
                }

            }

            jsonObject.put(SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY, smtPayload)
            updatedPayload = jsonObject.toString()
        } catch (e: Exception) {
            e.printStackTrace()
        }
        notification.mPayload = updatedPayload ?: ""
        return notification
    }

    private fun checkEmptyKeyAttributes(obj: JSONObject, key: String, hMap: HashMap<String, String>) {

        val value = obj.optString(key)
        if (!value.isEmpty()) {
            hMap[key] = value
        }
    }

    private fun getEncodedValue(staParams: String?): Array<String> {
        var staParam = staParams
        if (staParam == null || staParam.trim { it <= ' ' }.isEmpty()) {
            return arrayOf()
        }

        try {
            staParam = URLDecoder.decode(staParam, SMTNotificationConstants.NOTIF_UTF_ENCODING)
            val encodedSta = staParam!!.toCharArray()
            val encryptionKey = "SXGWLZPDOKFIVUHJYTQBNMACERxswgzldpkoifuvjhtybqmncare0159378264"
            var paramString = ""

            for (i in encodedSta.indices.reversed()) {
                val encodedChar = encodedSta[i]
                paramString += if (encodedChar in ('a'..'z') || encodedChar in ('A'..'Z') || encodedChar == '&')

                    encryptionKey.let {
                        ((38 + it.indexOf(encodedChar)).toChar()).toString()
                    }
                else
                    encodedChar
            }

            return paramString.split("\\|".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()

        } catch (e: Exception) {
            e.printStackTrace()
        }

        return arrayOf()
    }

}