package com.netcore.android.inbox.network

import android.content.Context
import android.content.Intent
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTAppInboxTable
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventCommonDataDump
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.inbox.utility.SMTInboxMessageStatus
import com.netcore.android.inbox.utility.SMTInboxMessageType
import com.netcore.android.listeners.SMTInboxCallback
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTDownloaderUtility
import com.netcore.android.network.SMTEnumHttpMethodType
import com.netcore.android.network.SMTNetworkManager
import com.netcore.android.network.SMTRequestQueue
import com.netcore.android.network.SMTResponseListener
import com.netcore.android.network.models.SMTInboxResponse
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTResponse
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import org.json.JSONArray
import org.json.JSONObject
import java.lang.ref.WeakReference

internal class SMTInboxApiService private constructor(val context: WeakReference<Context>) : SMTResponseListener {

    private val TAG = SMTInboxApiService::class.java.simpleName
    private var mInboxCallback: SMTInboxCallback? = null

    @SMTInboxMessageType.Type
    private var mMsgType = SMTInboxMessageType.Type.INBOX_MESSAGE

    companion object {

        @Volatile
        private var INSTANCE: SMTInboxApiService? = null

        /**
         * Getting instance of the class
         */
        fun getInstance(context: WeakReference<Context>): SMTInboxApiService =
                INSTANCE
                        ?: synchronized(SMTInboxApiService::class.java) {
                            INSTANCE
                                    ?: buildInstance(context).also { INSTANCE = it }
                        }

        private fun buildInstance(context: WeakReference<Context>): SMTInboxApiService {
            return SMTInboxApiService(context)
        }

    }

    /**
     * Initialise API call on app launch
     */
    internal fun makeAppInboxApiCall(inboxCallback: SMTInboxCallback?, @SMTInboxMessageType.Type messageType: Int) {
        mInboxCallback = inboxCallback
        mMsgType = messageType
        val requestBuilder = SMTRequest.Builder()
                .setHttpMethod(SMTEnumHttpMethodType.POST)
                .setEndPoint(getAppInboxApiEndPoint())
                .setApiId(SMTRequest.SMTApiTypeID.INBOX_API)
                .setResponseListener(this)
                .setParams(getInboxApiParam())
                .setBaseUrl(SMTPreferenceHelper.getAppPreferenceInstance(context.get()!!, null).getString(SMTPreferenceConstants.SMT_BASE_URL))
        //.setBaseUrl(SMTPreferenceHelper.getAppPreferenceInstance(context.get()!!, null).getString(SMTPreferenceConstants.SMT_BASE_URL_INBOX))

        mInboxCallback?.onInboxProgress()
        SMTNetworkManager.getInstance(SMTRequestQueue.getInstance()).processRequest(requestBuilder.build())

        SMTLogger.i(TAG, "INBOX API called")
    }

    /**
     * provides pushamp request post params
     * @return JSONArray - post request params
     */
    private fun getInboxApiParam(): JSONArray {
        val hashMap = HashMap<String, Any?>()
        val jsonArray = JSONArray()
        hashMap[SMTConfigConstants.REQUEST_PARAM_KEY_APP_ID] = SMTPreferenceHelper.getAppPreferenceInstance(context.get()!!, null)
                .getString(SMTPreferenceConstants.SMT_MF_APP_ID)
        hashMap[SMTConfigConstants.REQUEST_PARAM_KEY_OS] = SMTConfigConstants.OS_NAME
        hashMap[SMTConfigConstants.REQUEST_PARAM_KEY_IDENTITY] = SMTPreferenceHelper.getAppPreferenceInstance(context.get()!!, null)
                .getString(SMTPreferenceConstants.SMT_USER_IDENTITY)

        return jsonArray.put(JSONObject(hashMap))
    }

    /**
     * Provides Push Amp end point
     * @return String - API end point
     */
    private fun getAppInboxApiEndPoint(): String {
        var urlParams = ""
        context.get()?.let {
            urlParams = SMTEventCommonDataDump(it).getURLParameters(false)
        }
        return "${SMTConfigConstants.APP_INBOX_API_PATH}?$urlParams}"
    }

    override fun onResposneSuccess(response: SMTResponse) {
        storeInboxMessages(response)
        mInboxCallback?.onInboxSuccess(SMTDataBaseService.getInstance(context).getInboxMessages(mMsgType))
        mInboxCallback = null
        mMsgType = SMTInboxMessageType.Type.INBOX_MESSAGE
    }

    override fun onResponseFailure(response: SMTResponse) {
        mInboxCallback?.onInboxFail()
        mInboxCallback = null
    }

    /**
     * Stores inbox response to DB and
     * alslo records inbox message deliver event
     * @param response received from server
     */
    private fun storeInboxMessages(response: SMTResponse) {
        val inboxResponse = response as SMTInboxResponse
        if (inboxResponse.inboxResponse?.isNotEmpty() == true) {
            SMTDataBaseService.getInstance(context).clearInboxTable()
            SMTDownloaderUtility.deleteInboxDirectory(context)
            inboxResponse.inboxResponse?.let {
                it.forEach { item ->
                    SMTDataBaseService.getInstance(context).storeInboxNotificaiton(item)
                    // For sent status only record deliver event
                    if (item.status == SMTAppInboxTable.STATUS_SENT) {
                        SMTEventRecorder.getInstance(context).recordInboxEvent(item.trid, null, SMTInboxMessageStatus.Status.DELIVERED)
                    }
                }
                broacastInboxResfresh()
            }
        }
    }

    private fun broacastInboxResfresh() {
        context.get()?.let {
            if (SMTPreferenceHelper.getAppPreferenceInstance(it, null).getInt(SMTPreferenceConstants.SMT_MF_IS_AUTO_FETCH_INBOX_MESSAGE) == 1) {
                val intent = Intent(SMTConfigConstants.BROADCAST_EVENT_INBOX_REFRESH)
                // You can also include some extra data.
                LocalBroadcastManager.getInstance(it).sendBroadcast(intent)
            }
        }
    }
}