package com.nearform.patrun;


import org.apache.commons.lang3.StringUtils;

import java.lang.Object;
import java.lang.String;
import java.util.*;
import java.util.regex.Matcher;


/**
 *  Package patrun is a fast pattern matcher on Go map properties
 *
 * For a full guide visit https://github.com/colmharte/patrun-java
 *
 * Need to pick out an object based on a subset of its properties? Say you've got:
 *
 * { x: 1       } -> A
 * { x: 1, y: 1 } -> B
 * { x: 1, y: 2 } -> C
 *
 * Then patrun can give you the following results:
 *
 * { x: 1 }      -> A
 * { x: 2 }      -> no match
 * { x: 1, y: 1 } -> B
 * { x: 1, y: 2 } -> C
 * { x: 2, y: 2 } -> no match
 * { y: 1 }      -> no match
 *
 * It's basically _query-by-example_ for property sets.
 */

public class Patrun {

    private Node tree;
    private Customiser custom;

    public Patrun() {
        this(null);
    }

    /**
     * Constructor taking a customiser object to allow the add / find / remove method results to be customised
     * <p>
     *
     * @param  custom  an instance of the Customiser interface
     */
    public Patrun(Customiser custom) {
        this.tree = new Node("root");
        this.custom = custom;
    }

    /**
     * Register a pattern, and the object that will be returned if an input
     * matches.
     * <p>
     *
     * @param  pat  a map of the pattern
     * @param  data the data to store for this pattern
     * @return      a reference to this to allow chaining of add calls
     */
    public Patrun add(Map<String, String> pat, Object data) {
        Modifier custom = null;

        if (this.custom != null) {
            custom = this.custom.add(this, pat, data);
        }

        Vector<String> keys = sortKeys(pat);

        Node currentNode = this.tree;
        Node lastNode;
        String val;
        Boolean justCreated;
        int position = 0;

        for (String key : keys) {

            val = pat.get(key);

            lastNode = currentNode;
            currentNode = currentNode.getValues().get(key);

            if (currentNode == null) {
                lastNode.getValues().put(key, new Node(key));
                currentNode = lastNode.getValues().get(key);
            }

            lastNode = currentNode;
            currentNode = currentNode.getValues().get(val);
            if (currentNode == null) {
                justCreated = true;
                if (position == keys.size() - 1) {
                    lastNode.getValues().put(val, new Node(val, data, custom));
                } else {
                    lastNode.getValues().put(val, new Node(val, custom));
                }
                currentNode = lastNode.getValues().get(val);
            } else {
                justCreated = false;
            }

            if (position == keys.size() - 1 && !justCreated) {
                Node item = lastNode.getValues().get(val);
                item.setData(data);
                item.setModifier(custom);
            }
            position++;
        }

        if (keys.size() == 0) {
            this.tree.setData(data);
            this.tree.setModifier(custom);
        }

        return this;
    }


    /**
     * Register a pattern, and the object that will be returned if an input
     * matches.
     * <p>
     *
     * @param  pat  a string defintiion of the pattern
     *              takes the form key:value,key:value
     *              eg "a:1,b:2"
     * @param  data the data to store for this pattern
     * @return      a reference to this to allow chaining of add calls
     */
    public Patrun add(String pat, Object data) {

        return add(convertPatternString(pat), data);
    }


    /**
     * Return the unique match for this subject, or null if not found. The
     * properties of the subject are matched against the patterns previously
     * added, and the most specifc pattern wins. Unknown properties in the
     * subject are ignored. If exact is set to true then all properties of a pattern must match.
     * exact defaults to false.
     * <p>
     *
     * @param  pat  a map of the pattern to look for
     * @param  exact set to true if the match must exactly match the pattern
     * @return      the data stored against the matching pattern or null if no match
     */
    public Object find(Map<String, String> pat, Boolean exact) {
        Vector<String> keys = sortKeys(pat);

        Node currentNode = this.tree;
        Node lastGoodNode = currentNode;
        Vector<String> foundKeys = new Vector<String>();
        Object lastData  = this.tree.getData();
        Modifier lastModifier = this.tree.getModifier();
        Vector<Node> stars = new Vector<Node>();
        int keyPointer = 0;

        while (keyPointer < keys.size()) {
            String key = keys.get(keyPointer);
            String val = pat.get(key);

            currentNode = currentNode.getValues().get(key);

            if (currentNode != null) {
                currentNode = currentNode.getValues().get(val);
            }

            if (currentNode != null) {
                if (lastGoodNode.getValues().size() > 0) {
                    stars.add(lastGoodNode);
                }

                lastGoodNode = currentNode;
                foundKeys.add(key);
                if (lastGoodNode.getData() != null) {
                    lastData = lastGoodNode.getData();
                }
                lastModifier = lastGoodNode.getModifier();

                keyPointer++;

            } else if (lastData == null && stars.size() > 0) {
                currentNode = stars.get(stars.size() - 1);
                stars.removeElementAt(stars.size() - 1);
                lastGoodNode = currentNode;

            } else {
                currentNode = lastGoodNode;
                keyPointer++;
            }

        }

        if (exact && foundKeys.size() != keys.size()) {
            lastData = null;
        }

        if (lastModifier != null) {
            lastData = lastModifier.find(this, pat, lastData);
        }

        return lastData;

    }

    /**
     * Return the unique match for this subject, or null if not found. The
     * properties of the subject are matched against the patterns previously
     * added, and the most specifc pattern wins. Unknown properties in the
     * subject are ignored.
     * <p>
     *
     * @param  pat  a Map representation of the pattern to look for
     * @return      the data stored against the matching pattern or null if no match
     */
    public Object find(Map<String, String> pat) {
        return find(pat, false);
    }

    /**
     * Return the unique match for this subject, or null if not found. The
     * properties of the subject are matched against the patterns previously
     * added, and the most specifc pattern wins. Unknown properties in the
     * subject are ignored.
     * <p>
     *
     * @param  pat  a string representation of the pattern to look for
     * @return      the data stored against the matching pattern or null if no match
     */
    public Object find(String pat) {
        return find(convertPatternString(pat), false);
    }

    /**
     * Return the unique match for this subject, or null if not found. The
     * properties of the subject are matched against the patterns previously
     * added, and the most specifc pattern wins. Unknown properties in the
     * subject are ignored.
     * <p>
     *
     * @param  pat  a string representation of the pattern to look for
     * @param  exact set to true if the match must exactly match the pattern
     * @return      the data stored against the matching pattern or null if no match
     */
    public Object find(String pat, Boolean exact) {
        return find(convertPatternString(pat), exact);
    }

    /**
     * Remove this pattern, and it's object, from the matcher.
     * <p>
     *
     * @param  pat  a Map representation of the pattern to remove
     */
    public void remove(Map<String, String> pat) {
        Vector<String> keys = sortKeys(pat);

        Node currentNode = this.tree;
        Node lastGoodNode = currentNode;
        Vector<String> foundKeys = new Vector<String>();
        String val = "";
        Node lastParent = currentNode;

        for (String key : keys) {
            val = pat.get(key);

            currentNode = currentNode.getValues().get(key);

            if (currentNode != null) {
                lastParent = currentNode;
                lastGoodNode = currentNode;
            }

            if (currentNode != null) {
                currentNode = currentNode.getValues().get(val);
            }

            if (currentNode != null) {
                lastGoodNode = currentNode;
                foundKeys.add(key);
            }
        }

        //found a match so delete the data element
        if (foundKeys.size() == keys.size()) {
            Node item;

            if (pat.size() == 0) {
                item = this.tree;
            } else {
                item = lastParent.getValues().get(val);
            }

            Boolean okToDel = true;

            if (lastGoodNode.getModifier() != null) {
                okToDel = lastGoodNode.getModifier().remove(this, pat, item.getData());
            }
            if (okToDel) {
                item.setData(null);
                item.setModifier(null);
            }
        }

    }

    /**
     * Remove this pattern, and it's object, from the matcher.
     * <p>
     *
     * @param  pat  a String representation of the pattern to remove
     */
    public void remove(String pat) {
        remove(convertPatternString(pat));
    }

    /**
     * Return the list of registered patterns that contain this partial
     * pattern. You can use wildcards for property values.  Omitted values
     * are *not* equivalent to a wildcard of _"*"_, you must specify each
     * property explicitly. You can provide a second boolean
     * parameter, _exact_. If true, then only those patterns matching the
     * pattern-partial exactly are returned.
     * <p>
     *
     * @param  pat  a Map representation of the pattern to match
     * @param  exact set to true if the match must exactly match the pattern
     * @return      a Vector of matched Patterns
     */
    public Vector<Pattern> list(Map<String, String> pat, Boolean exact) {
        Vector<Pattern> items = new Vector<Pattern>();
        Vector<String> keyMap = new Vector<String>();

        if (pat == null) {
            pat = new HashMap<String, String>();
        }

        if (this.tree.getData() != null) {
            items.add(createMatchList(keyMap, this.tree.getData(), this.tree.getModifier()));
        }

        if (this.tree != null) {
            descendTree(items, pat, exact, true, this.tree.getValues(), keyMap);
        }

        return items;
    }

    /**
     * Return the list of all registered patterns.
     * <p>
     *
     * @return      a Vector of matched Patterns
     */
    public Vector<Pattern> list() {
        return list(new HashMap<String, String>(), false);
    }

    /**
     * Return the list of registered patterns that contain this partial
     * pattern. You can use wildcards for property values.  Omitted values
     * are *not* equivalent to a wildcard of _"*"_, you must specify each
     * property explicitly.
     * <p>
     *
     * @param  pat  a Map representation of the pattern to match
     * @return      a Vector of matched Patterns
     */
    public Vector<Pattern> list(Map<String, String> pat) {
        return list(pat, false);
    }

    /**
     * Return the list of registered patterns that contain this partial
     * pattern. You can use wildcards for property values.  Omitted values
     * are *not* equivalent to a wildcard of _"*"_, you must specify each
     * property explicitly.
     * <p>
     *
     * @param  pat  a String representation of the pattern to match
     * @return      a Vector of matched Patterns
     */
    public Vector<Pattern> list(String pat) {
        return list(convertPatternString(pat), false);
    }

    /**
     * Return the list of registered patterns that contain this partial
     * pattern. You can use wildcards for property values.  Omitted values
     * are *not* equivalent to a wildcard of _"*"_, you must specify each
     * property explicitly. You can provide a second boolean
     * parameter, _exact_. If true, then only those patterns matching the
     * pattern-partial exactly are returned.
     * <p>
     *
     * @param  pat  a String representation of the pattern to match
     * @param  exact set to true if the match must exactly match the pattern
     * @return      a Vector of matched Patterns
     */
    public Vector<Pattern> list(String pat, Boolean exact) {
        return list(convertPatternString(pat), exact);
    }

    /**
     * Generate a string representation of the decision tree for debugging.
     * <p>
     *
     * @return   a String representation of the decision tree
     */
    public String toString() {

        Vector<Pattern> items = this.list();

        Vector<String> data =  new Vector<String>();

        for (Pattern p : items) {

            data.add(formatMatch(p.getMatch()) + " -> <" + p.getData() + ">");
        }

        return StringUtils.join(data, "\n");
    }


    /**
     * Generate a string representation of the decision tree for debugging.
     * <p>
     *
     * @param  printer  an object i9mplementing the Printer interface to allow for customising the string representation of the data objects
     * @return   a String representation of the decision tree
     */
    public String toString(Printer printer) {

        Vector<Pattern> items = this.list();

        Vector<String> data =  new Vector<String>();

        for (Pattern p : items) {

            data.add(formatMatch(p.getMatch()) + " -> <" + printer.toString(p.getData()) + ">");
        }

        return StringUtils.join(data, "\n");
    }


    /**
     * A method to recursively traverse the tree to build a list of paterns matching the provided input.
     * <p>
     *
     * @param  items  a collection to hold the matched items
     * @param pat the pattern to match tree items against
     * @param exact specify if patterns must mathc exactly
     * @param rootLevel flag to indicate if the crrentbranch is the root level of the tree
     * @param values a collection of items stopred at the current branch level
     * @param keyMap a collection of the keys traversed
     */
    private void descendTree(Vector<Pattern> items, Map<String, String> pat, Boolean exact, Boolean rootLevel, Map<String, Node> values, Vector<String> keyMap) {

        Vector<String> localKeyMap = new Vector<String>();

        localKeyMap.addAll(0, keyMap);

        Vector<String> keys = new Vector<String>();
        for (String key : values.keySet()) {
            keys.add(key);
        }
        Collections.sort(keys);

        for (String key : keys) {
            Node val = values.get(key);

            if (rootLevel) {
                keyMap = new Vector<String>();
            }

            if (val.getData() == null && val.getValues().size() > 0) {
                Vector<String> newMap = new Vector<String>(keyMap);
                newMap.add(key);
                descendTree(items, pat, exact, false, val.getValues(), newMap);

            } else if (val.getData() != null) {
                localKeyMap.clear();
                localKeyMap.addAll(0, keyMap);
                localKeyMap.add(val.getKey());
                if (validatePatternMatch(pat, exact, localKeyMap)) {
                    items.add(createMatchList(localKeyMap, val.getData(), val.getModifier()));
                }

                if (val.getValues().size() > 0) {
                    Vector<String> newMap = new Vector<String>(keyMap);
                    newMap.add(key);
                    descendTree(items, pat, exact, false, val.getValues(), newMap);
                }
            }
        }
    }

    /**
     * A method to format a match intop a readable string
     * <p>
     *
     * @param  items  the pattern to format
     * @return a string representation of the pattern
     */
    private String formatMatch(Map<String, String> items) {
        Vector<String> points = new Vector<String>();

        Vector<String> keys  = sortKeys(items);

        for (String key : keys) {

            String val = items.get(key);

            points.add(key + ":" + val);
        }

        return StringUtils.join(points, ", ");
    }

    /**
     * A method to construct a Pattern object based on a node in the tree
     * <p>
     *
     * @param  keyMap  the key pattern for this node
     * @param dataItem the data stored at this node
     * @param modifier the custom modifier applicable to this node
     * @return a Pattern object
     */
    private Pattern createMatchList(Vector<String> keyMap, Object dataItem, Modifier modifier) {

        Map<String, String> keys =  new HashMap<String, String>();

        for (int i = 0; i < keyMap.size(); i += 2) {
            if (i + 1 < keyMap.size()) {
                keys.put(keyMap.get(i), keyMap.get(i+1));
            }
        }

        return new Pattern(keys, dataItem, modifier);


    }

    /**
     * Validates if a node in the tree matches the provided pattern
     * <p>
     *
     * @param  pat  the pattern to match
     * @param exact set to true if the pattern must match exactly
     * @param matchedKeys the keys for the current node
     * @return true if node matches
     */
    private Boolean validatePatternMatch(Map<String, String> pat, Boolean exact, Vector<String> matchedKeys) {
        Vector<String> keys = sortKeys(pat);

        if (keys.size() == 0) {
            return true;
        }

        Map<String, String> pathMap = convertListToMap(matchedKeys);

        Boolean matched = true;

        for (String key : pat.keySet()) {
            String val = pat.get(key);
            if (pathMap.get(key) == null || !gexval(val, pathMap.get(key))) {
                matched = false;
                break;
            }
        }

        if (exact && pat.size() != pathMap.size()) {
            matched = false;
        }

        return matched;
    }

    /**
     * A method to convert a list of keys into a Map
     * <p>
     *
     * @param  listItems  the keys to convert
     * @return a Map representation of the pattern
     */
    private Map<String, String> convertListToMap(Vector<String> listItems) {
        Map<String, String> mapData = new HashMap<String, String>();

        for (int k = 0; k < listItems.size(); k+=2) {
            mapData.put(listItems.get(k), listItems.get(k + 1));
        }

        return mapData;
    }

    /**
     * A method to sort keys in a Map
     * <p>
     *
     * @param  pat  the pattern to sort
     * @return a Vector of sorted keys
     */
    private Vector<String> sortKeys(Map<String, String> pat) {
        Vector<String> keys = new Vector<String>();

        for (String key : pat.keySet()) {
            keys.add(key);
        }
        Collections.sort(keys);

        return keys;
    }

    /**
     * A method to convert the string representation of a pattern into a Mapg
     * <p>
     *
     * @param  pat  the string pattern
     * @return a Map construct of the pattern
     */
    private Map<String, String> convertPatternString(String pat) {
        Map<String, String> map = new HashMap<String, String>();


        String[] items = StringUtils.split(pat, ",");

        for (String k : items) {
            String item = StringUtils.trim(k);

            if (item.length() > 0) {
                String[] parts = StringUtils.split(item, ":");
                if (parts.length == 2) {
                    map.put(StringUtils.trim(parts[0]), StringUtils.trim(parts[1]));
                }
            }
        }

        return map;
    }

    /**
     * A method to regex match a value against a pattern
     * <p>
     *
     * @param  pattern the simple regex pattern (* for wildcard, ? for single character wildcard)
     * @param value the value to compare against the pattern
     * @return true if there is a match
     */
    private Boolean gexval(String pattern, String value) {

        pattern = pattern.replaceAll("([-\\[\\]{}()*+?.,\\\\^$|#\\s])", "\\\\$1");

        // use [\s\S] instead of . to match newlines

        pattern = pattern.replaceAll("\\\\\\*", "[\\\\s\\\\S]*");

        pattern = pattern.replaceAll("\\\\\\?", "[\\\\s\\\\S]");

        // escapes ** and *?
        pattern = pattern.replaceAll("\\[\\\\s\\\\S\\]\\*\\[\\\\s\\\\S\\]\\*", "\\\\\\*");

        pattern = pattern.replaceAll("\\[\\\\s\\\\S\\]\\*\\[\\\\s\\\\S\\]", "\\\\\\?");

        pattern = "^" + pattern + "$";

        java.util.regex.Pattern p = java.util.regex.Pattern.compile(pattern);

        Matcher m = p.matcher(value);
        return m.matches();

    }


}

