package com.natradac.android.tokeninterceptor

import android.content.Context
import android.os.Build
import com.google.gson.Gson
import com.natradac.android.tokeninterceptor.db.PreferenceHelper
import com.natradac.android.tokeninterceptor.db.PreferenceHelper.set
import com.natradac.android.tokeninterceptor.db.PreferenceHelper.get
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import java.time.Instant

object RefreshTokenManager {

    fun init(
        context: Context,
        prefName: String,
        endpoint: String,
        requestMethod: String,
        jsonRequestBody: String? = null
    ) {
        val pref = PreferenceHelper.defaultPrefs(context, prefName)
        pref[PreferenceHelper.ENDPOINT] = endpoint
        pref[PreferenceHelper.REQUEST_METHOD] = requestMethod
        jsonRequestBody?.also {
            pref[PreferenceHelper.RESPONSE_BODY_JSON] = it
        }
    }

    fun updateResponseKey(
        context: Context,
        prefName: String,
        accessToken: String,
        refreshToken: String,
        accessValid: String,
        refreshValid: String
    ) {
        val pref = PreferenceHelper.defaultPrefs(context, prefName)
        pref[PreferenceHelper.TOKEN_KEY] = accessToken
        pref[PreferenceHelper.REFRESH_TOKEN_KEY] = refreshToken
        pref[PreferenceHelper.ACCESS_VALID_KEY] = accessValid
        pref[PreferenceHelper.REFRESH_VALID_KEY] = refreshValid
    }

    fun updateToken(
        context: Context,
        prefName: String,
        token: String,
        refreshToken: String,
        accessValid: Long,
        refreshValid: Long
    ) {
        val pref = PreferenceHelper.defaultPrefs(context, prefName)
        pref[PreferenceHelper.TOKEN] = token
        pref[PreferenceHelper.REFRESH_TOKEN] = refreshToken
        pref[PreferenceHelper.ACCESS_VALID] = accessValid
        pref[PreferenceHelper.REFRESH_VALID] = refreshValid
        pref[PreferenceHelper.TIME_STAMP] = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            Instant.now().epochSecond
        } else {
            System.currentTimeMillis() / 1000L
        }
    }

    fun getRequest(
        context: Context,
        prefName: String
    ): Request {
        val pref = PreferenceHelper.defaultPrefs(context, prefName)
        var endpoint = pref[PreferenceHelper.ENDPOINT] ?: "".replace(
            "{{${pref[PreferenceHelper.TOKEN_KEY, "access_token"]}}}",
            pref[PreferenceHelper.TOKEN] ?: "",
            false
        )
        endpoint = endpoint.replace(
            "{{${pref[PreferenceHelper.REFRESH_TOKEN_KEY, "refresh_token"]}}}",
            pref[PreferenceHelper.REFRESH_TOKEN]
                ?: "",
            false
        )

        val modifiedJson : String? = if(pref.getString(PreferenceHelper.RESPONSE_BODY_JSON, null) != null){
            pref.getString(PreferenceHelper.RESPONSE_BODY_JSON, "")?.replace(
                "{{${pref[PreferenceHelper.REFRESH_TOKEN_KEY, "refresh_token"]}}}",
                pref[PreferenceHelper.REFRESH_TOKEN]
                    ?: "",
                false
            )
        }else null

        val modifiedRequestBody: RequestBody? =
            modifiedJson?.toRequestBody("application/json".toMediaTypeOrNull())

        return Request.Builder()
            .method(pref[PreferenceHelper.REQUEST_METHOD] ?: "", modifiedRequestBody)
            .url(endpoint)
            .build()
    }


}