package com.nanorep.nanoclient;

import android.graphics.Bitmap;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.style.StyleSpan;
import android.util.Base64;
import android.util.Log;

import com.nanorep.nanoclient.Channeling.NRChanneling;
import com.nanorep.nanoclient.Connection.NRConnection;
import com.nanorep.nanoclient.Connection.NRError;
import com.nanorep.nanoclient.Connection.NRErrorCodes;
import com.nanorep.nanoclient.Connection.NRUtilities;
import com.nanorep.nanoclient.Handlers.NRErrorHandler;
import com.nanorep.nanoclient.Interfaces.NRExtraDataListener;
import com.nanorep.nanoclient.Interfaces.NRQueryResult;
import com.nanorep.nanoclient.Log.NRLogger;
import com.nanorep.nanoclient.RequestParams.NRFAQLikeParams;
import com.nanorep.nanoclient.RequestParams.NRSearchLikeParams;
import com.nanorep.nanoclient.Response.NRConfiguration;
import com.nanorep.nanoclient.Response.NRFAQAnswer;
import com.nanorep.nanoclient.Response.NRSearchResponse;
import com.nanorep.nanoclient.Response.NRSuggestions;
import com.nanorep.nanoclient.exception.NRConnectionException;
import com.nanorep.nanoclient.model.DefaultResponse;
import com.nanorep.nanoclient.model.HelloResponse;
import com.nanorep.nanoclient.model.SupportLabel;
import com.nanorep.nanoclient.network.OnDataResponse;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import kotlin.jvm.functions.Function1;

import static com.nanorep.nanoclient.Connection.NRErrorCodes.CONNECTION;
import static com.nanorep.nanoclient.Connection.NRErrorCodes.SESSION;
import static com.nanorep.nanoclient.Connection.NRErrorCodes.noSuggestions;
import static com.nanorep.nanoclient.Connection.NRErrorCodes.noSuggestionsInCache;
import static com.nanorep.nanoclient.Interfaces.ArticleParamsKeys.ArticleId;

public class Nanorep {

    private static String TAG = Nanorep.class.getName();

    private static Nanorep instance;
    private int httpRequestTimeout;
    private Runnable keepAliveRunnable;
    private int keepAliveCounter;
    private boolean helloInProgress;
    private Handler keepAliveHandler;
    private String currentAnswerID;
    private Object searchViewsProvider;
    private boolean startedWithCategories = true;

    public Object getSearchViewsProvider() {
        return searchViewsProvider;
    }

    public void setSearchViewsProvider(Object searchViewsProvider) {
        this.searchViewsProvider = searchViewsProvider;
    }

    /**
     * @deprecated widget listener should be initialized independently
     * */
    @Deprecated
    public static void initialize(AccountParams accountParams, NanoRepWidgetListener widgetListener) {
        instance = new Nanorep();
        instance.init(accountParams, widgetListener);
    }

    public static void initialize(AccountParams accountParams) {
        instance = new Nanorep();
        instance.init(accountParams);
    }

    public static Nanorep getInstance() {
        if (instance == null) {
            throw new RuntimeException("Nanorep is not initialized");
        }
        return instance;
    }

    public static void clearInstance() {
        instance = null;
    }

    private AccountParams mAccountParams;
    private NRConfiguration mCnf;
    private NRLogger nrLogger;

    private String mSessionId;
    private long mDelay;
    private HashMap<String, NRSearchResponse> mCachedSearches;
    private HashMap<String, NRSuggestions> mCachedSuggestions;
    private HashMap<String, ArrayList<OnFAQAnswerFetchedListener>> faqRequestListenersMap;
    private NanoRepWidgetListener widgetListener;
    private String searchQuery;
    private boolean keepAliveRunning;
    private String previousQueryText;
    private ArrayList<Spannable> previousSuggestions;

    public NanoRepWidgetListener getWidgetListener() {
        return widgetListener;
    }

    public void setWidgetListener(NanoRepWidgetListener widgetListener) {
        this.widgetListener = widgetListener;
    }

    public String getSessionId() {
        return mSessionId;
    }

    public int getHttpRequestTimeout() {
        return httpRequestTimeout;
    }

    public static boolean isInitialized() {
        return instance != null;
    }

    public void setHttpRequestTimeout(int httpRequestTimeout) {
        if (httpRequestTimeout >= 3 && httpRequestTimeout <= 60) {
            this.httpRequestTimeout = httpRequestTimeout;
        }
    }

    public interface NanoRepWidgetListener extends ErrorListener {
        void personalInfoWithExtraData(String channelDescription, @Nullable String extraData, NRExtraDataListener listener);

        void onSubmitSupportForm(String formData, ArrayList<String> fileUploadPaths);

        void onConfigurationFetched();

        void onCachedImageRequest(String url, NRCachedImageResponder responder);

        void onInitializationFailure();

        void onEmptyDataResponse();

        void onChannel(NRChanneling.NRChannelingType channelingType, Object extraData);
    }

    public interface ErrorListener{
        void onError(NRError error);
    }

    public interface NRCachedImageResponder {
        void onBitmapResponse(Bitmap bitmap);
    }

    public interface OnSearchResultsFetchedListener {
        void onSearchResponse(NRSearchResponse response, NRError error);
    }

    public interface OnSuggestionsFetchedListener {
        void onSuggestionsFetched(NRSuggestions suggestions, NRError error);
    }

    public interface OnLikeSentListener {
        void onLikeSent(boolean success);
    }

    public interface OnFAQAnswerFetchedListener {
        void onFAQAnswerFetched(NRFAQAnswer faqAnswer, NRError error);
    }

    public interface OnConfigurationFetchedListener {
        void onConfigurationFetched(NRError error);
    }

    public void setAccountParams(AccountParams accountParams) {
        this.mAccountParams = accountParams;
    }

    /**
     * @deprecated widget listener should be initialized independently
     * */
    @Deprecated
    private void init(AccountParams accountParams, NanoRepWidgetListener widgetListener) {
        keepAliveHandler = new Handler();
        mCachedSearches = new HashMap<>();
        clearCache();
        this.widgetListener = widgetListener;
        if (getAccountParams() != null && accountParams.compareTo(getAccountParams()) != 0) {
            mCnf = null;
        }

        setAccountParams(accountParams);
        this.nrLogger = new NRLogger();

        fetchConfiguration();
    }

    private void init(AccountParams accountParams) {

        keepAliveHandler = new Handler();

        mCachedSearches = new HashMap<>();

        clearCache();

        if (getAccountParams() != null && accountParams.compareTo(getAccountParams()) != 0) {
            mCnf = null;
        }

        setAccountParams(accountParams);
        this.nrLogger = new NRLogger();

        fetchConfiguration();
    }

    /**
     * Sets the session id to null
     */
    public void clearSession() {
        if (mSessionId != null) {
            mSessionId = null;
            if (keepAliveHandler != null) {
                keepAliveHandler.removeCallbacksAndMessages(null);
                keepAliveRunning = false;
            }
        }
    }

    private HashMap<String, NRSearchResponse> getCachedSearches() {
        if (mCachedSearches == null) {
            mCachedSearches = new HashMap<>();
        }
        return mCachedSearches;
    }

    private HashMap<String, NRSuggestions> getCachedSuggestions() {
        if (mCachedSuggestions == null) {
            mCachedSuggestions = new HashMap<>();
        }
        return mCachedSuggestions;
    }

    private HashMap<String, ArrayList<OnFAQAnswerFetchedListener>> getFaqRequestListenersMap() {
        if (faqRequestListenersMap == null) {
            faqRequestListenersMap = new HashMap<>();
        }
        return faqRequestListenersMap;
    }

    private void clearCache() {
        if (mCachedSearches != null) {
            Log.i("formData", "cache cleared");
            mCachedSearches.clear();
        }
    }

    private void keepAlive(final long interval) {
        if (mSessionId == null) {
            keepAliveHandler.removeCallbacksAndMessages(null);
            return;
        }

        keepAliveCounter++;
        Log.i("nanorep","keepalive called (" + keepAliveCounter + ")");
        if (keepAliveRunnable == null) {
            keepAliveRunnable = new Runnable() {
                public void run() {
                    keepAlive(interval);
                }
            };
        }
        keepAliveHandler.postDelayed(keepAliveRunnable, interval);


        if(keepAliveRunning) {
            Log.i("nanorep","keepalive - in progress deteceted");
            return;
        }

        Log.i("nanorep","keepalive - sending http request");
        keepAliveRunning = true;

        Uri.Builder uri = mAccountParams.getUri();
        uri.appendEncodedPath("api/widget/v1/keepAlive.js");
        uri.appendQueryParameter("sid", mSessionId);
        NRConnection.connection(uri, new OnDataResponse<String>() {
            @Override
            public void onSuccess(String response) {
                keepAliveRunning = false;
                Log.i("nanorep", "keepalive - success " + response);
            }

            @Override
            public void onError(NRConnectionException error) {
                Log.i("nanorep", "keepalive - error ");
                keepAliveRunning = false;
                keepAliveHandler.removeCallbacks(keepAliveRunnable);

                createSession(new OnDataResponse<HelloResponse>() {
                    @Override
                    public void onSuccess(HelloResponse response) {

                    }

                    @Override
                    public void onError(NRConnectionException error) {
                        widgetListener.onInitializationFailure();
                    }
                });
            }
        });
    }

    /**
     * * Reports channeling to the Nanorep analytics
     * @param nrQueryResult - The current result (session / sessionLess)
     * @param channeling - The channel
     */

    public void reportChanneling(NRQueryResult nrQueryResult, NRChanneling channeling) {

        Uri.Builder uri = mAccountParams.getUri();
        String articleId = "0";

        if (nrQueryResult != null) {
            articleId = nrQueryResult.getId();
        }

        uri.appendQueryParameter("articleId", articleId);
        uri.appendQueryParameter("apiName", channeling.getName());
        
        if (nrQueryResult == null || nrQueryResult instanceof NRFAQAnswer) {
            // SessionLess
            uri.appendEncodedPath("api/channeling/v1/report");
            uri.appendQueryParameter("account", mAccountParams.getAccount());
            uri.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
            uri.appendQueryParameter("trafficSource", "FAQ");
            uri.appendQueryParameter("reportedEscalationType", String.valueOf(channeling.getChannelCode()));

        } else if (mSessionId != null && !mSessionId.isEmpty()){
            // With Session
            uri.appendEncodedPath("api/channeling/v1/escalate");
            uri.appendQueryParameter("type", String.valueOf(channeling.getChannelCode()));
            uri.appendQueryParameter("sid", mSessionId);
            uri.appendQueryParameter("text", searchQuery);
        }

        NRConnection.connection(uri, new OnDataResponse<String>() {
            @Override
            public void onSuccess(String response) {
                Log.d("ReportSent", "Report sent successfully");
            }

            @Override
            public void onError(NRConnectionException error) {
                Log.d("ReportError", "Error on channeling reporting");
            }
        });
    }

    private void executeRequest(final Uri.Builder uriBuilder, final NRConnection.Listener executeListener) {
        if (mSessionId != null) {
            uriBuilder.appendQueryParameter("sid", mSessionId);

            NRConnection.getInstance().connectionWithRequest(uriBuilder.build(), new NRConnection.Listener() {
                @Override
                public void response(final Object responseParam, int status, final NRError error) {
                    if (error != null || responseParam == null) {

                        Uri.Builder uri = mAccountParams.getUri();
                        uri.appendEncodedPath("api/widget/v1/keepAlive.js");
                        uri.appendQueryParameter("sid", mSessionId);
                        NRConnection.connection(uri, new OnDataResponse<String>() {
                            @Override
                            public void onSuccess(String response) {
                                Log.i("nanorep", "keepalive - success " + response);
                                onError(new NRConnectionException("response error"));
                            }

                            @Override
                            public void onError(NRConnectionException error) {
                                Log.i("nanorep", "keepalive - error, calling createSession ");
                                recreateSession(executeListener, responseParam);
                            }
                        });
                    } else {
                        executeListener.response(responseParam, 200, null);
                    }
                }

                @Override
                public void log(String tag, String msg) {

                }
            });

        } else {
            createSession(new OnDataResponse<HelloResponse>() {
                @Override
                public void onSuccess(HelloResponse response) {
                    executeRequest(uriBuilder, executeListener);
                }

                @Override
                public void onError(NRConnectionException error) {
                    Nanorep.getInstance().getWidgetListener().onError(NRError.error(TAG, SESSION, "session creation error"));
                }
            });
        }
    }

    /**
     * Creates a new session
     * @param executeListener
     * @param responseParam
     */
    private void recreateSession(final NRConnection.Listener executeListener, final Object responseParam) {

        clearSession();

        createSession(new OnDataResponse<HelloResponse>() {
            @Override
            public void onSuccess(HelloResponse response) {
                executeListener.response(responseParam, 200, null);
            }

            @Override
            public void onError(NRConnectionException error) {
                widgetListener.onInitializationFailure();
            }
        });
    }

    private void createSession(final OnDataResponse<HelloResponse> sessionListener) {
        final Uri.Builder uri = mAccountParams.getUri();
        uri.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
        uri.appendEncodedPath("api/widget/v1/hello.js");
        uri.appendQueryParameter("nostats", "false");
        uri.appendQueryParameter("url", "mobile");
        if (mAccountParams.getContext() != null) {
            uri.appendQueryParameter("context", mAccountParams.getContext());
        }

        NRConnection.connection(uri, new OnDataResponse<HelloResponse>() {
            @Override
            public void onSuccess(HelloResponse response) {
                if (response != null) {
                    mSessionId = response.getSessionId();
                    Log.i("nanorep", "createSession success - calling keepalive");
                    keepAlive(response.getTimeout() * 1000 / 2); //TODO: every request post should clearSession the keepAlive timeout
                    sessionListener.onSuccess(response);
                } else {
                    Nanorep.getInstance().getWidgetListener().onError(NRError.error(TAG, SESSION, "session creation error"));
                }
            }

            @Override
            public void onError(NRConnectionException error) {
                NRErrorHandler.getInstance().handleError(CONNECTION);
                Nanorep.getInstance().getWidgetListener().onError(NRError.error(TAG, CONNECTION, error.getMessage()));
                Nanorep.this.getFaqRequestListenersMap().remove(currentAnswerID);
            }
        });
    }

    @Deprecated
    private void hello(final NRConnection.Listener listener) {
        if (helloInProgress) {
            return;
        }
        final Uri.Builder _uriBuilder = mAccountParams.getUri();
        _uriBuilder.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
        _uriBuilder.appendEncodedPath("api/widget/v1/hello.js");
        _uriBuilder.appendQueryParameter("nostats", "false");
        _uriBuilder.appendQueryParameter("url", "mobile");
        String context = mAccountParams.getContext();
        if (context != null) {
            _uriBuilder.appendQueryParameter("context", context);
        }

        helloInProgress = true;
        NRConnection.getInstance().connectionWithRequest(_uriBuilder.build(), new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                helloInProgress = false;
                if (error != null) {
                   Nanorep.getInstance().getWidgetListener().onError(error);
                } else {
                    mSessionId = (String) ((HashMap) responseParam).get("sessionId");
                    mDelay = ((Integer) ((HashMap) responseParam).get("timeout")).longValue() * 500;
                    Log.i("nanorep","_hello success - calling keepalive");
                    keepAlive(mDelay);

                    listener.response(responseParam, status, error);
                }
            }

            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void removeFromSearchCache(String text) {
        if (mCachedSearches != null && mCachedSearches.containsKey(text)) {
            mCachedSearches.remove(text);
        }
    }

    public void searchText(final String text, final OnSearchResultsFetchedListener onSearchResultsFetchedListener) {
        if (mCachedSearches != null && mCachedSearches.get(text) != null) {
            onSearchResultsFetchedListener.onSearchResponse(mCachedSearches.get(text), null);
        } else if (text != null && text.length() > 0) {
            String encodedText = null;
            try {
                encodedText = URLEncoder.encode(text, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
            Uri.Builder uriBuilder = mAccountParams.getUri();
            uriBuilder.appendEncodedPath("api/widget/v1/q.js");
            uriBuilder.appendQueryParameter("text", text);
            executeRequest(uriBuilder, new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if (error != null) {
                        onSearchResultsFetchedListener.onSearchResponse(null, error);
                    } else if (responseParam != null) {
                        NRSearchResponse response = new NRSearchResponse((HashMap) responseParam);
                        Nanorep.this.getCachedSearches().put(text, response);
                        onSearchResultsFetchedListener.onSearchResponse(response, null);
                    }
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    public void getContextValues(final String context, OnDataResponse<String> listener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/kb/v1/contextValues");
        uriBuilder.appendQueryParameter("contexts", context);
        uriBuilder.appendQueryParameter("account", getAccountParams().getAccount());
        uriBuilder.appendQueryParameter("kb", getAccountParams().getKnowledgeBase());
        uriBuilder.appendQueryParameter("cb", "");
        NRConnection.connection(uriBuilder, listener);
    }

    public void setContext(final String context, final OnDataResponse<DefaultResponse> listener) {
        if (mSessionId == null) {
            createSession(new OnDataResponse<HelloResponse>() {
                @Override
                public void onSuccess(HelloResponse response) {
                    setSessionParams(context, listener);
                }

                @Override
                public void onError(NRConnectionException error) {
                    widgetListener.onEmptyDataResponse();
                }
            });

        } else {
            setSessionParams(context, listener);
        }
    }

    private void setSessionParams(String context, OnDataResponse<DefaultResponse> listener) {
        if (context == null) {
            clearCache();
            mAccountParams.clearSelectedContext();
        } else {
            mAccountParams.setSelectedContext(context);
            context = mAccountParams.getSelectedContext();
        }
        clearCache();
        String initContext = mAccountParams.getContext();
        if (initContext != null && initContext.length() > 0) {
            context = initContext + "," + context;
        }
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/widget/v1/setParams");
        uriBuilder.appendQueryParameter("context", context == null ? "clearContext:" : context);
        uriBuilder.appendQueryParameter("sid", mSessionId);
        uriBuilder.appendQueryParameter("account", getAccountParams().getAccount());
        NRConnection.connection(uriBuilder, listener);
    }

    /**
     * Updates the channels as a result of to the user's feedback
     * @param articleId
     * @param isPositive
     * @param listener
     */
    public void updateDislikeChanneling(String articleId, boolean isPositive, final OnDataResponse<List<NRChanneling>> listener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/channeling/v1/get");
        uriBuilder.appendQueryParameter("account", getAccountParams().getAccount());
        uriBuilder.appendQueryParameter("kb", getAccountParams().getKnowledgeBase());
        uriBuilder.appendQueryParameter("articleId", articleId);
        uriBuilder.appendQueryParameter("clientState", isPositive ? "positiveFeedback" : "negativeFeedback");

        if (mAccountParams.getCurrentLabelContext() != null) {
            uriBuilder.appendQueryParameter("context", mAccountParams.getCurrentLabelContext());
        }

        executeRequest(uriBuilder, new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    listener.onError(new NRConnectionException("error on updating dislike channels"));
                } else if (responseParam != null) {
                    List<HashMap<String, Object>> channeling = (List<HashMap<String, Object>>) ((HashMap) responseParam).get("rechanneling");

                    if (channeling == null) {
                        return;
                    }

                    List<NRChanneling> nrChannels = new ArrayList<>();
                    for (HashMap<String, Object> channel : channeling) {
                        nrChannels.add(NRChanneling.channelForParams(channel));
                    }
                    listener.onSuccess(nrChannels);
                }
            }


            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    /**
     * If there is no context at the label, the faq's would be according to the label.
     * Else, the faq's would be according to the context.
     */
    public void updateFaqList(String hexLabelId, String nrContext, NRConnection.Listener listener) {
        AccountParams accountParams = instance.getAccountParams();
        Uri.Builder uri = accountParams.getUri();
        uri.appendEncodedPath("api/kb/v1/faq");
        uri.appendQueryParameter("account", accountParams.getAccount());
        uri.appendQueryParameter("kb", accountParams.getKnowledgeBase());
        uri.appendQueryParameter("allContextsMandatory", (accountParams.isContextExclusivity()? "true" : "false"));
        uri.appendQueryParameter("items", "1000");

        if (nrContext == null || nrContext.isEmpty()) {
            if (hexLabelId != null /*&& !hexLabelId.equals("0") -> needed for the future fix */) {
                uri.appendQueryParameter("labelId", hexLabelId);
            } else if (!accountParams.isLabelsMode() && accountParams.getLabelId() != 0) {
                uri.appendQueryParameter("labelId", Integer.toHexString(accountParams.getLabelId()).toUpperCase());
            /*  -> A needed fix - filter faqs by the initial context:
                }  else if (accountParams.isFaqsMode()){
                appendDecodeContext(uri, accountParams.getContext()); */
            }
        } else {
            String ctx;  // A string that will include the latest context for the faq's call
            String currentContext = accountParams.getContext(); // The context that the user started Nanorep with

            if (accountParams.isSupportCenterMode() && hexLabelId != null) {
                uri.appendQueryParameter("labelId", hexLabelId);
            }

            // Add the label's context to the api call:
            mAccountParams.setCurrentLabelContext(nrContext);
            if (currentContext == null || currentContext.isEmpty()) {
                ctx = nrContext;
            } else {
                ctx = currentContext + ("," + nrContext);
            }

            // ctx contains the latest context
            try {
                String b64ctx = Base64.encodeToString(ctx.getBytes("UTF-8"), Base64.DEFAULT);
                if (b64ctx.endsWith("\n")) {
                    b64ctx = b64ctx.substring(0, b64ctx.length() - 1);
                }
                uri.appendQueryParameter("context", URLDecoder.decode(b64ctx, "UTF-8"));

            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
        }
//      accountParams.setLabelsMode(false); //Legacy, reason unknown.
        NRConnection.getInstance().connectionWithRequest(uri.build(), listener);
    }

    public void getSupportCenterLabels(final Function1<List<SupportLabel>, Object> callback) {

        AccountParams accountParams = instance.getAccountParams();
        Uri.Builder uri = accountParams.getUri(false);
        uri.appendEncodedPath("api/kb/labels/v1/getUserLabels");
        uri.appendQueryParameter("kb", accountParams.getKnowledgeBaseId());
        uri.appendQueryParameter("apiKey", accountParams.getApiKey());

        NRConnection.connection(uri, new OnDataResponse<List<SupportLabel>>(){

            @Override
            public void onSuccess(List<SupportLabel> response) {
                callback.invoke(response);
            }

            @Override
            public void onError(NRConnectionException error) {
                widgetListener.onInitializationFailure();
            }
        });
    }

    public void suggestionsForText(final String text, final OnSuggestionsFetchedListener onSuggestionsFetchedListener) {
        if (mCachedSuggestions != null && mCachedSuggestions.get(text) != null) {
            onSuggestionsFetchedListener.onSuggestionsFetched(mCachedSuggestions.get(text), null);
        } else if (text != null && text.length() > 0) {
//            String encodedText = null;
//            try {
//                encodedText = URLEncoder.encode(text, "UTF-8");
//            } catch (UnsupportedEncodingException e) {
//                e.printStackTrace();
//            }
            Uri.Builder uriBuilder = mAccountParams.getUri();
            uriBuilder.appendEncodedPath("api/kb/v1/autoComplete");
            uriBuilder.appendQueryParameter("text", text);
            uriBuilder.appendQueryParameter("stemming", "true");

            executeRequest(uriBuilder, new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if (responseParam != null) {
                        ArrayList<String> answers = (ArrayList) ((HashMap) responseParam).get("a");
                        searchQuery = text;
                        if (answers != null) {
                            ArrayList<Spannable> suggestions = getSpannables(answers);

                            ((HashMap) responseParam).put("a", suggestions);
                            getCachedSuggestions().put(text, new NRSuggestions((HashMap) responseParam));
                            onSuggestionsFetchedListener.onSuggestionsFetched(Nanorep.this.mCachedSuggestions.get(text), null);
                        } else {
                            onSuggestionsFetchedListener.onSuggestionsFetched(null, NRError.error("nanorep", noSuggestions, "No suggestions"));
                        }

                    } else if (error != null) {
                        NRSuggestions storedSuggestions = getCachedSuggestions().get(text);
                        if (storedSuggestions == null) {
                            onSuggestionsFetchedListener.onSuggestionsFetched(null, NRError.error("nanorep", noSuggestionsInCache, "No suggestions in cache"));
                        } else {
                            onSuggestionsFetchedListener.onSuggestionsFetched(storedSuggestions, null);
                        }
                    }
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    private ArrayList<Spannable> getSpannables(ArrayList<String> answers) {
        ArrayList<Spannable> suggestions = new ArrayList<>();
        for (String answer : answers) {
            suggestions.add(getSpannable(answer));
        }

        return suggestions;
    }

    public Spannable getSpannable(String text) {
        final SpannableStringBuilder str = new SpannableStringBuilder();
        String[] originalWords = searchQuery.split("\\s+");
        String[] pipes = text.split("\\|");
        for (int i = 0; i < pipes.length; i++) {
            String[] words = pipes[i].split(",");

            String word = words[0];
            StyleSpan styleSpan;

            if (words[words.length - 1].endsWith("*")) {
                if (words.length == 1) {
                    word = word.substring(0, word.length() - 1);
                }

                styleSpan = new StyleSpan(isWordInArray(word, originalWords) ? Typeface.NORMAL : Typeface.BOLD);
            } else {
                styleSpan = new StyleSpan(Typeface.NORMAL);
            }

            String wordSpace = word + " ";

            if (i == pipes.length - 1) { // last
                wordSpace = word;
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                str.append(wordSpace, styleSpan, 0);
            } else {
                int startIndex = str.length();
                str.append(wordSpace);
                str.setSpan(styleSpan, startIndex, str.length(), 0);
            }
        }
        return str;
    }

    private boolean isWordInArray(String needle, String[] originalWords) {
        if (originalWords == null || originalWords.length == 0) {
            return false;
        }

        for (String word : originalWords) {
            if (word.equals(needle)) {
                return true;
            }
        }
        return false;
    }

    public void likeForSearchResult(final NRSearchLikeParams likeParams, final OnLikeSentListener onLikeSentListener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/widget/v1/thumb.js");
        for (String key : likeParams.getParams().keySet()) {
            uriBuilder.appendQueryParameter(key, likeParams.getParams().get(key));
        }
        executeRequest(uriBuilder, new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    onLikeSentListener.onLikeSent(false);
                } else if (responseParam != null) {
                    onLikeSentListener.onLikeSent(true);
                }
            }

            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void fetchFAQAnswer(@NonNull final Uri articleParams, final OnFAQAnswerFetchedListener onFAQAnswerFetchedListener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();

        uriBuilder.appendEncodedPath("api/kb/v1/getArticle");

        for (String param: articleParams.getQueryParameterNames()) {
            uriBuilder.appendQueryParameter(param, articleParams.getQueryParameter(param));
        }

        uriBuilder.appendQueryParameter("kbLC", Nanorep.this.mCnf.getKbLanguageCode());

        currentAnswerID = articleParams.getQueryParameter(ArticleId);

        // if exist and updated in cache, fetch from cache,
        // else call to server

        // check if we already called fetch answer for this answer id
        ArrayList<OnFAQAnswerFetchedListener> onFAQAnswerFetchedListenerArr = Nanorep.this.getFaqRequestListenersMap().get(currentAnswerID);

        if (onFAQAnswerFetchedListenerArr == null) {// array is empty, no requests for this answer id

            onFAQAnswerFetchedListenerArr = new ArrayList<OnFAQAnswerFetchedListener>();

            onFAQAnswerFetchedListenerArr.add(onFAQAnswerFetchedListener);
            Nanorep.this.getFaqRequestListenersMap().put(currentAnswerID, onFAQAnswerFetchedListenerArr);

            final ArrayList<OnFAQAnswerFetchedListener> finalOnFAQAnswerFetchedListenerArr = onFAQAnswerFetchedListenerArr;

            executeRequest(uriBuilder, new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if (responseParam != null) {
                        ((HashMap<String, Object>) responseParam).put("id", currentAnswerID);
                    } else {
                        onFAQAnswerFetchedListener.onFAQAnswerFetched(null, NRError.error(Nanorep.class.getName(), NRErrorCodes.ARTICLE, "ArticleID doesn\\'t exists for the given account parameters"));
                        return;
                    }

                    for (OnFAQAnswerFetchedListener listener : finalOnFAQAnswerFetchedListenerArr) {
                        if (error != null) {
                            listener.onFAQAnswerFetched(null, error);
                        } else {
                            ArrayList<HashMap<String, Object>> answers = (ArrayList<HashMap<String, Object>>)((HashMap<String, Object>) responseParam).get("answers");
                            if (answers != null && answers.size() > 0) {
                                listener.onFAQAnswerFetched(new NRFAQAnswer(answers.get(0)), null);
                            }
                        }
                    }


                    Nanorep.this.getFaqRequestListenersMap().remove(currentAnswerID);
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    public void likeForFAQResult(final NRFAQLikeParams likeParams, final OnLikeSentListener onLikeSentListener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/analytics/v1/addFeedback");
        uriBuilder.appendQueryParameter("ignoreValidateCookie", "true");
        if (mAccountParams.getKnowledgeBase() != null) {
            uriBuilder.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
        }
        for (String key : likeParams.getParams().keySet()) {
            uriBuilder.appendQueryParameter(key, likeParams.getParams().get(key));
        }
        if (mSessionId != null) {
            uriBuilder.appendQueryParameter("sid", mSessionId);
        }
        NRConnection.getInstance().connectionWithRequest(uriBuilder.build(), new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    onLikeSentListener.onLikeSent(false);
                } else if (responseParam instanceof HashMap) {
                    onLikeSentListener.onLikeSent(true);
                }
            }

            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void fetchConfiguration() {
        if (mAccountParams != null) {
            final Uri.Builder uri = mAccountParams.getUri(false);
            uri.appendEncodedPath("widget/scripts/cnf.json");
            if (mAccountParams.getKnowledgeBase() != null) {
                uri.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
            }
            uri.appendQueryParameter("isFloat", "true");
            uri.appendQueryParameter("context", mAccountParams.getContext());
            uri.appendQueryParameter("referer", NRUtilities.buildReferer(getAccountParams().getReferrer()));

            NRConnection.connection(uri, new OnDataResponse<String>() {
                @Override
                public void onSuccess(String response) {
                    Object retMap = NRUtilities.jsonStringToPropertyList(response);
                    if(retMap == null || !(retMap instanceof Map)) {
                        widgetListener.onInitializationFailure();
                        return;
                    }

                    Nanorep.this.mCnf = new NRConfiguration((HashMap) retMap);
                    mAccountParams.setKnowledgeBaseId(mCnf.getKbId());
                    widgetListener.onConfigurationFetched();
                }

                @Override
                public void onError(NRConnectionException error) {
                    widgetListener.onInitializationFailure();
                }
            });
        }
    }

    public void onSuggestionsPresented(ArrayList<Spannable> previousSuggestions, String previousQueryText) {
        this.previousSuggestions = previousSuggestions;
        this.previousQueryText = previousQueryText;
    }

    public String getPreviousQueryText() {
        String temp = previousQueryText;
        previousQueryText = null;
        return temp;
    }

    public ArrayList<Spannable> getPreviousSuggestions() {
        ArrayList<Spannable> temp = previousSuggestions;
        previousSuggestions = null;
        return temp;
    }

    public void clearSuggestionsData() {
        previousQueryText = null;
        previousSuggestions = null;
    }

    public NRConfiguration getNRConfiguration() {
        if (mCnf == null)
            mCnf = new NRConfiguration();

        return mCnf;
    }

    public AccountParams getAccountParams() {
        return mAccountParams;
    }
}