/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.printer;

import org.mule.munit.Coverage;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.printer.ResultPrinter;
import org.mule.munit.plugin.maven.runtime.TargetProduct;
import org.mule.munit.remote.coverage.model.ApplicationCoverageReport;
import org.mule.munit.remote.coverage.model.MuleFlow;
import org.mule.munit.remote.coverage.model.MuleResource;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.logging.Log;

/**
 * <p>
 * Prints the Coverage Summary through the Console
 * </p>
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageSummaryPrinter implements ResultPrinter {

  private Double requiredApplicationCoverage = 1.0;
  private Double requiredResourceCoverage = -1.0;
  private Double requiredFlowCoverage = -1.0;
  private Boolean failBuild = false;

  private final Log log;

  public CoverageSummaryPrinter(Log log, Coverage coverage) {
    this.log = log;
    this.requiredApplicationCoverage = coverage.getRequiredApplicationCoverage();
    this.requiredResourceCoverage = coverage.getRequiredResourceCoverage();
    this.requiredFlowCoverage = coverage.getRequiredFlowCoverage();
    this.failBuild = coverage.getFailBuild();
  }

  @Override
  public void print(TargetProduct targetProduct, RunResult result) {
    if (!result.getApplicationCoverageReport().isPresent()) {
      log.warn("Coverage Report will not be printed. No coverage data found");
      return;
    }
    ApplicationCoverageReport report = result.getApplicationCoverageReport().get();
    log.info("===============================================================================");
    log.info("MUnit Coverage Summary");
    log.info("===============================================================================");
    log.info(" * Resources: " + report.getResources().size() + " - Flows: " + report.getApplicationFlowCount()
        + " - Processors: " + report.getApplicationMessageProcessorCount());
    if (report.getApplicationFlowCount() > 0) {
      log.info(" * Application Coverage: " + formatCoverage(report.getCoverage()) + "%");
    } else {
      log.info(" * Application Coverage: N/A");
    }
    printCoverageWarnings(report);
  }

  private String formatCoverage(Double coverage) {
    NumberFormat formatter = new DecimalFormat("#00.00");
    return formatter.format(coverage);
  }


  private void printCoverageWarnings(ApplicationCoverageReport report) {
    List<String> warnings = new ArrayList<>();

    if (requiredApplicationCoverage != -1 && report.getApplicationFlowCount() > 0) {
      if (report.getCoverage() < requiredApplicationCoverage) {
        warnings.add(
                     " * Application coverage is below defined limit. Required: " + requiredApplicationCoverage + "% - Current: "
                         + formatCoverage(report.getCoverage()) + "%");
      }
    }

    if (requiredResourceCoverage != -1) {
      for (MuleResource mr : report.getResources()) {
        if (mr.getCoverage() != -1 && mr.getCoverage() < requiredResourceCoverage) {
          warnings.add(
                       " * Resource " + mr.getName() + " coverage is below defined limit. Required: " + requiredResourceCoverage
                           + "% - Current: " + formatCoverage(mr.getCoverage()) + "%");
        }
      }
    }

    if (requiredFlowCoverage != -1) {
      for (MuleResource mr : report.getResources()) {
        for (MuleFlow mf : mr.getFlows()) {
          if (mf.getCoverage() < requiredFlowCoverage) {
            warnings.add(
                         " * Flow: " + mr.getName() + " -> " + mf.getName() + " coverage is below defined limit. Required: "
                             + requiredFlowCoverage + "% - Current: " + formatCoverage(mf.getCoverage()) + "%");
          }
        }
      }
    }

    if (!warnings.isEmpty()) {
      log.info("");
      if (!failBuild) {
        log.warn("----------------------------- WARNING --------------------------------------");
        for (String w : warnings) {
          log.warn(w);
        }
      } else {
        log.error("------------------------------ FAILURE ---------------------------------------");
        for (String w : warnings) {
          log.error(w);
        }
      }
    }
  }

}
