/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core.model;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.munit.plugins.coverage.core.model.CoverageComponentIdentifier.fromComponentIdentifier;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;

import java.io.Serializable;
import java.util.Optional;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;

/**
 * Descriptor for a component in the configuration along with it's {@link ComponentType}.
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class CoverageTypedComponentIdentifier implements TypedComponentIdentifier, Serializable {

  private static final long serialVersionUID = -6585884125494525933L;

  private CoverageComponentIdentifier identifier;
  private TypedComponentIdentifier.ComponentType type;

  public CoverageTypedComponentIdentifier(CoverageComponentIdentifier identifier, ComponentType type) {
    this.identifier = identifier;
    this.type = type;
  }

  public ComponentIdentifier getIdentifier() {
    return identifier;
  }

  public ComponentType getType() {
    return type;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    CoverageTypedComponentIdentifier that = (CoverageTypedComponentIdentifier) o;

    if (!getIdentifier().equals(that.getIdentifier())) {
      return false;
    }
    return getType() == that.getType();
  }

  @Override
  public int hashCode() {
    int result = getIdentifier().hashCode();
    result = 31 * result + getType().hashCode();
    return result;
  }

  @Override
  public String toString() {
    return "CoverageTypedComponentIdentifier{" +
        "identifier=" + identifier +
        ", type=" + type +
        '}';
  }


  public static Optional<CoverageTypedComponentIdentifier> fromTypedComponentIdentifier(Optional<TypedComponentIdentifier> identifier) {
    if (identifier.isPresent()) {
      return Optional.of(fromTypedComponentIdentifier(identifier.get()));
    }
    return Optional.empty();
  }

  public static CoverageTypedComponentIdentifier fromTypedComponentIdentifier(TypedComponentIdentifier identifier) {
    return new CoverageTypedComponentIdentifier(fromComponentIdentifier(identifier.getIdentifier()), identifier.getType());
  }

  static ComponentIdentifier parseComponentIdentifier(String componentIdentifier) {
    // This is reading the alias of the namespace in a given xml, not the actual namespace
    checkArgument(!isEmpty(componentIdentifier), "identifier cannot be an empty string or null");
    String[] values = componentIdentifier.split(":");
    String namespace;
    String identifier;
    if (values.length == 2) {
      namespace = values[0];
      identifier = values[1];
    } else {
      namespace = CORE_PREFIX;
      identifier = values[0];
    }
    return CoverageComponentIdentifier.parseComponentIdentifier(componentIdentifier);
  }
}
