/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core.model;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.munit.plugins.coverage.core.model.CoverageComponentIdentifier.parseComponentIdentifier;
import static org.mule.munit.plugins.coverage.core.model.CoverageLocationPart.fromLocationPart;
import static org.mule.munit.plugins.coverage.core.model.CoverageTypedComponentIdentifier.fromTypedComponentIdentifier;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;

import java.io.Serializable;
import java.net.URI;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Collectors;

import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.LocationPart;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;

import com.google.common.collect.ImmutableList;

/**
 * A component location describes where the component is defined in the configuration of the artifact.
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageComponentLocation implements ComponentLocation, Serializable {

  private static final long serialVersionUID = 4958158607813720623L;

  private String name;
  private LinkedList<CoverageLocationPart> parts;
  private volatile String location;

  /**
   * Creates a virtual {@link ComponentLocation} for a single element, using the core namespace and using UNKNOWN as type. Only
   * meant for situations where a real location cannot be obtained.
   *
   * @param component the name of the element
   * @return a location for it
   */
  public static CoverageComponentLocation fromSingleComponent(String component) {
    CoverageTypedComponentIdentifier identifier =
        new CoverageTypedComponentIdentifier(parseComponentIdentifier(component), UNKNOWN);

    CoverageLocationPart part = new CoverageLocationPart(component, of(identifier), empty(), empty());
    return new CoverageComponentLocation(of(component), asList(part));
  }

  /**
   * @param name the name of the global element in which the specific component is located.
   * @param parts the set of parts to locate the component.
   */
  public CoverageComponentLocation(Optional<String> name, List<CoverageLocationPart> parts) {
    this.name = name.orElse(null);
    this.parts = new LinkedList<>(parts);
  }

  /**
   * {@inheritDoc}
   */
  public Optional<String> getName() {
    return ofNullable(name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<LocationPart> getParts() {
    return unmodifiableList(parts);
  }

  @Override
  public TypedComponentIdentifier getComponentIdentifier() {
    return parts.get(parts.size() - 1).getPartIdentifier().get();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<String> getFileName() {
    return parts.getLast().getFileName();
  }

  @Override
  public List<URI> getImportChain() {
    return Collections.emptyList();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<Integer> getLineInFile() {
    return parts.getLast().getLineInFile();
  }

  /**
   * @return a string representation of the {@link CoverageComponentLocation}.
   */
  @Override
  public String getLocation() {
    if (location == null) {
      synchronized (this) {
        if (location == null) {
          StringBuilder locationBuilder = new StringBuilder();
          for (CoverageLocationPart part : parts) {
            locationBuilder.append("/").append(part.getPartPath());
          }
          location = locationBuilder.replace(0, 1, "").toString();
        }
      }
    }
    return location;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getRootContainerName() {
    return getParts().stream().findFirst().get().getPartPath();
  }

  /**
   * Creates a new instance of ComponentLocation adding the specified part.
   *
   * @param partPath the path of this part
   * @param partIdentifier the component identifier of the part if it's not a synthetic part
   * @return a new instance with the given location part appended.
   */
  public CoverageComponentLocation appendLocationPart(String partPath, Optional<TypedComponentIdentifier> partIdentifier,
                                                      Optional<String> fileName, Optional<Integer> lineInFile) {
    return new CoverageComponentLocation(ofNullable(name), ImmutableList.<CoverageLocationPart>builder().addAll(parts)
        .add(new CoverageLocationPart(partPath, fromTypedComponentIdentifier(partIdentifier), fileName, lineInFile)).build());
  }

  /**
   * Utility method that adds a processors part to the location. This is the part used for nested processors in configuration
   * components.
   *
   * @return a new instance with the processors location part appended.
   */
  public CoverageComponentLocation appendProcessorsPart() {
    return new CoverageComponentLocation(ofNullable(name), ImmutableList.<CoverageLocationPart>builder().addAll(parts)
        .add(new CoverageLocationPart("processors", empty(), empty(), empty())).build());
  }

  /**
   * Utility method that adds a router part to the location. This is the part used for nested processors in configuration
   * components.
   *
   * @return a new instance with the processors location part appended.
   */
  public CoverageComponentLocation appendRoutePart() {
    return new CoverageComponentLocation(ofNullable(name), ImmutableList.<CoverageLocationPart>builder().addAll(parts)
        .add(new CoverageLocationPart("route", empty(), empty(), empty())).build());
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    CoverageComponentLocation that = (CoverageComponentLocation) o;

    if (getName() != null ? !getName().equals(that.getName()) : that.getName() != null) {
      return false;
    }
    if (!getParts().equals(that.getParts())) {
      return false;
    }
    return getLocation() != null ? getLocation().equals(that.getLocation()) : that.getLocation() == null;
  }

  @Override
  public int hashCode() {
    int result = getName() != null ? getName().hashCode() : 0;
    result = 31 * result + getParts().hashCode();
    result = 31 * result + (getLocation() != null ? getLocation().hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "DefaultComponentLocation{" +
        "name='" + name + '\'' +
        ", parts=" + parts +
        ", location='" + getLocation() + '\'' +
        '}';
  }

  public static CoverageComponentLocation fromComponentLocation(ComponentLocation location) {
    // TODO ask mule to expose the name in ComponentLocation
    Optional<String> name = ((DefaultComponentLocation) location).getName();
    List<CoverageLocationPart> locationParts =
        location.getParts().stream().map(lp -> fromLocationPart(lp)).collect(Collectors.toList());

    return new CoverageComponentLocation(name, locationParts);
  }

  @Override
  public Optional<Integer> getStartColumn() {
    return Optional.empty();
  }

  @Override
  public OptionalInt getLine() {
    return OptionalInt.empty();
  }

  @Override
  public OptionalInt getColumn() {
    return OptionalInt.empty();
  }
}
