/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util.queue.api;

import static java.util.Objects.isNull;

import java.io.Serializable;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

import javax.inject.Inject;
import javax.inject.Named;

import org.mule.munit.runner.component.rules.TestDescription;
import org.mule.munit.runner.component.rules.TestRule;
import org.mule.munit.tools.util.queue.internal.error.QueueTimeOutException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.core.api.config.MuleProperties;
import org.mule.runtime.core.api.util.queue.DefaultQueueConfiguration;
import org.mule.runtime.core.api.util.queue.Queue;
import org.mule.runtime.core.api.util.queue.QueueManager;

/**
 * Rule to manage temporary queues
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class TemporaryQueueRule implements TestRule, Initialisable {

  public static final String DEFAULT_QUEUE_NAME = "MUNIT_TOOLS_QUEUE";

  @Inject
  @Named(MuleProperties.OBJECT_QUEUE_MANAGER)
  protected QueueManager queueManager;

  private static final AtomicReference<QueueManager> queueManagerReference = new AtomicReference<>();
  private static final AtomicReference<Map<String, Queue>> queuesReference = new AtomicReference<>();

  private final Map<String, Queue> queues = new ConcurrentHashMap<>();

  @Override
  public synchronized void apply(TestDescription testDescription) {
    clearQueues();
  }

  public void push(Serializable value, String queueName) {
    executeOnQueue(getQueueName(queueName), queue -> {
      queue.put(value);
      return null;
    });
  }

  public Serializable pop(String queueName, Long timeOut) {
    if (isNull(timeOut)) {
      return executeOnQueue(getQueueName(queueName), Queue::take);
    } else {
      return executeOnQueue(getQueueName(queueName), queue -> queue.poll(timeOut));
    }
  }

  /**
   * Returns the size of the queue associated with the give name. Method is static since it must be able to be called from a
   * expression language function
   * 
   * @param queueName name of the queue
   * @return size of the queue
   */
  public synchronized static int size(String queueName) {
    return executeOnQueue(queueManagerReference.get(), queuesReference.get(), getQueueName(queueName), Queue::size);
  }

  @Override
  public synchronized void reset() {
    clearQueues();
  }

  private void clearQueues() {
    queues.keySet().forEach(q -> executeOnQueue(q, queue -> {
      queue.clear();
      return null;
    }));
  }

  private static <T> T executeOnQueue(QueueManager queueManager, Map<String, Queue> queues, String queueName,
                                      TemporaryQueueTask<T> task) {
    try {
      Queue queue = getOrCreateQueue(queueManager, queues, queueName);
      return task.run(queue);
    } catch (InterruptedException e) {
      throw new QueueTimeOutException(e, queueName);
    }
  }

  private static Queue getOrCreateQueue(QueueManager queueManager, Map<String, Queue> queues, String queueName) {
    return queues.computeIfAbsent(queueName, name -> {
      queueManager.setQueueConfiguration(name, new DefaultQueueConfiguration());
      return queueManager.getQueueSession().getQueue(name);
    });
  }

  private <T> T executeOnQueue(String queue, TemporaryQueueTask<T> task) {
    return executeOnQueue(queueManager, queues, queue, task);
  }

  private static String getQueueName(String queueName) {
    return queueName != null
        ? queueName
        : DEFAULT_QUEUE_NAME;
  }

  @Override
  public void initialise() {
    queueManagerReference.set(queueManager);
    queuesReference.set(queues);
  }

  @FunctionalInterface
  private interface TemporaryQueueTask<T> {

    T run(Queue queue) throws InterruptedException;
  }

}
