/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.server;

import static org.mule.munit.common.util.RunnerPortProvider.MUNIT_SERVER_PORT;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.List;
import java.util.Optional;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.util.RunnerPortProvider;
import org.mule.munit.runner.component.TestComponentInfoProvider;
import org.mule.munit.runner.config.TestComponentLocator;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.config.api.LazyComponentInitializer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The goal of this class is to offer an entry point to the container through the munit-runner plugin with the objective to run
 * MUnit Test Suites
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerServer implements Runnable, Lifecycle {

  private static final int SOCKET_TIMEOUT_MILLIS = 15000;

  protected static final String MUNIT_RUNNER_SERVER_TIME_OUT = "munit.runner.server.time.out";
  protected static final String MUNIT_SCHEDULER_NAME = "munit";

  private static transient Logger log = LoggerFactory.getLogger(RunnerServer.class);

  @Inject
  private SchedulerService schedulerService;

  @Inject
  private ConfigurationComponentLocator componentLocator;

  private Scheduler scheduler;

  @Inject
  private LazyComponentInitializer lazyComponentInitializer;

  @Inject
  private List<TestComponentInfoProvider> testComponentInfoProviders;

  private int port;
  private boolean keepRunning = true;
  private ServerSocket providerSocket = null;
  private TestComponentLocator testComponentLocator;

  public RunnerServer() {}

  @Override
  public void run() {
    try {
      providerSocket = createServerSocket();

      do {
        log.info("Waiting for client connection ");
        Socket connection = providerSocket.accept();
        log.info("Client connection received from " + connection.getInetAddress().getHostName() + " - " + keepRunning);

        ObjectOutput out = new ObjectOutputStream(connection.getOutputStream());
        ObjectInput in = new ObjectInputStream(connection.getInputStream());

        handleClientMessage(in, out);

      } while (keepRunning);

    } catch (SocketTimeoutException timeoutException) {
      log.debug("MUnit server time out");
      if (keepRunning) {
        log.error("Client connection timeout after " + String.valueOf(getServerTimeout()) + " milliseconds");
      }
    } catch (IOException ioException) {
      if (providerSocket != null && providerSocket.isClosed()) {
        if (keepRunning) {
          log.warn("Kill signal received before accept timeout in port [" + port + "]", ioException);
        } else {
          log.debug("Shut down signal received MUnit server running in port [" + port + "]...");
        }
      } else {
        log.error("Failed to start MUnit server in port [" + port + "]", ioException);
      }
    } catch (ClassNotFoundException e) {
      log.error("Fail to deserialize message.", e);
    } finally {
      try {
        log.debug("Shutting down MUnit server running in port [" + port + "]...");

        if (null != providerSocket) {
          providerSocket.close();
        }
        keepRunning = false;
        log.debug("MUnit server shutdown");
      } catch (IOException ioException) {
        log.debug("MUnit server error during shutdown.");
      }
    }
  }

  protected ServerSocket createServerSocket() throws IOException {
    RunnerPortProvider runnerPortProvider = new RunnerPortProvider();
    Optional<Integer> munitServerPort = runnerPortProvider.getPredefinedPort();
    if (!munitServerPort.isPresent()) {
      log.warn("Property " + MUNIT_SERVER_PORT + " has not been defined. Probably because this run hasn't been started by MUnit");
      log.warn("Starting server on random port...");
      munitServerPort = Optional.of(runnerPortProvider.findFreePort());
    }

    port = munitServerPort.get();
    ServerSocket providerSocket = new ServerSocket(port, 10);
    providerSocket.setSoTimeout(getServerTimeout());
    log.debug("MUnit server started listening in port [" + port + "]...");
    return providerSocket;
  }

  protected void handleClientMessage(ObjectInput in, ObjectOutput out) throws IOException, ClassNotFoundException {
    RunMessageHandler commander = new RunMessageHandler(in, out, testComponentLocator);
    commander.handle();
  }

  protected boolean isKeepRunning() {
    return keepRunning;
  }

  public void setSchedulerService(SchedulerService schedulerService) {
    this.schedulerService = schedulerService;
  }

  public void setComponentLocator(ConfigurationComponentLocator componentLocator) {
    this.componentLocator = componentLocator;
  }

  public void setTestComponentInfoProviders(List<TestComponentInfoProvider> testComponentInfoProviders) {
    this.testComponentInfoProviders = testComponentInfoProviders;
  }

  @Override
  public void initialise() throws InitialisationException {
    log.debug("Initializing MUnit server...");
    testComponentLocator = new TestComponentLocator(componentLocator, lazyComponentInitializer, testComponentInfoProviders);
    scheduler = createScheduler();
  }

  @Override
  public void start() throws MuleException {
    if (System.getProperty(MUNIT_SERVER_PORT) != null) {
      scheduler.submit(this);
    }
  }

  @Override
  public void stop() throws MuleException {
    log.debug("Stop signal received, shutting down MUnit server...");
    try {
      keepRunning = false;
      if (providerSocket != null) {
        providerSocket.close();
      }
    } catch (IOException e) {
      log.warn("Error when sending kill signal to MUnit server", e);
    }
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }

    keepRunning = false;
    log.debug("Dispose signal received, shutting down MUnit server...");
  }

  private Integer getServerTimeout() {
    String timeout = System.getProperty(MUNIT_RUNNER_SERVER_TIME_OUT);

    if (StringUtils.isBlank(timeout)) {
      return SOCKET_TIMEOUT_MILLIS;
    }

    try {
      log.debug("Runner server timeout defined by property. Attempting to use that...");
      Integer to = Integer.parseInt(timeout);
      log.debug("Runner server timeout defined by property as [" + timeout + "]");
      return to;
    } catch (NumberFormatException e) {
      log.warn("Runner server timeout defined by property invalid. Using default");
      return SOCKET_TIMEOUT_MILLIS;
    }

  }

  private Scheduler createScheduler() {
    return schedulerService
        .customScheduler(SchedulerConfig.config().withName(MUNIT_SCHEDULER_NAME).withMaxConcurrentTasks(1).withWaitAllowed(true));
  }

}
