/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.component.state;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.deployment.management.ComponentInitialStateManager;

/**
 * The goal of this class is to shutdown flow sources.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitComponentInitialStateManager implements ComponentInitialStateManager {

  private static final Boolean DEFAULT_MESSAGESOURCE_STATE = false;

  protected static final String MUNIT_DISABLE_INITIAL_STATE_MANAGER = "munit.disable.initial.state.manager";
  protected static final String MUNIT_ENABLE_SOURCES_ON_START = "munit.enable.sources.on.start";

  @Inject
  protected Registry registry;

  @Override
  public boolean mustStartMessageSource(Component component) {
    if (component == null) {
      return false;
    }

    if (disableInitialStateManager()) {
      return true;
    }

    if (!getMunitModule().isPresent()) {
      return DEFAULT_MESSAGESOURCE_STATE;
    }

    if (enableSourcesOnStartActivated()) {
      return enableSourceOnStart(component);
    } else {
      return false;
    }
  }

  private boolean enableSourceOnStart(Component component) {
    if (getMunitModule().get().getEnableFlowSources().getFlows().isEmpty()) {
      return !getMunitModule().get().isDisableFlowSources();
    } else {
      return shouldStartMessageSource(component);
    }
  }

  private boolean shouldStartMessageSource(Component component) {
    Boolean shouldStart = !getMunitModule().get().isDisableFlowSources();

    List<String> flowNameList = getMunitModule().get().getEnableFlowSources().getFlows().stream()
        .map(f -> f.getFlow().getRootContainerLocation().getGlobalName())
        .collect(Collectors.toList());

    if (flowNameList.contains(component.getLocation().getRootContainerName())) {
      return !shouldStart;
    } else {
      return shouldStart;
    }
  }

  private Optional<MunitModule> getMunitModule() {
    return this.registry.lookupByType(MunitModule.class);
  }

  private Boolean disableInitialStateManager() {
    return Boolean.valueOf(System.getProperty(MUNIT_DISABLE_INITIAL_STATE_MANAGER, "false"));
  }

  private Boolean enableSourcesOnStartActivated() {
    return Boolean.valueOf(System.getProperty(MUNIT_ENABLE_SOURCES_ON_START, "false"));
  }

}
