/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.component.factory;

import static java.lang.String.format;
import static org.mule.runtime.core.privileged.processor.chain.DefaultMessageProcessorChainBuilder.newLazyProcessorChainBuilder;

import java.util.List;

import javax.inject.Inject;

import org.mule.munit.runner.flow.TestFlow;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.api.processor.strategy.ProcessingStrategy;
import org.mule.runtime.core.api.processor.strategy.ProcessingStrategyFactory;
import org.mule.runtime.core.privileged.PrivilegedMuleContext;
import org.mule.runtime.core.privileged.processor.chain.DefaultMessageProcessorChainBuilder;
import org.mule.runtime.core.privileged.processor.chain.MessageProcessorChain;
import org.mule.runtime.dsl.api.component.AbstractComponentFactory;

/**
 * {@link AbstractComponentFactory} for processor chains inside the MUnit test
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class TestProcessorChainFactory extends AbstractComponentFactory<MessageProcessorChain> {

  @Inject
  protected PrivilegedMuleContext muleContext;
  protected List processors;
  protected String name;

  public void setMessageProcessors(List processors) {
    this.processors = processors;
  }

  @Override
  public MessageProcessorChain doGetObject() throws Exception {
    DefaultMessageProcessorChainBuilder builder = getBuilderInstance();
    for (Object processor : processors) {
      if (processor instanceof Processor) {
        builder.chain((Processor) processor);
      } else {
        throw new IllegalArgumentException(format("MessageProcessorBuilder should only have MessageProcessor's or MessageProcessorBuilder's configured. Found a %s",
                                                  processor.getClass().getName()));
      }
    }
    return newLazyProcessorChainBuilder(builder, muleContext,
                                        () -> getProcessingStrategy(muleContext.getConfigurationComponentLocator(),
                                                                    getDefaultProcessingStrategyFactory(),
                                                                    getRootContainerLocation()));
  }

  protected DefaultMessageProcessorChainBuilder getBuilderInstance() {
    DefaultMessageProcessorChainBuilder builder = new DefaultMessageProcessorChainBuilder();
    builder.setName("munit processor chain '" + name + "'");
    return builder;
  }

  public void setName(String name) {
    this.name = name;
  }

  private ProcessingStrategy getProcessingStrategy(ConfigurationComponentLocator locator,
                                                   ProcessingStrategyFactory processingStrategyFactory,
                                                   Location rootContainerLocation) {
    if (processingStrategyFactory == null) {
      return null;
    }
    TestFlow testFlow = findTest(locator, rootContainerLocation);
    ProcessingStrategy processingStrategy = processingStrategyFactory.create(muleContext, name);
    testFlow.addProcessingStrategy(name, processingStrategy);
    return processingStrategy;
  }

  private static TestFlow findTest(ConfigurationComponentLocator locator, Location rootContainerLocation) {
    return locator.find(rootContainerLocation).filter(loc -> loc instanceof TestFlow)
        .map(testFlow -> ((TestFlow) testFlow))
        .orElseThrow(() -> new IllegalStateException("Chain was expected to belong to a test. Root container was: "
            + rootContainerLocation.getGlobalName()));
  }

  /**
   * TODO: Inject this: MU-1566
   */
  private ProcessingStrategyFactory getDefaultProcessingStrategyFactory() {
    return muleContext.getConfiguration().getDefaultProcessingStrategyFactory();
  }

}
