/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.functions;

import static org.mule.runtime.api.metadata.DataType.fromFunction;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CONFIGURATION_PROPERTIES;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.inject.Named;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.el.Binding;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionFunction;
import org.mule.runtime.api.el.ExpressionModule;
import org.mule.runtime.api.el.ModuleNamespace;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.privileged.el.GlobalBindingContextProvider;

/**
 * {@link GlobalBindingContextProvider} that provides helper functions under the Munit namespace
 *
 * @since 2.2.0
 * @author Mulesoft Inc.
 */
public class MunitFunctionsBindingContextProvider implements GlobalBindingContextProvider {

  @Inject
  @Named(OBJECT_CONFIGURATION_PROPERTIES)
  private ConfigurationProperties configurationProperties;

  @Override
  public BindingContext getBindingContext() {
    ModuleNamespace munitNamespace = new ModuleNamespace("Munit");
    Map<String, TypedValue> bindings = getBindings();
    ExpressionModule expressionModule = new MunitExpressionModule(bindings, munitNamespace);
    return BindingContext.builder().addModule(expressionModule).build();
  }

  private Map<String, TypedValue> getBindings() {
    Map<String, TypedValue> bindings = new HashMap<>();
    addOsFunctions(bindings);
    addMuleVersionFunctions(bindings);
    return bindings;
  }

  private void addOsFunctions(Map<String, TypedValue> bindings) {
    addFunction(bindings, "osEqualTo", new OsEqualToFunction(configurationProperties));
  }

  private void addMuleVersionFunctions(Map<String, TypedValue> bindings) {
    addFunction(bindings, "muleVersionPriorTo", new MuleVersionPriorToFunction());
    addFunction(bindings, "muleVersionNewerThan", new MuleVersionNewerThanFunction());
    addFunction(bindings, "muleVersionEqualTo", new MuleVersionEqualToFunction());
  }

  private void addFunction(Map<String, TypedValue> bindings, String name, ExpressionFunction function) {
    bindings.put(name, new TypedValue<>(function, fromFunction(function)));
  }

  /**
   * The purpose of this class exists due to a limitation when registering functions through the api: MULE-16253.
   * 
   * This should be removed once the munit-runner minMuleVersion is able to use the api classes
   */
  private static class MunitExpressionModule implements ExpressionModule {

    private Map<String, TypedValue> bindings;
    private ModuleNamespace namespace;

    MunitExpressionModule(Map<String, TypedValue> bindings, ModuleNamespace namespace) {
      this.bindings = bindings;
      this.namespace = namespace;
    }

    @Override
    public Collection<Binding> bindings() {
      return this.bindings.entrySet().stream().map((entry) -> new Binding(entry.getKey(), entry.getValue()))
          .collect(Collectors.toList());
    }

    @Override
    public Collection<String> identifiers() {
      return bindings.keySet();
    }

    @Override
    public Optional<TypedValue> lookup(String identifier) {
      return Optional.ofNullable(bindings.get(identifier));
    }

    @Override
    public ModuleNamespace namespace() {
      return namespace;
    }
  }

}
