/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container.model;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.util.Optional;

import org.mule.munit.remote.properties.Parameterization;
import org.mule.runtime.api.deployment.meta.Product;

/**
 * Represents a suite to be run
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class SuiteRun {

  private String suitePath;
  private Parameterization parameterization;
  private boolean designTime;
  private boolean ignore;
  private String minMuleVersion;
  private Product requiredProduct;

  private SuiteRun(String suitePath, Parameterization parameterization, boolean designTime, boolean ignore,
                   String minMuleVersion, Product requiredProduct) {
    this.suitePath = suitePath;
    this.parameterization = parameterization;
    this.designTime = designTime;
    this.ignore = ignore;
    this.minMuleVersion = minMuleVersion;
    this.requiredProduct = requiredProduct;
  }

  public String getSuitePath() {
    return suitePath;
  }

  public Optional<Parameterization> getParameterization() {
    return ofNullable(parameterization);
  }

  public boolean isDesignTime() {
    return designTime;
  }

  public boolean isIgnored() {
    return ignore;
  }

  public Optional<String> getMinMuleVersion() {
    return ofNullable(minMuleVersion);
  }

  public static Builder builder() {
    return new Builder();
  }

  public Product getRequiredProduct() {
    return requiredProduct;
  }

  /**
   * Builds a {@link SuiteRun}
   *
   * @author Mulesoft Inc.
   * @since 2.2.0
   */
  public static class Builder {

    private String suitePath;
    private Parameterization parameterization;
    private boolean designTime = true;
    private boolean ignore = false;
    private String minMuleVersion;
    private Product requiredProduct = Product.MULE;

    private Builder() {}

    public Builder withSuitePath(String suitePath) {
      checkArgument(isNotBlank(suitePath), "Suite path cannot be blank");
      this.suitePath = suitePath;
      return this;
    }

    public Builder withParameterization(Parameterization parameterization) {
      checkArgument(parameterization != null, "Parameterization name cannot be null");
      this.parameterization = parameterization;
      return this;
    }

    public Builder withDesignTime(boolean designTime) {
      this.designTime = designTime;
      return this;
    }

    public Builder withIgnore(boolean ignore) {
      this.ignore = ignore;
      return this;
    }

    public Builder withMinMuleVersion(String minMuleVersion) {
      this.minMuleVersion = minMuleVersion;
      return this;
    }

    public Builder withRequiredProduct(Product requiredProduct) {
      this.requiredProduct = requiredProduct;
      return this;
    }

    public SuiteRun build() {
      return new SuiteRun(suitePath, parameterization, designTime, ignore, minMuleVersion, requiredProduct);
    }
  }

  @Override
  public String toString() {
    return suitePath + getParameterization().map(param -> "[" + param.getParameterizationName() + "]").orElse(EMPTY);
  }
}
