/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.api.configuration;

import java.io.IOException;

import org.mule.tools.model.Deployment;
import org.mule.tools.model.anypoint.ArmDeployment;
import org.mule.tools.model.anypoint.CloudHubDeployment;

import com.google.gson.ExclusionStrategy;
import com.google.gson.FieldAttributes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * Define the configuration required for a deployment
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
@JsonAdapter(DeploymentConfiguration.DeploymentConfigurationAdapter.class)
public class DeploymentConfiguration {

  private Deployment deployment;

  private DeploymentType deploymentType;

  public DeploymentConfiguration(DeploymentType deploymentType, Deployment deployment) {
    this.deployment = deployment;
    this.deploymentType = deploymentType;
  }

  public Deployment getDeployment() {
    return deployment;
  }

  public DeploymentType getDeploymentType() {
    return deploymentType;
  }

  static class DeploymentConfigurationAdapter extends TypeAdapter<DeploymentConfiguration> {

    private static final String TYPE = "type";
    private static final String VALUE = "value";
    private static final Gson gson = new GsonBuilder()
        .addSerializationExclusionStrategy(new UserCredentialsExclusionStrategy())
        .create();

    @Override
    public void write(JsonWriter out, DeploymentConfiguration value) throws IOException {
      out.beginObject();
      out.name(TYPE);
      out.value(value.getDeploymentType().name());
      out.name(VALUE);

      switch (value.getDeploymentType()) {
        case CLOUDHUB:
          gson.toJson(value.getDeployment(), CloudHubDeployment.class, out);
          break;
        case ARM:
          gson.toJson(value.getDeployment(), ArmDeployment.class, out);
          break;
      }
      out.endObject();
    }

    @Override
    public DeploymentConfiguration read(JsonReader in) throws IOException {
      in.beginObject();
      if (!TYPE.equals(in.nextName())) {
        throw new IOException("Deployment configuration serialized does not have the type field");
      }
      String containerTypeValue = in.nextString();
      DeploymentType containerType = DeploymentType.fromValue(containerTypeValue);
      if (containerType == null) {
        throw new IOException("Deployment configuration type not found, value was " + containerTypeValue);
      }

      if (!VALUE.equals(in.nextName())) {
        throw new IOException("Deployment configuration serialized does not have the value field");
      }

      Deployment deployment = null;

      switch (containerType) {
        case CLOUDHUB:
          deployment = gson.fromJson(in, CloudHubDeployment.class);
          break;
        case ARM:
          deployment = gson.fromJson(in, ArmDeployment.class);
          break;
      }
      in.endObject();
      return new DeploymentConfiguration(containerType, deployment);
    }
  }

  static class UserCredentialsExclusionStrategy implements ExclusionStrategy {

    @Override
    public boolean shouldSkipField(FieldAttributes fieldAttributes) {
      if (fieldAttributes.getName().equalsIgnoreCase("username") || fieldAttributes.getName().equalsIgnoreCase("password")) {
        return true;
      }

      return false;
    }

    @Override
    public boolean shouldSkipClass(Class<?> aClass) {
      return false;
    }
  }
}
