/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.munit.remote.api.configuration;

import static com.owlike.genson.Operations.checkNotNull;
import static org.mule.runtime.api.deployment.meta.Product.MULE_EE;

import org.mule.tools.model.Deployment;

import java.util.Map;

/**
 * Defines the Container configuration for CloudHub
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class CloudHubContainerConfiguration extends ContainerConfiguration {

  private DeploymentConfiguration deploymentConfiguration;
  private boolean debugEnabled;
  private String executionId;

  protected CloudHubContainerConfiguration(String runtimeId, String product, String munitWorkingDirectoryPath,
                                           String log4JConfigurationFilePath, DeploymentConfiguration deploymentConfiguration,
                                           boolean debugEnabled, String executionId, Map<String, String> systemPropertyVariables,
                                           Map<String, String> environmentVariables) {

    super(DeploymentType.CLOUDHUB.name(), runtimeId, product, munitWorkingDirectoryPath, log4JConfigurationFilePath,
          systemPropertyVariables, environmentVariables);
    this.deploymentConfiguration = deploymentConfiguration;
    this.debugEnabled = debugEnabled;
    this.executionId = executionId;
  }

  public DeploymentConfiguration getDeploymentConfiguration() {
    return deploymentConfiguration;
  }

  public boolean isDebugEnabled() {
    return debugEnabled;
  }

  @Override
  public DeploymentType getType() {
    return DeploymentType.CLOUDHUB;
  }

  public String getExecutionId() {
    return executionId;
  }

  /**
   * Cloud Hub Container Configuration Builder
   *
   * @author Mulesoft Inc.
   * @since 2.3.0
   */
  public static class CloudHubContainerConfigurationBuilder implements ContainerConfigurationBuilder {

    private String runtimeId;
    private String product;
    private String munitWorkingDirectoryPath;
    private String log4JConfigurationFilePath;
    private DeploymentConfiguration deploymentConfiguration;
    private boolean debugEnabled;
    private String executionId;
    private Map<String, String> systemPropertyVariables;
    private Map<String, String> environmentVariables;

    public static CloudHubContainerConfigurationBuilder from(CloudHubContainerConfiguration cc) {
      CloudHubContainerConfigurationBuilder builder = new CloudHubContainerConfigurationBuilder();

      builder.withLog4JConfigurationFilePath(cc.getLog4JConfigurationFilePath());
      builder.withMunitWorkingDirectoryPath(cc.getMunitWorkingDirectoryPath());
      builder.withDeploymentConfiguration(cc.getDeploymentConfiguration().getDeployment());
      builder.withDebugEnabled(cc.isDebugEnabled());
      return builder;
    }

    public CloudHubContainerConfigurationBuilder withMunitWorkingDirectoryPath(String munitWorkingDirectoryPath) {
      this.munitWorkingDirectoryPath = munitWorkingDirectoryPath;
      return this;
    }

    public CloudHubContainerConfigurationBuilder withLog4JConfigurationFilePath(String log4JConfigurationFilePath) {
      this.log4JConfigurationFilePath = log4JConfigurationFilePath;
      return this;
    }

    public CloudHubContainerConfigurationBuilder withDebugEnabled(boolean debugEnabled) {
      this.debugEnabled = debugEnabled;
      return this;
    }

    public CloudHubContainerConfigurationBuilder withDeploymentConfiguration(Deployment deployment) {
      checkNotNull(deployment, "must not be null");

      this.runtimeId =
          deployment.getMuleVersion().orElseThrow(() -> new IllegalArgumentException("Deployment has no mule version"));
      this.product = MULE_EE.name();
      this.deploymentConfiguration = new DeploymentConfiguration(DeploymentType.fromDeployment(deployment), deployment);
      return this;
    }

    public CloudHubContainerConfigurationBuilder withExecutionId(String executionId) {
      this.executionId = executionId;
      return this;
    }

    public CloudHubContainerConfigurationBuilder withSystemPropertyVariables(Map<String, String> systemPropertyVariables) {
      this.systemPropertyVariables = systemPropertyVariables;
      return this;
    }

    public CloudHubContainerConfigurationBuilder withEnvironmentVariables(Map<String, String> environmentVariables) {
      this.environmentVariables = environmentVariables;
      return this;
    }

    public CloudHubContainerConfiguration build() {
      return new CloudHubContainerConfiguration(runtimeId, product, munitWorkingDirectoryPath, log4JConfigurationFilePath,
                                                deploymentConfiguration, debugEnabled, executionId, systemPropertyVariables,
                                                environmentVariables);
    }

  }

}
