/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;

import static org.mule.munit.remote.RemoteRunner.MUNIT_DEPLOYMENT_PASSWORD;
import static org.mule.munit.remote.RemoteRunner.MUNIT_DEPLOYMENT_USERNAME;
import org.mule.munit.remote.api.configuration.CloudHubContainerConfiguration;
import org.mule.munit.remote.api.configuration.DeploymentType;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.container.ContainerSuiteRunDispatcher;
import org.mule.munit.remote.container.EmbeddedContainerFactory;
import org.mule.munit.remote.container.SuiteRunDispatcher;
import org.mule.munit.remote.container.SuiteRunLocator;
import org.mule.munit.remote.container.cloudhub.CloudHubRunDispatcher;
import org.mule.munit.remote.container.model.SuiteRun;
import org.mule.munit.remote.logging.MunitDeployerLog;
import org.mule.munit.remote.tools.client.BAT.BATClient;
import org.mule.tools.client.authentication.AuthenticationServiceClient;
import org.mule.tools.client.authentication.model.Credentials;
import org.mule.tools.client.cloudhub.CloudHubClient;
import org.mule.tools.model.anypoint.CloudHubDeployment;

import com.google.common.base.Strings;

import java.io.File;
import java.util.Set;

/**
 * Creates a {@Link SuiteRunDispatcher} based on {@Link DeploymentType}
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class SuiteRunDispatcherFactory {

  private final RunConfiguration runConfiguration;
  private final EmbeddedContainerFactory embeddedContainerFactory;

  public SuiteRunDispatcherFactory(RunConfiguration runConfiguration, EmbeddedContainerFactory embeddedContainerFactory) {
    this.runConfiguration = runConfiguration;
    this.embeddedContainerFactory = embeddedContainerFactory;
  }

  public SuiteRunDispatcher create() {
    switch (getDeploymentType()) {
      case CLOUDHUB:

        String username = System.getProperty(MUNIT_DEPLOYMENT_USERNAME);
        String password = System.getProperty(MUNIT_DEPLOYMENT_PASSWORD);

        if (Strings.isNullOrEmpty(username) || Strings.isNullOrEmpty(password)) {
          throw new RuntimeException("The username or password must not be null nor empty");
        }

        CloudHubContainerConfiguration containerConfiguration =
            (CloudHubContainerConfiguration) runConfiguration.getContainerConfiguration();

        CloudHubDeployment cloudHubDeployment =
            (CloudHubDeployment) containerConfiguration.getDeploymentConfiguration().getDeployment();

        cloudHubDeployment.setUsername(username);
        cloudHubDeployment.setPassword(password);

        CloudHubClient cloudHubClient =
            new CloudHubClient(cloudHubDeployment, new MunitDeployerLog(containerConfiguration.isDebugEnabled()));

        BATClient batClient =
            new BATClient(cloudHubDeployment.getUri(), new Credentials(username, password),
                          new AuthenticationServiceClient(cloudHubDeployment.getUri(), true));

        return new CloudHubRunDispatcher(runConfiguration, cloudHubClient, batClient);
      case EMBEDDED:
      default:
        File appDir = new File(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(),
                               runConfiguration.getProjectName());
        Set<SuiteRun> suiteRuns = new SuiteRunLocator(runConfiguration, appDir).locate();
        return new ContainerSuiteRunDispatcher(embeddedContainerFactory.createContainer(runConfiguration), runConfiguration,
                                               suiteRuns, appDir);
    }
  }

  private DeploymentType getDeploymentType() {
    return runConfiguration.getContainerConfiguration().getType();
  }
}
