/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.nio.charset.StandardCharsets.UTF_8;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.util.Set;

import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.util.FileUtils;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel.MuleApplicationModelBuilder;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;

/**
 * This class handles a mule-artifact.json file in an specific path.
 *
 * It stores its original state on creation and allow to update specific parts of it always overwriting the file in the path
 * provided
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MuleArtifactHandler {

  public static final String MULE_ARTIFACT_JSON = "mule-artifact.json";

  private Path muleArtifactJsonPath;
  private MuleApplicationModelBuilder originalMuleApplicationModelBuilder;

  public MuleArtifactHandler(Path muleArtifactJsonPath) {
    checkNotNull(muleArtifactJsonPath, "The mule artifact json path must not be null");
    checkArgument(muleArtifactJsonPath.toFile().exists(),
                  " The file " + muleArtifactJsonPath.toAbsolutePath().toString() + " does not exits");

    this.muleArtifactJsonPath = muleArtifactJsonPath;
    this.originalMuleApplicationModelBuilder = getApplicationModelBuilder(muleArtifactJsonPath.toFile());
  }

  /**
   * Restore the original content of the mule-artifact.json file
   * 
   * @throws IOException
   */
  public void restoreOriginal() throws IOException {
    updateMuleArtifactFile(originalMuleApplicationModelBuilder);
  }

  /**
   * Restore the original content of the mule-artifact.json file, forcing the value of redeployment. After the file is restored
   * the handler maintains the original state of the file used to create the object.
   * 
   * @param redeployment value to be set in the redeployment field
   * @throws IOException
   */
  public void restoreOriginal(Boolean redeployment) throws IOException {
    Boolean originalRedeploymentEnable = originalMuleApplicationModelBuilder.build().isRedeploymentEnabled();

    originalMuleApplicationModelBuilder.setRedeploymentEnabled(redeployment);
    updateMuleArtifactFile(originalMuleApplicationModelBuilder);

    originalMuleApplicationModelBuilder.setRedeploymentEnabled(originalRedeploymentEnable);
  }

  /**
   * It updates the configs and redeploymentEnabled fields in the mule-artifact.json file
   *
   * @param configPaths list of config path to be set in the file's field
   * @param redeployment value to be set in the redeployment field
   */
  public void updateConfigPathAndReDeployment(Set<String> configPaths, Boolean redeployment) {
    MuleApplicationModelBuilder builder = getApplicationModelBuilder(muleArtifactJsonPath.toFile());
    builder.setConfigs(configPaths);
    builder.setRedeploymentEnabled(redeployment);

    updateMuleArtifactFile(builder);
  }

  /**
   * It updates the configs field in the mule-artifact.json file
   * 
   * @param configPaths list of config path to be set in the file's field
   */
  public void updateConfigPath(Set<String> configPaths) {
    MuleApplicationModelBuilder builder = getApplicationModelBuilder(muleArtifactJsonPath.toFile());
    builder.setConfigs(configPaths);

    updateMuleArtifactFile(builder);
  }

  private void updateMuleArtifactFile(MuleApplicationModelBuilder builder) {
    String applicationDescriptorContent = new MuleApplicationModelJsonSerializer().serialize(builder.build());
    try (Writer writer = new OutputStreamWriter(new FileOutputStream(muleArtifactJsonPath.toFile()), UTF_8)) {
      writer.write(applicationDescriptorContent);
    } catch (IOException e) {
      throw new MunitError("Fail to update: " + muleArtifactJsonPath.toAbsolutePath().toString(), e);
    }
  }

  private MuleApplicationModelBuilder getApplicationModelBuilder(File applicationJsonFile) {
    MuleApplicationModel muleApplicationModel = getMuleApplicationModel(applicationJsonFile);

    MuleApplicationModelBuilder builder = new MuleApplicationModelBuilder();
    builder.setMinMuleVersion(muleApplicationModel.getMinMuleVersion());
    builder.setRedeploymentEnabled(muleApplicationModel.isRedeploymentEnabled());
    builder.setName(muleApplicationModel.getName());
    builder.setConfigs(muleApplicationModel.getConfigs());
    builder.setRequiredProduct(muleApplicationModel.getRequiredProduct());

    builder.withBundleDescriptorLoader(muleApplicationModel.getBundleDescriptorLoader());
    builder.withClassLoaderModelDescriptorLoader(muleApplicationModel.getClassLoaderModelLoaderDescriptor());

    return builder;
  }

  /**
   * It loads a {@link MuleApplicationModel} from a file.
   *
   * @param applicationJsonFile the mule-artifact.json file
   * @return the loaded {@link MuleApplicationModel}
   * @throws {@link MunitError} if fails to read the file provided
   */
  public static MuleApplicationModel getMuleApplicationModel(File applicationJsonFile) {
    String fileContent;
    try {
      fileContent = FileUtils.readFileToString(applicationJsonFile, Charset.defaultCharset());
    } catch (IOException e) {
      throw new MunitError("Fail to read: " + applicationJsonFile.toPath().toAbsolutePath().toString(), e);
    }
    return new MuleApplicationModelJsonSerializer().deserialize(fileContent);
  }

}
