/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.tool.spy;


import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;
import org.mule.munit.mock.behavior.SpyBehavior;
import org.mule.munit.mock.tool.AbstractMockingTool;
import org.mule.runtime.api.artifact.Registry;

/**
 * This class is a Munit Tool to create processor spiers
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Spy extends AbstractMockingTool {

  public Spy(Registry registry) {
    super(registry);
  }

  /**
   * Defines the name of the processor to spy
   *
   * @param name The name of the processor to spy
   * @return Itself
   */
  public Spy spyProcessor(String name) {
    this.processorName = name;
    return this;
  }

  /**
   * Defines the namespace of the message processor to spy
   *
   * @param namespace The namespace of the message processor to spy
   * @return Itself
   */
  public Spy ofNamespace(String namespace) {
    this.processorNamespace = namespace;
    return this;
  }

  /**
   * Defines the attributes of the processor to spy
   *
   * @param attributes a map of attributes name,value
   * @return Itself
   */
  public Spy withAttributes(Map<String, Object> attributes) {
    if (attributes != null) {
      this.processorAttributes = attributes;
    }
    return this;
  }

  /**
   * The {@link SpyProcess} to run before the processor
   *
   * @param withSpies Processes to run before the processor call
   */
  public Spy before(final List<SpyProcess> withSpies) {
    if (withSpies != null && !withSpies.isEmpty()) {
      ProcessorCall processorCall = createProcessorCall();
      getManager().addBeforeCallSpyBehavior(createSpyBehavior(processorCall, withSpies));
    }

    return this;
  }

  private ProcessorCall createProcessorCall() {
    ProcessorCall processorCall = new ProcessorCall(new ProcessorId(processorName, processorNamespace));
    processorCall.setAttributes(processorAttributes);
    return processorCall;
  }

  /**
   * The {@link SpyProcess}es to run before the processor
   *
   * @param withSpy Processes to run before the processor call
   */
  public Spy before(final SpyProcess... withSpy) {
    return before(Arrays.asList(withSpy));
  }

  /**
   * The {@link SpyProcess} to run after the processor
   *
   * @param withSpies Processes to run after the processor call
   */
  public Spy after(final List<SpyProcess> withSpies) {
    if (withSpies != null && !withSpies.isEmpty()) {
      ProcessorCall processorCall = createProcessorCall();
      getManager().addAfterCallSpyBehavior(createSpyBehavior(processorCall, withSpies));
    }
    return this;
  }

  /**
   * The {@link SpyProcess}es to run after the processor
   *
   * @param withSpy Processes to run after the processor call
   */
  public Spy after(final SpyProcess... withSpy) {
    return after(Arrays.asList(withSpy));
  }

  protected SpyBehavior createSpyBehavior(ProcessorCall call, List<SpyProcess> spyProcesses) {
    return new SpyBehavior(call, spyProcesses);
  }

}
