/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.behavior.BehaviorManager;
import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.model.Event;
import org.mule.munit.common.model.EventError;
import org.mule.munit.mock.behavior.Behavior;
import org.mule.munit.mock.behavior.CallBehaviour;
import org.mule.munit.mock.behavior.DefaultBehaviorManager;
import org.mule.munit.mock.behavior.SpyBehavior;
import org.mule.munit.mock.tool.spy.SpyExecutionException;
import org.mule.munit.mock.tool.spy.SpyProcess;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.interception.InterceptionAction;
import org.mule.runtime.api.interception.InterceptionEvent;
import org.mule.runtime.api.interception.ProcessorInterceptor;
import org.mule.runtime.api.interception.ProcessorParameterValue;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.interception.InternalInterceptionEvent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * It's the actual MUnit Interceptor called before, during and after the execution of each processor.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitProcessorInterceptor implements ProcessorInterceptor {

  protected static final String ENABLE_INCONSISTENT_MOCKING_PROPERTY = "enable-inconsistent-mocking";

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private DefaultBehaviorManager manager;

  private ErrorTypeRepository errorTypeRepository;

  private Map<String, ProcessorCall> processorCalls = new ConcurrentHashMap<>();

  void setManager(BehaviorManager manager) {
    this.manager = (DefaultBehaviorManager) manager;
  }

  protected DefaultBehaviorManager getManager() {
    if (manager == null) {
      throw new IllegalStateException("There is no manager defined");
    }

    return manager;
  }

  void setErrorTypeRepository(ErrorTypeRepository errorTypeRepository) {
    this.errorTypeRepository = errorTypeRepository;
  }

  @Override
  public void before(ComponentLocation location, Map<String, ProcessorParameterValue> parameters, InterceptionEvent event) {
    logger.debug("Retrieving Spy Before behavior for processor: " + locationToStringLog(location));

    Map<String, Object> resolvedParameters = resolveParameters(parameters);

    ProcessorCall processorCall = buildCall(getIdentifier(location), resolvedParameters);
    processorCalls.put(event.getContext().getId(), processorCall);
    getManager().addCall(processorCall);

    Optional<SpyBehavior> betterMatchingBehavior = getManager().getBetterMatchingBeforeSpyBehavior(processorCall);
    betterMatchingBehavior.ifPresent(spyBehavior -> runSpy(spyBehavior, event));
  }

  @Override
  public CompletableFuture<InterceptionEvent> around(ComponentLocation location,
                                                     Map<String, ProcessorParameterValue> parameters,
                                                     InterceptionEvent event, InterceptionAction action) {
    logger.debug("Retrieving mocked behavior for processor: " + locationToStringLog(location));

    Map<String, Object> resolvedParameters = resolveParameters(parameters);
    ProcessorCall processorCall = buildCall(getIdentifier(location), resolvedParameters);

    Optional<Behavior> betterMatchingBehavior = getManager().getBetterMatchingBehavior(processorCall);
    if (betterMatchingBehavior.isPresent()) {
      Behavior behavior = betterMatchingBehavior.get();
      if (behavior instanceof CallBehaviour) {
        InternalInterceptionEvent interceptionEvent = (InternalInterceptionEvent) event;
        behavior = ((CallBehaviour) behavior).copyWithoutState();
        ((CallBehaviour) behavior).evaluate(interceptionEvent.resolve());
      }

      BehaviourValidator validator = new BehaviourValidator(location);

      haltIfMockingNotAllowed(validator);
      haltIfInvalidBehavior(validator, behavior);

      if (shouldFailProcessor(behavior)) {
        logger.debug("Mock behavior found. Throwing exception instead of " + locationToStringLog(location));

        return failProcessor(event, action, behavior);
      }

      logger.debug("Mock behavior found. Executing that instead of " + locationToStringLog(location));
      action.skip();

      return returnBehavior(event, behavior);
    }

    return action.proceed();
  }

  @Override
  public void after(ComponentLocation location, InterceptionEvent event, Optional<Throwable> thrown) {
    logger.debug("Retrieving Spy After behavior for processor: " + locationToStringLog(location));

    String eventContextId = event.getContext().getId();
    if (thrown.isPresent()) {
      logger.debug("Spy after won't run." + " The processor " + locationToStringLog(location) + " failed: "
          + thrown.get().getMessage());

    } else if (!processorCalls.containsKey(eventContextId)) {
      throw new MunitError("Spy after won't run." + " The processor " + locationToStringLog(location)
          + " was not executed before");

    } else {
      Optional<SpyBehavior> betterMatchingBehavior =
          getManager().getBetterMatchingAfterSpyBehavior(processorCalls.get(eventContextId));
      betterMatchingBehavior.ifPresent(spyBehavior -> runSpy(spyBehavior, event));
    }
  }

  private ProcessorCall buildCall(ComponentIdentifier componentIdentifier, Map<String, Object> parameters) {
    ProcessorId id = new ProcessorId(componentIdentifier.getName(), componentIdentifier.getNamespace());
    ProcessorCall call = new ProcessorCall(id);
    call.setAttributes(parameters);

    return call;
  }

  private boolean shouldFailProcessor(Behavior behavior) {
    Optional<Event> behaviorEvent = behavior.getEvent();

    EventError behaviorError = behaviorEvent.map(Event::getError).orElse(null);

    boolean isThereAnEvent = behaviorEvent.isPresent();
    boolean isThereAnErrorCause = behaviorError != null && behaviorError.getCause() != null;
    boolean isThereAnErrorTypeId = behaviorError != null && StringUtils.isNotBlank(behaviorError.getTypeId());

    return isThereAnEvent && (isThereAnErrorCause || isThereAnErrorTypeId);
  }

  private CompletableFuture<InterceptionEvent> returnBehavior(InterceptionEvent event, Behavior behavior) {
    Optional<Event> mockedEvent = behavior.getEvent();
    if (behavior instanceof CallBehaviour) {
      ((CallBehaviour) behavior).clearEvent();
    }
    return CompletableFuture.completedFuture(buildInterceptingEvent(event, mockedEvent));
  }

  private CompletableFuture<InterceptionEvent> failProcessor(InterceptionEvent event, InterceptionAction action,
                                                             Behavior behavior) {
    Optional<Event> optionalEvent = behavior.getEvent();
    if (behavior instanceof CallBehaviour) {
      ((CallBehaviour) behavior).clearEvent();
    }

    buildInterceptingEvent(event, optionalEvent);
    Event behaviorEvent = optionalEvent.get();
    if (behaviorEvent.getError().getCause() != null) {
      return action.fail((Throwable) behaviorEvent.getError().getCause());

    } else {
      ComponentIdentifier componentIdentifier =
          ComponentIdentifier.buildFromStringRepresentation(behaviorEvent.getError().getTypeId());
      Optional<ErrorType> errorType = errorTypeRepository.getErrorType(componentIdentifier);

      return action.fail(errorType.get());
    }
  }

  private InterceptionEvent buildInterceptingEvent(InterceptionEvent originalEvent, Optional<Event> mockedEvent) {
    return mockedEvent.map(event -> new InterceptingEventBuilder().build(originalEvent, event)).orElse(originalEvent);
  }

  private String locationToStringLog(ComponentLocation location) {
    return getIdentifier(location)
        + " in "
        + location.getFileName().orElse(" ? ")
        + "[line: " + location.getLineInFile().orElse(-1) + "].";

  }

  private ComponentIdentifier getIdentifier(ComponentLocation location) {
    return location.getComponentIdentifier().getIdentifier();
  }

  private void haltIfMockingNotAllowed(BehaviourValidator validator) {
    if (enableInconsistentMocking()) {
      return;
    }

    if (!validator.allowMocking()) {
      String errorMessage = validator.getBaseErrorMessage() + ". Mocking of such elements is not allowed.";
      logger.error(errorMessage);

      throw new MunitError(errorMessage);
    }
  }

  private void haltIfInvalidBehavior(BehaviourValidator validator, Behavior behavior) {
    if (enableInconsistentMocking()) {
      return;
    }

    if (!validator.isBehaviorValid(behavior)) {
      String errorMessage = validator.getBaseErrorMessage() + ". The behavior contains invalid elements for the component.";
      logger.error(errorMessage);

      throw new MunitError(errorMessage);
    }
  }

  private void runSpy(SpyBehavior spyBehavior, InterceptionEvent event) {
    logger.debug("Spy behavior found. Running spy");
    CoreEvent coreEvent = ((InternalInterceptionEvent) event).resolve();
    for (SpyProcess spyProcess : spyBehavior.getSpyProcesses()) {
      spyProcess.spy(coreEvent).getThrowable().ifPresent(error -> {
        throw new SpyExecutionException("An error occurred during the execution of the spy", error);
      });
    }
  }

  private Map<String, Object> resolveParameters(Map<String, ProcessorParameterValue> parameters) {
    Map<String, Object> resolvedParameters = new HashMap<>();
    parameters.keySet().forEach(key -> resolvedParameters.put(key, new LazyValue<>(() -> parameters.get(key).resolveValue())));

    return resolvedParameters;
  }

  private Boolean enableInconsistentMocking() {
    return Boolean.valueOf(System.getProperty(ENABLE_INCONSISTENT_MOCKING_PROPERTY));
  }

  @Override
  public boolean isErrorMappingRequired(ComponentLocation location) {
    if (location.getComponentIdentifier().getIdentifier().getNamespace().equals("http")
        && location.getComponentIdentifier().getIdentifier().getName().equals("request")) {
      return true;
    } else {
      return false;
    }

  }
}
