/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static org.mule.munit.common.util.Preconditions.checkNotNull;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.maven.plugin.logging.Log;
import org.mule.munit.plugin.maven.runner.printer.MavenResultPrinter;
import org.mule.munit.plugin.maven.runner.printer.ResultPrinter;
import org.mule.munit.plugin.maven.runner.printer.SonarResultFilePrinter;
import org.mule.munit.plugin.maven.runner.printer.TestOutputFilePrinter;
import org.mule.munit.plugin.maven.runner.printer.TextResultFilePrinter;
import org.mule.munit.plugin.maven.runner.printer.XmlResultPrinter;

/**
 * <p>
 * Returns the result printers based on the user configuration
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class DefaultResultPrinterFactory implements ResultPrinterFactory {

  protected Log log;

  private File testOutputDirectory;
  private File surefireReportsFolder;
  private File sonarReportsFolder;
  private Map<String, String> systemProperties;

  private boolean enableSurefireReports;
  private boolean enableSonarReports;
  private boolean redirectTestOutputToFile;

  public DefaultResultPrinterFactory(Log log) {
    this.log = log;
  }

  public DefaultResultPrinterFactory withSurefireReports(Boolean enableSurefireReports, File surefireReportsFolder,
                                                         Map<String, String> systemProperties) {
    this.surefireReportsFolder = surefireReportsFolder;
    this.enableSurefireReports = enableSurefireReports;
    this.systemProperties = systemProperties;
    return this;
  }

  public DefaultResultPrinterFactory withSonarReports(Boolean enableSonarReports, File sonarReportsFolder) {
    this.sonarReportsFolder = sonarReportsFolder;
    this.enableSonarReports = enableSonarReports;
    return this;
  }

  public DefaultResultPrinterFactory withTestOutputReports(Boolean redirectTestOutputToFile, File testOutputDirectory) {
    checkNotNull(redirectTestOutputToFile, "Redirect Output to File property cannot be null");
    checkNotNull(testOutputDirectory, "Test Output Directory cannot be null");
    this.redirectTestOutputToFile = redirectTestOutputToFile;
    this.testOutputDirectory = testOutputDirectory;
    return this;
  }

  @Override
  public List<ResultPrinter> create() {
    List<ResultPrinter> printers = new ArrayList<>();

    printers.add(new MavenResultPrinter(log));

    if (enableSurefireReports) {
      printers.addAll(getSurefireResultPrinters(redirectTestOutputToFile));
    }

    if (redirectTestOutputToFile) {
      printers.add(new TestOutputFilePrinter(testOutputDirectory, log));
    }

    if (enableSonarReports) {
      printers.addAll(getSonarResultPrinters());
    }

    return printers;
  }

  private List<ResultPrinter> getSurefireResultPrinters(Boolean redirectTestOutputToFile) {
    List<ResultPrinter> printers = new ArrayList<>();

    printers.add(new XmlResultPrinter(surefireReportsFolder, systemProperties, log));
    printers.add(new TextResultFilePrinter(surefireReportsFolder, log));
    if (redirectTestOutputToFile) {
      printers.add(new TestOutputFilePrinter(surefireReportsFolder, log));
    }

    return printers;
  }

  private List<ResultPrinter> getSonarResultPrinters() {
    List<ResultPrinter> printers = new ArrayList<>();

    printers.add(new SonarResultFilePrinter(sonarReportsFolder, log));

    return printers;
  }

}
