/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runtime;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;

import static org.apache.commons.lang3.StringUtils.split;

import java.util.List;
import java.util.Optional;
import java.util.Properties;

/**
 * Model of the {@link ProductConfiguration} discoveryProducts parameter. It holds information about the products discovery
 *
 * @author Mulesoft Inc.
 * @since 2.4.0
 */
public class DiscoverProduct {

  public static final String DISCOVER_PRODUCT = "discoverProducts.product";
  public static final String DISCOVER_PRODUCTS_INCLUDE_SNAPSHOTS = "discoverProducts.includeSnapshots";
  public static final String DISCOVER_PRODUCTS_MIN_VERSION = "discoverProducts.minMuleVersion";
  public static final String DISCOVER_PRODUCTS_SKIP = "discoverProducts.skip";
  public static final String DISCOVER_PRODUTCS_REPOSITORIES = "discoverProducts.runtimeRepositories";
  public static final String DISCOVER_PRODUCTS_LATEST_PATCHES = "discoverProducts.latestPatches";
  public static final String DISCOVER_PRODUCTS_REPOSITORIES_SEPARATOR = ",";

  private Product id;
  private String minVersion;
  private boolean includeSnapshots = false;
  private boolean skip = false;
  private boolean latestPatches = true;
  private List<String> repositories = emptyList();

  public DiscoverProduct() {}

  private DiscoverProduct(Product id, String minVersion, boolean includeSnapshots, boolean skip,
                          List<String> repositories, boolean useLatestPatches) {
    this.id = id;
    this.minVersion = minVersion;
    this.includeSnapshots = includeSnapshots;
    this.skip = skip;
    this.repositories = repositories;
    this.latestPatches = useLatestPatches;
  }

  public Optional<String> getMinVersion() {
    return ofNullable(minVersion);
  }

  public Optional<Product> getProductId() {
    return ofNullable(id);
  }

  public boolean isIncludeSnapshots() {
    return includeSnapshots;
  }

  public boolean isSkipped() {
    return skip;
  }

  public boolean isUseLatestPatches() {
    return latestPatches;
  }

  public List<String> getRepositories() {
    return repositories;
  }

  public static Builder builder() {
    return new Builder();
  }

  /**
   * Builder for a {@link DiscoverProduct}
   *
   * @author Mulesoft Inc.
   * @since 2.4.0
   */
  public static class Builder {

    private Product productId;
    private String minVersion;
    private boolean includeSnapshots = false;
    private boolean skip = false;
    private boolean useLatestPatches = true;
    private List<String> runtimeRepositories = emptyList();

    public Builder from(DiscoverProduct discoverProduct) {
      discoverProduct.getMinVersion().ifPresent(minVersion -> this.minVersion = minVersion);
      discoverProduct.getProductId().ifPresent(productId -> this.productId = productId);
      this.includeSnapshots = discoverProduct.isIncludeSnapshots();
      this.skip = discoverProduct.isSkipped();
      this.useLatestPatches = discoverProduct.isUseLatestPatches();
      this.runtimeRepositories = discoverProduct.getRepositories();

      return this;
    }

    public Builder withProductId(Product productId) {
      this.productId = productId;
      return this;
    }

    public Builder withMinVersion(String minVersion) {
      this.minVersion = minVersion;
      return this;
    }

    public Builder includingSnapshots(Boolean includeSnapshots) {
      this.includeSnapshots = includeSnapshots;
      return this;
    }

    public Builder skip(Boolean skip) {
      this.skip = skip;
      return this;
    }

    public Builder useLatestPatches(Boolean useLatestPatches) {
      this.useLatestPatches = useLatestPatches;
      return this;
    }

    public Builder withRuntimeRepositories(List<String> runtimeRepositories) {
      this.runtimeRepositories = runtimeRepositories;
      return this;
    }

    public Builder withOverrides(Properties userProperties) {
      if (userProperties.containsKey(DISCOVER_PRODUCTS_INCLUDE_SNAPSHOTS)) {
        this.includeSnapshots = Boolean.valueOf(userProperties.getProperty(DISCOVER_PRODUCTS_INCLUDE_SNAPSHOTS));
      }
      if (userProperties.containsKey(DISCOVER_PRODUCT)) {
        this.productId = Product.valueOf(userProperties.getProperty(DISCOVER_PRODUCT));
      }
      if (userProperties.containsKey(DISCOVER_PRODUCTS_MIN_VERSION)) {
        this.minVersion = userProperties.getProperty(DISCOVER_PRODUCTS_MIN_VERSION);
      }
      if (userProperties.containsKey(DISCOVER_PRODUCTS_SKIP)) {
        this.skip = Boolean.valueOf(userProperties.getProperty(DISCOVER_PRODUCTS_SKIP));
      }
      if (userProperties.containsKey(DISCOVER_PRODUTCS_REPOSITORIES)) {
        this.runtimeRepositories = asList(split(userProperties.getProperty(DISCOVER_PRODUTCS_REPOSITORIES),
                                                DISCOVER_PRODUCTS_REPOSITORIES_SEPARATOR));
      }
      if (userProperties.containsKey(DISCOVER_PRODUCTS_LATEST_PATCHES)) {
        this.useLatestPatches = Boolean.valueOf(userProperties.getProperty(DISCOVER_PRODUCTS_LATEST_PATCHES));
      }
      return this;
    }

    public DiscoverProduct build() {
      return new DiscoverProduct(productId, minVersion, includeSnapshots, skip, runtimeRepositories, useLatestPatches);
    }

  }

}
