/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.handler;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.model.SuiteResult;

/**
 * <p>
 * RunnerMessageHandler that stores the standard output of the JVM in a map for each suite
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class TestOutputMessageHandler implements RunnerMessageHandler {

  private Map<String, List<String>> suiteOutputs = new HashMap<String, List<String>>();
  private List<String> suiteOutput = new ArrayList<String>();
  private String currentSuite;
  private Log log;

  public TestOutputMessageHandler(Log log) {
    this.log = log;
  }

  @Override
  public void handleSuiteFinished(SuiteResult suiteResult) {
    suiteOutputs.put(suiteResult.getSuitePath(), new ArrayList<>(suiteOutput));
    printTestSuiteFinished(suiteResult);
    currentSuite = null;
  }

  @Override
  public void handleSuiteStartFailure(String stackTrace) {
    printContainerFailedToStart(stackTrace);
    currentSuite = null;
  }

  @Override
  public void handleSuiteStartFailure(String suite, String parameterization, String stackTrace) {
    printContainerFailedToStart(suite, parameterization, stackTrace);
  }

  @Override
  public void handleSuiteStart(String suitePath, String parameterization) {
    printTestSuiteStart(suitePath, parameterization);
    suiteOutput.clear();
    currentSuite = suitePath;
  }

  @Override
  public void handleStandardOutputLine(String message) {
    if (shouldPrintDebugLine()) {
      System.out.println(message);
    }
    if (isNotBlank(currentSuite)) {
      suiteOutput.add(message);
    }
  }

  @Override
  public void handleRunResult(RunResult runResult) {
    runResult.setSuiteOutputs(suiteOutputs);
  }

  @Override
  public void handleSuiteFailure(SuiteResult suiteResult) {
    handleSuiteErrorOrFailure(suiteResult, "FAILURE");
  }

  @Override
  public void handleSuiteError(SuiteResult suiteResult) {
    handleSuiteErrorOrFailure(suiteResult, "ERROR");
  }

  @Override
  public void handleUnexpectedError(String stackTrace) {
    System.out.println("An unexpected error occurred");
    System.out.println(stackTrace);
  }

  protected void handleSuiteErrorOrFailure(SuiteResult suiteResult, String tag) {
    printTestSuiteFailedOrError(FilenameUtils.getName(suiteResult.getSuitePath()), suiteResult.getCause(), tag);
    suiteOutput.add(suiteResult.getCause());
    suiteOutputs.put(suiteResult.getSuitePath(), new ArrayList<>(suiteOutput));
    currentSuite = null;
  }

  private boolean shouldPrintDebugLine() {
    return currentSuite == null && log.isDebugEnabled();
  }

  private void printTestSuiteStart(String suitePath, String parameterization) {
    String parameterizationString = isNotBlank(parameterization) ? "[" + parameterization + "]" : StringUtils.EMPTY;
    System.out.format("Running %s%s\n", suitePath, parameterizationString);
  }

  private void printTestSuiteFinished(SuiteResult suiteResult) {
    System.out.format("Tests run: %d, Failures: %d, Errors: %d, Skipped: %d, Time elapsed: %.3f sec\n",
                      suiteResult.getNumberOfTests(),
                      suiteResult.getNumberOfFailures(),
                      suiteResult.getNumberOfErrors(),
                      suiteResult.getNumberOfIgnores(),
                      suiteResult.getTime() / 1000.0f);
  }

  private void printContainerFailedToStart(String cause) {
    System.out.println("Container failed to start");
    System.out.println(cause);
  }

  private void printContainerFailedToStart(String suite, String parameterization, String cause) {
    String message = "Container failed to start for suite " + suite;
    if (isNotBlank(parameterization)) {
      message += "[" + parameterization + "]";
    }
    System.out.println(message);
    System.out.println(cause);
  }

  private void printTestSuiteFailedOrError(String suitePath, String cause, String tag) {
    System.out.format("%s failed <<< %s\n", suitePath, tag);
    System.out.println(cause);
  }

}
