/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.locators;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.maven.project.MavenProject;
import org.eclipse.aether.repository.RemoteRepository;

import static org.mule.munit.common.util.Collections.isEmpty;
import static org.mule.munit.common.util.Preconditions.checkArgument;


/**
 * <p>
 * Locates all {@link RemoteRepository} for a {@link MavenProject}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class RemoteRepositoriesLocator {

  private List<RemoteRepository> remoteRepositories;
  private List<String> whitelistRepositories = Collections.emptyList();

  public RemoteRepositoriesLocator(MavenProject mavenProject) {
    checkArgument(mavenProject != null, "Maven Project cannot be null");
    this.remoteRepositories = mavenProject.getRemoteProjectRepositories();
  }

  public RemoteRepositoriesLocator setWhiteListRepositories(List<String> whitelistRepositories) {
    checkArgument(whitelistRepositories != null, "Whitelist repositories cannot be null");
    this.whitelistRepositories = whitelistRepositories;
    return this;
  }

  public List<String> getWhitelistedRepositories() {
    return whitelistRepositories;
  }

  public List<RemoteRepository> locateAll() {
    return remoteRepositories;
  }

  public List<RemoteRepository> locate() {
    return isEmpty(whitelistRepositories) ? remoteRepositories : filterRepositories(remoteRepositories);
  }

  private List<RemoteRepository> filterRepositories(List<RemoteRepository> repositories) {
    return repositories.stream().filter(this::isWhiteListed).collect(Collectors.toList());
  }

  private boolean isWhiteListed(RemoteRepository remoteRepository) {
    return whitelistRepositories.stream().anyMatch(whitelistedRepo -> remoteRepository.getUrl().startsWith(whitelistedRepo));
  }

}
