/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.util;

import java.io.IOException;
import java.net.ServerSocket;
import java.util.Optional;

/**
 * The goal of this class is to obtain the port number that will be used to communicate between RemoteRunner and the MUnit runner
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerPortProvider {

  public static final int MIN_PORT_NUMBER = 60000;
  public static final int MAX_PORT_NUMBER = 60500;

  public static final String MUNIT_SERVER_PORT = "munit.server.port";

  /**
   * Returns the port number to be used.
   *
   * It will look for it in the defined system property, if not defined or illegal it will find one and define the system property
   * MUNIT_SERVER_PORT.
   * 
   * @return a port value
   */
  public Integer getPort() throws IllegalPortDefinitionException {
    Optional<Integer> predefinedPort = getPredefinedPort();
    Integer port = predefinedPort.orElseGet(this::getAndSetFreePort);

    if (port <= 0) {
      throw new IllegalPortDefinitionException("The defined port [" + port + "] is outside the valid range");
    }
    return port;
  }

  /**
   * Looks for a port number defined in a the property MUNIT_SERVER_PORT
   * 
   * @return empty Optional if no port defined or invalid value, the port otherwise
   */
  public Optional<Integer> getPredefinedPort() {
    Integer predefinedPort;
    try {
      predefinedPort = Integer.parseInt(System.getProperty(MUNIT_SERVER_PORT));
    } catch (NumberFormatException e) {
      predefinedPort = null;
    }
    return Optional.ofNullable(predefinedPort);
  }

  /**
   * Looks for a free port in the predefined range
   * 
   * @return a free port number
   */
  public Integer findFreePort() {
    FreePortFinder freePortFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
    return freePortFinder.find();
  }

  private Integer getAndSetFreePort() {
    Integer port = findFreePort();

    System.setProperty(MUNIT_SERVER_PORT, port.toString());
    return port;
  }

  public static boolean isPortAvailable(int port) {
    try (ServerSocket ignored = new ServerSocket(port)) {
      return true;
    } catch (IOException e) {
      return false;
    }
  }



}
