/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.properties;

import org.apache.commons.lang3.Validate;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

/**
 * {@link MUnitUserPropertiesManager} allows to set the system properties
 * defined by the user for the execution of the MUnit tests. It also allows
 * to save and restore the state of the system properties on call.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MUnitUserPropertiesManager {

  private static Properties initialSystemProperties;

  private static Set<String> notAllowedPropertyKeys = defineNotAllowedPropertyKeys();

  private Map<String, String> applicationProperties = new ConcurrentHashMap<String, String>();


  public MUnitUserPropertiesManager() {
    storeInitialSystemProperties();
  }

  public static void storeInitialSystemProperties() {
    initialSystemProperties = (Properties) System.getProperties().clone();
  }

  public static void restoreInitialSystemProperties() {
    System.setProperties(initialSystemProperties);
  }

  public static void addUserPropertiesToSystem(Map<String, String> userProperties) {
    if (userProperties != null) {
      for (Entry<String, String> entry : userProperties.entrySet()) {
        addUserPropertyToSystem(entry.getKey(), entry.getValue());
      }
    }
  }

  public static void addUserPropertyToSystem(String key, String value) {
    Validate.notBlank(key, "The property key must not be null nor empty");
    if (isPropertyKeyAllowed(key)) {
      System.setProperty(key, value);
    }
  }

  public static void loadPropertiesToSystem(Properties properties) {
    properties.forEach((k, v) -> System.setProperty((String) k, (String) v));
  }

  public static boolean hasSystemProperty(String key) {
    return System.getProperty(key) != null;
  }

  public static boolean hasEnvironmentProperty(String key) {
    return System.getenv(key) != null;
  }

  private static Boolean isPropertyKeyAllowed(String key) {
    return !notAllowedPropertyKeys.contains(key);
  }

  private static Set<String> defineNotAllowedPropertyKeys() {
    Set<String> propertyKeysSet = new HashSet<String>();
    propertyKeysSet.add("java.library.path");
    propertyKeysSet.add("file.encoding");
    propertyKeysSet.add("jdk.map.althashing.threshold");

    return propertyKeysSet;
  }

  public void addApplicationProperty(String key, String value) {
    Validate.notBlank(key, "The property key must not be null nor empty");
    if (isPropertyKeyAllowed(key)) {
      applicationProperties.put(key, value);
      System.setProperty(key, value);
    }
  }

  public Map<String, String> getApplicationProperties() {
    Map<String, String> appProp = new HashMap<String, String>();
    appProp.putAll(applicationProperties);
    return appProp;
  }

}
