/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.api.event;

import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.event.EventContext;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.event.EventContextFactory;
import org.mule.runtime.core.api.exception.NullExceptionHandler;
import org.mule.runtime.core.api.message.GroupCorrelation;
import org.mule.runtime.core.internal.message.ErrorBuilder;
import org.mule.runtime.core.internal.message.InternalMessage;
import org.mule.runtime.core.privileged.event.PrivilegedEvent;

import java.util.Map;
import java.util.Optional;
import java.util.UUID;

/**
 * This class provides a friendly API to build a {@link Event}
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class EventBuilder {

  private PrivilegedEvent.Builder muleEventBuilder;
  private InternalMessage.Builder muleMessageBuilder;

  /**
   * Generates an {@link EventContext from scratch}
   *
   * @param location
   * @return
   */
  private EventContext createEventContext(ComponentLocation location) {
    String id = UUID.randomUUID().toString();
    String serverId = UUID.randomUUID() + "-munit";
    return EventContextFactory.create(id, serverId, location, NullExceptionHandler.getInstance());
  }

  /**
   * Creates the event builder from a flow construct
   *
   * @param location location from where to create the new event
   */
  public EventBuilder(ComponentLocation location) {
    this.muleEventBuilder = PrivilegedEvent.builder(createEventContext(location));
    this.muleMessageBuilder = InternalMessage.builder().nullValue();
  }

  /**
   * Creates the event builder from an event context and a flow construct
   *
   * @param eventContext Flow construct from where to create the new event
   */
  public EventBuilder(EventContext eventContext) {
    this.muleEventBuilder = PrivilegedEvent.builder(eventContext);
    this.muleMessageBuilder = InternalMessage.builder().nullValue();
  }

  /**
   * Creates the event builder from an event
   *
   * @param originalEvent Base event from where to create the new one
   */
  public EventBuilder(Event originalEvent) {
    this.muleEventBuilder = PrivilegedEvent.builder((PrivilegedEvent) originalEvent);
    this.muleMessageBuilder = InternalMessage.builder(originalEvent.getMessage());
  }

  /**
   * Prepares the given data to be sent as the payload of the product.
   *
   * @param payload the payload to use in the message
   * @return this {@link EventBuilder}
   */
  public EventBuilder withPayload(Object payload) {
    muleMessageBuilder = muleMessageBuilder.value(payload);
    return this;
  }


  /**
   * Prepares the given data to be sent as the mediaType of the payload of the {@link Event} to the configured flow.
   *
   * @param mediaType the mediaType to use in the message
   * @return this {@link EventBuilder}
   */
  public EventBuilder withMediaType(MediaType mediaType) {
    muleMessageBuilder = muleMessageBuilder.mediaType(mediaType);
    return this;
  }

  /**
   * Sets the {@link org.mule.runtime.api.message.Message#getAttributes()} value of the produced message
   *
   * @param attributes the attributes object for the produced {@link org.mule.runtime.api.message.Message}
   * @return this {@link EventBuilder}
   */
  public EventBuilder withAttributes(Object attributes) {
    muleMessageBuilder = muleMessageBuilder.attributesValue(attributes);
    return this;
  }

  /**
   * Prepares a flow variable with the given key and value to be set in the product.
   *
   * @param attributes the attributes object for the produced {@link org.mule.runtime.api.message.Message}
   * @param mediaType the media type of the variable to add
   * @param charset the charset of the property to add
   * @return this {@link EventBuilder}
   */
  public EventBuilder withAttributes(Object attributes, String mediaType, String charset) {
    muleMessageBuilder =
        muleMessageBuilder.attributes(new TypedValue<>(attributes, buildDataType(attributes, mediaType, charset)));
    return this;
  }

  /**
   * Prepares a flow variable with the given key and value to be set in the product.
   *
   * @param key the key of the flow variable to put
   * @param value the value of the flow variable to put
   * @return this {@link EventBuilder}
   */
  public EventBuilder addVariable(String key, Object value) {
    muleEventBuilder = muleEventBuilder.addVariable(key, value);
    return this;
  }

  /**
   * Prepares a flow variable with the given key and value to be set in the product.
   *
   * @param key the key of the flow variable to put
   * @param value the value of the flow variable to put
   * @param mediaType the media type of the variable to add
   * @param charset the charset of the property to add
   * @return this {@link EventBuilder}
   */
  public EventBuilder addVariable(String key, Object value, String mediaType, String charset) {
    muleEventBuilder = muleEventBuilder.addVariable(key, value, buildDataType(value, mediaType, charset));
    return this;
  }

  /**
   * Prepares the flow variables with the given map. Should the map be null or empty the new event will contain an empty map.
   * Otherwise the content will be added to the pre existing map.
   *
   * @param variables the map to be set
   * @return this {@link EventBuilder}
   */
  public EventBuilder withVariables(Map<String, Object> variables) {
    muleEventBuilder = muleEventBuilder.variables(variables);
    return this;
  }

  /**
   * Configures the product event to have the provided {@code sourceCorrelationId}.
   *
   * @return this {@link EventBuilder}
   */
  public EventBuilder withSourceCorrelationId(String sourceCorrelationId) {
    muleEventBuilder = muleEventBuilder.correlationId(sourceCorrelationId);
    return this;
  }

  /**
   * Configures the product event to have the provided {@code correlation}.
   *
   * @return this {@link EventBuilder}
   */
  public EventBuilder withGroupCorrelation(GroupCorrelation correlation) {
    muleEventBuilder = muleEventBuilder.groupCorrelation(Optional.ofNullable(correlation));
    return this;
  }

  /**
   * Prepares the error to be added to the event
   *
   * @param errorType the type of the error
   * @param exception the exception of the error
   * @return this {@link EventBuilder}
   */
  public EventBuilder withError(ErrorType errorType, Throwable exception) {

    ErrorBuilder errorBuilder = ErrorBuilder.builder();
    errorBuilder
        .errorType(errorType)
        .description(exception.getMessage())
        .detailedDescription(exception.getMessage())
        .exception(exception);

    muleEventBuilder = muleEventBuilder.error(errorBuilder.build());
    return this;
  }

  /**
   * Produces an event with the specified configuration.
   *
   * @return an event with the specified configuration.
   */
  public Event build() {
    return muleEventBuilder.message(muleMessageBuilder.build()).build();
  }

  private DataType buildDataType(Object value, String mediaType, String charset) {
    return DataType.builder().fromObject(value).mediaType(mediaType).charset(charset).build();
  }

}
