/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.assertion.internal;

import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.everyItem;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.Matchers.closeTo;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasKey;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.hasValue;
import org.mule.munit.assertion.api.matchers.Matcher;
import org.mule.munit.assertion.internal.matchers.CallbackMatcher;
import org.mule.munit.assertion.internal.matchers.EncodingMatcher;
import org.mule.munit.assertion.internal.matchers.IterableMatcher;
import org.mule.munit.assertion.internal.matchers.MediaTypeMatcher;
import org.mule.munit.assertion.internal.matchers.TypedValueMatcher;
import org.mule.munit.assertion.internal.matchers.ValueMatcher;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;

import java.nio.charset.Charset;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * <p>
 * Factory that has the ability to create a Hamcrest Matcher based on a MUnit one
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class HamcrestFactory {


  /**
   * Creates the proper Hamcrest matcher for an MUnit matcher
   *
   * @param munitMatcher MUnit matcher that represents an assertion logic
   * @return Hamcrest matcher that corresponds to the MUnit one
   */
  public static org.hamcrest.Matcher<TypedValue> create(Matcher munitMatcher) {
    switch (munitMatcher.getType()) {
      case "equalTo":
      case "nullValue":
      case "notNullValue":
      case "matches":
      case "containsString":
      case "startsWith":
      case "endsWith":
      case "isEmptyString":
      case "isEmptyOrNullString":
      case "equalToIgnoringCase":
      case "equalToIgnoringWhiteSpace":
      case "stringContainsInOrder":
      case "greaterThan":
      case "greaterThanOrEqualTo":
      case "lessThan":
      case "lessThanOrEqualTo":
        return new CallbackMatcher((Function) munitMatcher.getExpected()[0]);
      case "not":
        return not(create((Matcher) (munitMatcher.getExpected())[0]));
      case "both":
        return allOf(create((Matcher) munitMatcher.getExpected()[0]), create((Matcher) munitMatcher.getExpected()[1]));
      case "either":
        return anyOf(create((Matcher) munitMatcher.getExpected()[0]), create((Matcher) munitMatcher.getExpected()[1]));
      case "withEncoding":
        String charset = (String) munitMatcher.getExpected()[0];
        return new EncodingMatcher(equalTo(charset == null ? null : Charset.forName(charset)));
      case "withMediaType":
        String mediaType = (String) munitMatcher.getExpected()[0];
        return new MediaTypeMatcher(mediaType == null ? null : MediaType.parse(mediaType));
      case "allOf":
        return allOf(Stream.of(munitMatcher.getExpected()).map(matcher -> (Matcher) matcher).map(HamcrestFactory::create)
            .collect(toList()));
      case "anyOf":
        return anyOf(Stream.of(munitMatcher.getExpected()).map(matcher -> (Matcher) matcher).map(HamcrestFactory::create)
            .collect(toList()));
      case "closeTo":
        return new ValueMatcher(closeTo(((Number) munitMatcher.getExpected()[0]).doubleValue(),
                                        ((Number) munitMatcher.getExpected()[1]).doubleValue()));
      case "everyItem":
        return new IterableMatcher(everyItem(toPlainMatcher(create((Matcher) munitMatcher.getExpected()[0]))));
      case "hasItem":
        return new IterableMatcher(hasItem(toPlainMatcher(create((Matcher) munitMatcher.getExpected()[0]))));
      case "hasSize":
        return new IterableMatcher(hasSize(toPlainMatcher(create((Matcher) munitMatcher.getExpected()[0]))));
      case "isEmpty":
        return new IterableMatcher(empty());
      case "hasKey":
        return new ValueMatcher(hasKey(toPlainMatcher(create((Matcher) munitMatcher.getExpected()[0]))));
      case "hasValue":
        return new ValueMatcher(hasValue(toPlainMatcher(create((Matcher) munitMatcher.getExpected()[0]))));
    }
    throw new IllegalArgumentException("Unknown Matcher: " + munitMatcher.getType());
  }

  private static org.hamcrest.Matcher<Object> toPlainMatcher(org.hamcrest.Matcher<TypedValue> typedValueMatcher) {
    if (typedValueMatcher instanceof TypedValueMatcher) {
      return ((TypedValueMatcher) typedValueMatcher).toPlainValueMatcher();
    }
    throw new IllegalArgumentException("The provided matcher should only be applicable for the value of TypedValue");
  }
}
