/*
 * Copyright 2012 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mobgame;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.os.Handler;
import android.os.Vibrator;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.TaskStackBuilder;
import android.support.v4.content.LocalBroadcastManager;
import android.text.TextUtils;
import android.util.Log;

import com.google.android.gcm.GCMBaseIntentService;
import com.google.android.gcm.GCMRegistrar;
import com.mobgame.api.SaveGCMTask;
import com.mobgame.component.GameConfigManager;
import com.mobgame.utils.Constants;
import com.mobgame.utils.DeviceUtils;
import com.mobgame.utils.Preference;
import com.mobgame.utils.Res;
import com.mobgame.utils.Utils;

/**
 * IntentService responsible for handling GCM messages.
 */
public class MobGameGCMIntentService extends GCMBaseIntentService {

	private static final String TAG = MobGameGCMIntentService.class.getSimpleName();
	private static final String GCM_SENDER_ID = "461098885975";

	private static Class<?> mainClass;
	private static Handler registerHandler = new Handler();

	public MobGameGCMIntentService() {
		super(GCM_SENDER_ID);
	}

	protected int getDrawableSmall(Context c) {
		return Res.drawableResource(c, R.drawable.ic_stat_mob);
	}

	@Override
	protected void onRegistered(Context c, String registrationId) {
		Log.i(TAG, "onRegistered:" + registrationId);
		registerToServer(c, registrationId);
	}

	@Override
	protected void onUnregistered(Context context, String regId) {
		Log.i(TAG, "onUnregistered:" + regId);
	}
	
	@Override
    protected void onError(Context context, String errorId) {
        Log.i(TAG, "onError: " + errorId);
    }
	
	@Override
	protected boolean onRecoverableError(Context context, String errorId) {
		Log.i(TAG, "onRecoverableError: " + errorId);
		return super.onRecoverableError(context, errorId);
	}

	@Override
	protected void onMessage(Context context, Intent messageIntent) {
		Log.i(TAG, "onMessage: " + messageIntent);
		Vibrator v = (Vibrator) context.getSystemService(Context.VIBRATOR_SERVICE);
		v.vibrate(30); // Vibrate for 30 ms

		Bundle extras = messageIntent.getExtras();
		String title = extras.getString("title");
		String message = extras.getString("message");
		String id = extras.getString("id");

		Log.i(TAG, "id: " + id);
		Log.i(TAG, "title: " + title);
		Log.i(TAG, "message: " + message);
		// Send broadcast
		Intent intent = new Intent(Constants.INTENT_FILTER);
		intent.putExtra("category", "gcm");
		intent.putExtra("id", id);
		intent.putExtra("title", title);
		intent.putExtra("mesage", message);
		LocalBroadcastManager.getInstance(context).sendBroadcast(intent);
		
		GameConfigManager.getInstance().setHasUnreadNotifications(true);
		if (!TextUtils.isEmpty(id)) {
			Utils.addNtf(context, Integer.parseInt(id));
		}
		if (!TextUtils.isEmpty(title)
		||	!TextUtils.isEmpty(message)) {
			showNotification(context, title, message);
		}
	}

	protected void showNotification(Context context, String title, String message) {
		Log.i(TAG, "showNotification:title=" + title + ";msg=" + message);
    	try {
    		Log.i(TAG, "mainClass:" + mainClass);
    		if (mainClass == null) {
    			try {
	    			String className = Preference.getString(context, 
	        				Constants.SHARED_PREF_MAIN_ACTIVITY);
	    			mainClass = Class.forName(className);
    			} catch (Exception e) {
    				e.printStackTrace();
    				return;
    			}
    		}
    		Drawable ico = getApplicationInfo().loadIcon(getPackageManager());
    		Bitmap bitmap = Utils.drawableToBitmap(ico);
    		int bitmapSize = (int) Math.min(64 * DeviceUtils.getDensity(context), bitmap.getWidth());

    		CharSequence appTitle = context.getApplicationInfo().loadLabel(context.getPackageManager());
    		
    		NotificationCompat.BigTextStyle bigTextStyle = new NotificationCompat.BigTextStyle();
    		bigTextStyle.setBigContentTitle(title);
    		bigTextStyle.bigText(message);
    		bigTextStyle.setSummaryText(appTitle);

			NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(this)
				.setWhen(System.currentTimeMillis())
				.setContentText(message)
				.setContentTitle(title)
				.setSmallIcon(getDrawableSmall(context))
				.setLargeIcon(Bitmap.createScaledBitmap(bitmap, bitmapSize, bitmapSize, false))
				.setAutoCancel(true)
				.setTicker(title)
				.setNumber(Utils.countNtfs(context))
				.setStyle(bigTextStyle)
				.setDefaults(Notification.DEFAULT_LIGHTS | Notification.DEFAULT_VIBRATE | Notification.DEFAULT_SOUND);			

	        // Creates an explicit intent for an Activity in your app
			Intent resultIntent = new Intent(this, mainClass);
			resultIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TOP);
//			resultIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
			resultIntent.putExtra("title", title);
			resultIntent.putExtra("message", message);

	        // The stack builder object will contain an artificial back stack for
	        // the started Activity.
	        // This ensures that navigating backward from the Activity leads out of
	        // your application to the Home screen.
			TaskStackBuilder stackBuilder = TaskStackBuilder.create(this);
			// Adds the back stack for the Intent (but not the Intent itself)
			stackBuilder.addParentStack(mainClass);
			// Adds the Intent that starts the Activity to the top of the stack
			stackBuilder.addNextIntent(resultIntent);
			//PendingIntent resultPendingIntent = stackBuilder.getPendingIntent(0, PendingIntent.FLAG_UPDATE_CURRENT);
			PendingIntent resultPendingIntent = PendingIntent.getActivity(context,
						0, resultIntent, 0);
			mBuilder.setContentIntent(resultPendingIntent);
			NotificationManager mNotificationManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
			// mId allows you to update the notification later on.
			mNotificationManager.notify(1, mBuilder.build());
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	static void register(Context context) {
    	Log.i(TAG, "register");
    	try {
	    	if (context instanceof Activity) {
	    		Activity activity = (Activity) context;
	    		mainClass = activity.getClass();
	    		Preference.save(context, 
	    				Constants.SHARED_PREF_MAIN_ACTIVITY, 
	    				mainClass.getName());
	    		 Log.i(TAG, "mainClass:" + mainClass.getName());
	    	}
	        // Make sure the device has the proper dependencies.
	    	final Context appContext = context.getApplicationContext();
	        GCMRegistrar.checkDevice(appContext);
	        // Make sure the manifest was properly set - comment out this line
	        // while developing the app, then uncomment it when it's ready.
	        GCMRegistrar.checkManifest(appContext);
	
	        final String regId = GCMRegistrar.getRegistrationId(appContext);
	        Log.i(TAG, "regId:" + regId);
	        if (TextUtils.isEmpty(regId)) {
	            // Automatically registers application on startup.
	        	Log.i(TAG, "Automatically registers application on startup");
	            GCMRegistrar.register(appContext, GCM_SENDER_ID);
	        } else {
	        	registerToServer(appContext, regId);
			}
    	} catch (Exception e) {
    		// Exception might occur on devices without GG Play Store
    	}
	}

	static void destroy(Context context) {
		if (registerHandler != null) {
			registerHandler.removeCallbacksAndMessages(null);
		}
		GCMRegistrar.onDestroy(context.getApplicationContext());
	}

	private static void registerToServer(Context context, String regId) {
    	Log.i(TAG, "register");
    	if (!TextUtils.isEmpty(regId)) {
			(new SaveGCMTask(regId)).execute();
		}
	}

}
