"use strict";
/**
 * Copyright 2017 Google Inc. All rights reserved.
 * Modifications copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.isBrowserDirectory = exports.markerFilePath = exports.browserDirectory = exports.browsersPath = exports.executablePath = exports.windowsExeAndDllDirectories = exports.linuxLddDirectories = exports.hostPlatform = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const ubuntuVersion_1 = require("./ubuntuVersion");
const utils_1 = require("./utils");
exports.hostPlatform = (() => {
    const platform = os.platform();
    if (platform === 'darwin') {
        const [major, minor] = child_process_1.execSync('sw_vers -productVersion', {
            stdio: ['ignore', 'pipe', 'ignore']
        }).toString('utf8').trim().split('.').map(x => parseInt(x, 10));
        let arm64 = false;
        // BigSur is the first version that might run on Apple Silicon.
        if (major >= 11) {
            arm64 = child_process_1.execSync('sysctl -in hw.optional.arm64', {
                stdio: ['ignore', 'pipe', 'ignore']
            }).toString().trim() === '1';
        }
        // We do not want to differentiate between minor big sur releases
        // since they don't change core APIs so far.
        const macVersion = major === 10 ? `${major}.${minor}` : `${major}`;
        const archSuffix = arm64 ? '-arm64' : '';
        return `mac${macVersion}${archSuffix}`;
    }
    if (platform === 'linux') {
        const ubuntuVersion = ubuntuVersion_1.getUbuntuVersionSync();
        if (parseInt(ubuntuVersion, 10) <= 19)
            return 'ubuntu18.04';
        return 'ubuntu20.04';
    }
    if (platform === 'win32')
        return os.arch() === 'x64' ? 'win64' : 'win32';
    return platform;
})();
function linuxLddDirectories(browserPath, browser) {
    if (browser.name === 'chromium')
        return [path.join(browserPath, 'chrome-linux')];
    if (browser.name === 'firefox')
        return [path.join(browserPath, 'firefox')];
    if (browser.name === 'webkit') {
        return [
            path.join(browserPath, 'minibrowser-gtk'),
            path.join(browserPath, 'minibrowser-gtk', 'bin'),
            path.join(browserPath, 'minibrowser-gtk', 'lib'),
            path.join(browserPath, 'minibrowser-wpe'),
            path.join(browserPath, 'minibrowser-wpe', 'bin'),
            path.join(browserPath, 'minibrowser-wpe', 'lib'),
        ];
    }
    return [];
}
exports.linuxLddDirectories = linuxLddDirectories;
function windowsExeAndDllDirectories(browserPath, browser) {
    if (browser.name === 'chromium')
        return [path.join(browserPath, 'chrome-win')];
    if (browser.name === 'firefox')
        return [path.join(browserPath, 'firefox')];
    if (browser.name === 'webkit')
        return [browserPath];
    return [];
}
exports.windowsExeAndDllDirectories = windowsExeAndDllDirectories;
function executablePath(browserPath, browser) {
    let tokens;
    if (browser.name === 'chromium') {
        tokens = new Map([
            ['ubuntu18.04', ['chrome-linux', 'chrome']],
            ['ubuntu20.04', ['chrome-linux', 'chrome']],
            ['mac10.13', ['chrome-mac', 'Chromium.app', 'Contents', 'MacOS', 'Chromium']],
            ['mac10.14', ['chrome-mac', 'Chromium.app', 'Contents', 'MacOS', 'Chromium']],
            ['mac10.15', ['chrome-mac', 'Chromium.app', 'Contents', 'MacOS', 'Chromium']],
            ['mac11', ['chrome-mac', 'Chromium.app', 'Contents', 'MacOS', 'Chromium']],
            ['mac11-arm64', ['chrome-mac', 'Chromium.app', 'Contents', 'MacOS', 'Chromium']],
            ['win32', ['chrome-win', 'chrome.exe']],
            ['win64', ['chrome-win', 'chrome.exe']],
        ]).get(exports.hostPlatform);
    }
    if (browser.name === 'firefox') {
        tokens = new Map([
            ['ubuntu18.04', ['firefox', 'firefox']],
            ['ubuntu20.04', ['firefox', 'firefox']],
            ['mac10.13', ['firefox', 'Nightly.app', 'Contents', 'MacOS', 'firefox']],
            ['mac10.14', ['firefox', 'Nightly.app', 'Contents', 'MacOS', 'firefox']],
            ['mac10.15', ['firefox', 'Nightly.app', 'Contents', 'MacOS', 'firefox']],
            ['mac11', ['firefox', 'Nightly.app', 'Contents', 'MacOS', 'firefox']],
            ['mac11-arm64', ['firefox', 'Nightly.app', 'Contents', 'MacOS', 'firefox']],
            ['win32', ['firefox', 'firefox.exe']],
            ['win64', ['firefox', 'firefox.exe']],
        ]).get(exports.hostPlatform);
    }
    if (browser.name === 'webkit') {
        tokens = new Map([
            ['ubuntu18.04', ['pw_run.sh']],
            ['ubuntu20.04', ['pw_run.sh']],
            ['mac10.13', undefined],
            ['mac10.14', ['pw_run.sh']],
            ['mac10.15', ['pw_run.sh']],
            ['mac11', ['pw_run.sh']],
            ['mac11-arm64', ['pw_run.sh']],
            ['win32', ['Playwright.exe']],
            ['win64', ['Playwright.exe']],
        ]).get(exports.hostPlatform);
    }
    return tokens ? path.join(browserPath, ...tokens) : undefined;
}
exports.executablePath = executablePath;
function cacheDirectory() {
    if (process.platform === 'linux')
        return process.env.XDG_CACHE_HOME || path.join(os.homedir(), '.cache');
    if (process.platform === 'darwin')
        return path.join(os.homedir(), 'Library', 'Caches');
    if (process.platform === 'win32')
        return process.env.LOCALAPPDATA || path.join(os.homedir(), 'AppData', 'Local');
    throw new Error('Unsupported platform: ' + process.platform);
}
const defaultBrowsersPath = (() => {
    const envDefined = utils_1.getFromENV('PLAYWRIGHT_BROWSERS_PATH');
    if (envDefined === '0')
        return undefined;
    return envDefined || path.join(cacheDirectory(), 'ms-playwright');
})();
function browsersPath(packagePath) {
    return defaultBrowsersPath || path.join(packagePath, '.local-browsers');
}
exports.browsersPath = browsersPath;
function browserDirectory(browsersPath, browser) {
    return path.join(browsersPath, `${browser.name}-${browser.revision}`);
}
exports.browserDirectory = browserDirectory;
function markerFilePath(browsersPath, browser) {
    return path.join(browserDirectory(browsersPath, browser), 'INSTALLATION_COMPLETE');
}
exports.markerFilePath = markerFilePath;
function isBrowserDirectory(browserPath) {
    const baseName = path.basename(browserPath);
    return baseName.startsWith('chromium-') || baseName.startsWith('firefox-') || baseName.startsWith('webkit-');
}
exports.isBrowserDirectory = isBrowserDirectory;
//# sourceMappingURL=browserPaths.js.map