"use strict";
/**
 * Copyright 2017 Google Inc. All rights reserved.
 * Modifications copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.exceptionToError = exports.toConsoleMessageLocation = exports.readProtocolStream = exports.releaseObject = exports.getExceptionMessage = void 0;
const fs = require("fs");
const util = require("util");
const utils_1 = require("../../utils/utils");
function getExceptionMessage(exceptionDetails) {
    if (exceptionDetails.exception)
        return exceptionDetails.exception.description || String(exceptionDetails.exception.value);
    let message = exceptionDetails.text;
    if (exceptionDetails.stackTrace) {
        for (const callframe of exceptionDetails.stackTrace.callFrames) {
            const location = callframe.url + ':' + callframe.lineNumber + ':' + callframe.columnNumber;
            const functionName = callframe.functionName || '<anonymous>';
            message += `\n    at ${functionName} (${location})`;
        }
    }
    return message;
}
exports.getExceptionMessage = getExceptionMessage;
async function releaseObject(client, objectId) {
    await client.send('Runtime.releaseObject', { objectId }).catch(error => { });
}
exports.releaseObject = releaseObject;
async function readProtocolStream(client, handle, path) {
    let eof = false;
    let fd;
    if (path) {
        await utils_1.mkdirIfNeeded(path);
        fd = await util.promisify(fs.open)(path, 'w');
    }
    const bufs = [];
    while (!eof) {
        const response = await client.send('IO.read', { handle });
        eof = response.eof;
        const buf = Buffer.from(response.data, response.base64Encoded ? 'base64' : undefined);
        bufs.push(buf);
        if (path)
            await util.promisify(fs.write)(fd, buf);
    }
    if (path)
        await util.promisify(fs.close)(fd);
    await client.send('IO.close', { handle });
    return Buffer.concat(bufs);
}
exports.readProtocolStream = readProtocolStream;
function toConsoleMessageLocation(stackTrace) {
    return stackTrace && stackTrace.callFrames.length ? {
        url: stackTrace.callFrames[0].url,
        lineNumber: stackTrace.callFrames[0].lineNumber,
        columnNumber: stackTrace.callFrames[0].columnNumber,
    } : { url: '', lineNumber: 0, columnNumber: 0 };
}
exports.toConsoleMessageLocation = toConsoleMessageLocation;
function exceptionToError(exceptionDetails) {
    const messageWithStack = getExceptionMessage(exceptionDetails);
    const lines = messageWithStack.split('\n');
    const firstStackTraceLine = lines.findIndex(line => line.startsWith('    at'));
    let message = '';
    let stack = '';
    if (firstStackTraceLine === -1) {
        message = messageWithStack;
    }
    else {
        message = lines.slice(0, firstStackTraceLine).join('\n');
        stack = messageWithStack;
    }
    const match = message.match(/^[a-zA-Z0-0_]*Error: (.*)$/);
    if (match)
        message = match[1];
    const err = new Error(message);
    err.stack = stack;
    return err;
}
exports.exceptionToError = exceptionToError;
//# sourceMappingURL=crProtocolHelper.js.map