(:
  Copyright (c) 2021 MarkLogic Corporation
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
:)
xquery version "1.0-ml";

module namespace hent = "http://marklogic.com/data-hub/hub-entities/latest";

import module namespace es = "http://marklogic.com/entity-services"
  at "/data-hub/core/entity-services/entity-services.xqy";

import module namespace ext = "http://marklogic.com/data-hub/extensions/entity"
  at "/data-hub/extensions/entity/post-process-search-options.xqy";

 import module namespace sem = "http://marklogic.com/semantics" at "/MarkLogic/semantics.xqy";

import module namespace functx = "http://www.functx.com" at "/MarkLogic/functx/functx-1.0-nodoc-2007-01.xqy";

declare namespace search = "http://marklogic.com/appservices/search";
declare namespace tde = "http://marklogic.com/xdmp/tde";


declare option xdmp:mapping "true";

declare %private variable $DEFAULT_BASE_URI := "http://example.org/";

declare function extraction-template-generate(
  $model as map:map,
  $generate-triples as xs:boolean,
  $generate-views as xs:boolean
) as element(tde:template)
{
  let $top-id := map:get($model,"$id")
  let $schema-name := $model=>map:get("info")=>map:get("title")
  let $entity-type-names := $model=>map:get("definitions")=>map:keys()
  let $scalar-rows := map:map()
  let $secure-tde-name := fn:replace(?, "-", "_")
  let $model-definitions := $model=>map:get("definitions")
  let $path-namespaces :=
    for $definition in  ($model-definitions=>map:keys()) ! ($model-definitions=>map:get(.))
    let $namespace-prefix := $definition=>map:get("namespacePrefix")
    let $namespace := $definition=>map:get("namespace")
    where $namespace and $namespace-prefix
    return
      <tde:path-namespace>
        <tde:prefix>{$namespace-prefix}</tde:prefix>
        <tde:namespace-uri>{$namespace}</tde:namespace-uri>
      </tde:path-namespace>
  let $local-references :=
    if (count($entity-type-names)=1) then ()
    else local-references($model)
  let $top-entity := top-entity($model, true())
  let $top-entity-type := $model=>map:get("definitions")=>map:get($top-entity)
  let $top-primary-key-name := map:get($top-entity-type, "primaryKey")
  let $namespace-prefix := $top-entity-type=>map:get("namespacePrefix")
  let $namespace-uri := $top-entity-type=>map:get("namespace")
  let $prefix-value :=
           if ($namespace-uri)
           then
             $namespace-prefix || ":"
           else ""

  let $maybe-local-refs := if (exists($top-entity)) then () else local-references($model)
  let $entity-type-templates := build-nested-templates($schema-name, (), $secure-tde-name, $model, $top-entity, $top-entity-type)
  let $model-version := $model=>map:get("info")=>map:get("version")
  return
    <tde:template xmlns:tde="http://marklogic.com/xdmp/tde" xml:lang="zxx">
      <tde:description>
        Extraction Template Generated from Entity Type Document
        graph uri: {model-graph-iri($model)}
      </tde:description>
      <!-- The following line matches JSON and XML instances, but may be slower to index documents. -->
      <tde:context>/(es:envelope|envelope)/(es:instance|instance)[es:info/es:version = '{$model-version}' or info/version = '{$model-version}'][{ if ($prefix-value) then "(" || $prefix-value || $top-entity || "|" || $top-entity || ")" else $top-entity }]</tde:context>
      <tde:vars>
        <tde:var><tde:name>RDF</tde:name><tde:val>"http://www.w3.org/1999/02/22-rdf-syntax-ns#"</tde:val></tde:var>
        <tde:var><tde:name>RDF_TYPE</tde:name><tde:val>sem:iri(concat($RDF, "type"))</tde:val></tde:var>
         <tde:var><tde:name>top-primary-key-val</tde:name><tde:val>fn:encode-for-uri(fn:head(./{ if ($prefix-value) then "(" || $prefix-value || $top-entity || "|" || $top-entity || ")" else $top-entity }/{ if ($prefix-value) then "(" || 	$prefix-value || $top-primary-key-name || "|" || $top-primary-key-name || ")" else $top-primary-key-name } ! xs:string(.)[. ne ""]))</tde:val></tde:var>
         <tde:var><tde:name>subject-iri</tde:name><tde:val>sem:iri(concat("{ model-graph-prefix($model) }/{ $top-entity }/", if (fn:empty($top-primary-key-val) or $top-primary-key-val eq "") then sem:uuid-string() else $top-primary-key-val))</tde:val></tde:var>
      </tde:vars>
      <tde:path-namespaces>
        <tde:path-namespace>
          <tde:prefix>es</tde:prefix>
          <tde:namespace-uri>http://marklogic.com/entity-services</tde:namespace-uri>
        </tde:path-namespace>
        { $path-namespaces }
      </tde:path-namespaces>
      {
        if ( $entity-type-templates/element() )
        then
          <tde:templates>
            {
              $entity-type-templates
            }
          </tde:templates>
        else
          comment { "An entity type must have at least one required column or a primary key to generate an extraction template." }
      }
    </tde:template>
};

declare function model-graph-iri(
    $model as map:map
) as sem:iri
{
    let $info := map:get($model, "info")
    let $base-uri-prefix := resolve-base-uri($info)
    return
    sem:iri(
        concat( $base-uri-prefix,
               map:get($info, "title"),
               "-" ,
               map:get($info, "version")))
};

(: resolves the default URI from a model's info section :)
declare function resolve-base-uri(
    $info as map:map
) as xs:string
{
    let $base-uri := fn:head((map:get($info, "baseUri"), $DEFAULT_BASE_URI))
    return
        if (fn:matches($base-uri, "[#/]$"))
        then $base-uri
        else concat($base-uri, "#")
};

declare function model-graph-prefix(
    $model as map:map
) as sem:iri
{
    let $info := map:get($model, "info")
    let $base-uri-prefix := resolve-base-prefix($info)
    return
    sem:iri(
        concat( $base-uri-prefix,
               map:get($info, "title"),
               "-" ,
               map:get($info, "version")))
};


declare %private function build-nested-templates-for-ref-type($schema-name, $secure-tde-name, $model, $entity-type-name, $property-name) {
  let $ref-type-name := ref-type-name($model, $entity-type-name, $property-name)
  let $ref-type := ref-type($model, $entity-type-name, $property-name)
  where fn:exists($ref-type)
  return
    build-nested-templates($schema-name, $entity-type-name, $secure-tde-name, $model, $ref-type-name, $ref-type)
};

declare %private function build-nested-templates($schema-name, $parent-entity-type-name, $secure-tde-name, $model, $ref-type-name, $ref-type) {
  let $ref-primary-key := $ref-type=>map:get("primaryKey")
  let $ref-prefix := $ref-type=>map:get("namespacePrefix")
  let $ref-prefix-value := if (fn:exists($ref-prefix)) then $ref-prefix || ":" else ()
  let $ref-primary-key := $ref-type=>map:get("primaryKey")
  let $ref-properties := $ref-type=>map:get("properties")
  let $ref-primary-key-type := $ref-properties => map:get($ref-primary-key) => map:get("datatype")
  let $related-entity-type-property-names :=
                     for $property-name in map:keys($ref-properties)
                     let $property-info := map:get($ref-properties, $property-name)
                     let $property-info := if (map:contains($property-info, "items")) then map:get($property-info, "items") else $property-info
                     let $related-entity-type := map:get($property-info, "relatedEntityType" )
                     where exists($related-entity-type)
                     return $property-name
  let $sub-templates := (
    join-name-triples($model, $ref-type-name, $ref-prefix-value, $ref-primary-key),
    let $visited-schema-view := map:map()
    for $prop-name in $ref-properties=>map:keys()
    let $property := $ref-properties=>map:get($prop-name)
    where $property => map:contains("items")
    return
      let $datatype := $property => map:get("items") => map:get("datatype")
      let $view-name-prefix := ($ref-type-name=>$secure-tde-name()) || "_"
      let $is-local-ref := is-local-reference($model, $ref-type-name, $prop-name)
      let $sub-ref-type-name := if ($is-local-ref) then ref-type-name($model, $ref-type-name, $prop-name) else ()
      let $sub-ref-type := if ($is-local-ref) then ref-type($model, $ref-type-name, $prop-name) else ()
      let $sub-related-entity-type := ($property => map:get("items") => map:get("relatedEntityType"))[. ne ""]
      where (fn:exists($datatype) or $is-local-ref) and fn:not(map:contains($visited-schema-view, $sub-ref-type-name))
      return (
        if ($sub-ref-type-name) then (
          let $nested-template := build-nested-templates($schema-name, $ref-type-name, $secure-tde-name, $model, $sub-ref-type-name, $sub-ref-type)
          where fn:exists($nested-template)
          return (
            $nested-template,
            map:put($visited-schema-view, $sub-ref-type-name, fn:true())
          )
        ) else (),
        <tde:template>
          <tde:context>{
            if ($ref-prefix-value) then
              "(" || $ref-prefix-value || $prop-name || "|" || $prop-name ||")"
            else $prop-name
          }</tde:context>
          {if ($sub-ref-type-name) then
              <tde:vars>
                <tde:var>
                  <tde:name>subject-iri</tde:name>
                  <tde:val>{'sem:iri(concat("'|| model-graph-prefix($model) ||'/'|| $sub-ref-type-name || '/", if (fn:empty($primary-key-val) or $primary-key-val eq "") then sem:uuid-string() else $primary-key-val))'}</tde:val>
                </tde:var>
              </tde:vars>
            else (
          <tde:rows>
            <tde:row>
              <tde:schema-name>{ $schema-name=>$secure-tde-name() }</tde:schema-name>
              <tde:view-name>{ $view-name-prefix || $prop-name=>$secure-tde-name() }</tde:view-name>
              <tde:view-layout>sparse</tde:view-layout>
              <tde:columns>
                <tde:column>
                  { comment { "This column joins to property",
                  $ref-primary-key, "of",
                  $ref-type-name } }
                  <tde:name>{ $ref-type-name=>$secure-tde-name() }_{ $ref-primary-key=>$secure-tde-name() }</tde:name>
                  <tde:scalar-type>{ if ($ref-primary-key-type="iri") then "IRI" else $ref-primary-key-type }</tde:scalar-type>
                  <tde:val>fn:head({if ($ref-prefix-value) then "(ancestor::" || $ref-prefix-value || $ref-type-name || "|" || "ancestor::" || $ref-type-name || ")" else "(ancestor::" || $ref-type-name || ")"}/{ if ($ref-prefix-value) then  "("|| $ref-prefix-value || $ref-primary-key || "|" || $ref-primary-key || ")" else $ref-primary-key }[xs:string(.) ne ""])</tde:val>
                </tde:column>
                <tde:column>
                  <tde:name>{ $secure-tde-name($view-name-prefix || "dataHubIRI") }</tde:name>
                  <tde:scalar-type>IRI</tde:scalar-type>
                  <tde:val>$subject-iri</tde:val>
                </tde:column>
                {
                  if ($sub-ref-type-name) then (
                    <tde:column>
                      <tde:name>{ $secure-tde-name($view-name-prefix || "propertyName") }</tde:name>
                      <tde:scalar-type>string</tde:scalar-type>
                      <tde:val>fn:string(fn:node-name(..))</tde:val>
                    </tde:column>
                  ) else ()
                }
                {
                  if (fn:exists($datatype)) then
                    <tde:column>
                      <tde:name>{ $prop-name=>$secure-tde-name() }</tde:name>
                      <tde:scalar-type>{ if ($datatype="iri") then "IRI" else $datatype }</tde:scalar-type>
                      <tde:val>.[xs:string(.) ne ""]</tde:val>
                    </tde:column>
                  else
                    build-columns-for-local-ref("", "", $secure-tde-name, $model, $ref-type-name, $prop-name)
                }</tde:columns>
            </tde:row>
          </tde:rows>)
          }
          {
          let $triples := (
              if ($is-local-ref) then (
                  let $model-iri  := model-graph-iri($model)
                  let $sub-ref-type-properties := $sub-ref-type => map:get("properties")
                  for $ref-prop-name in $sub-ref-type-properties => map:keys()
                  let $ref-prop := $sub-ref-type-properties => map:get($ref-prop-name)
                  let $ref-related-entity-type := $sub-ref-type => map:get($ref-prop-name) => map:get("relatedEntityType")
                  where fn:exists($ref-related-entity-type)
                  return (
                      <tde:triple>
                        <tde:subject>
                      <tde:val>$subject-iri</tde:val>
                        <tde:invalid-values>ignore</tde:invalid-values>
                      </tde:subject>
                      <tde:predicate>
                        <tde:val>sem:iri("{ $model-iri }/{ $sub-ref-type-name }/{ $ref-prop-name}")</tde:val>
                        <tde:invalid-values>ignore</tde:invalid-values>
                        </tde:predicate>
                      <tde:object>
                        <tde:val>sem:iri(concat("{ $ref-related-entity-type }/", fn:encode-for-uri(xs:string({$ref-prop-name}))))</tde:val>
                          <tde:invalid-values>ignore</tde:invalid-values>
                      </tde:object>
                      </tde:triple>
                  )
                ) else (),
                if (fn:exists($sub-related-entity-type)) then (
                  let $model-iri  := model-graph-iri($model)
                  return (
                      <tde:triple>
                        <tde:subject>
                      <tde:val>$subject-iri</tde:val>
                        <tde:invalid-values>ignore</tde:invalid-values>
                      </tde:subject>
                      <tde:predicate>
                        <tde:val>sem:iri("{ $model-iri }/{ $ref-type-name }/{ $prop-name}")</tde:val>
                        <tde:invalid-values>ignore</tde:invalid-values>
                        </tde:predicate>
                      <tde:object>
                        <tde:val>sem:iri(concat("{ $sub-related-entity-type }/", fn:encode-for-uri(xs:string(.))))</tde:val>
                          <tde:invalid-values>ignore</tde:invalid-values>
                      </tde:object>
                      </tde:triple>
                    )
                ) else ()
              )
            where fn:exists($triples)
            return <tde:triples>{$triples}</tde:triples>
          }
        </tde:template>
      ),
      if (fn:empty($parent-entity-type-name)) then (
        let $map-related-concepts := map:get($ref-type, "relatedConcepts")
        let $defaultValueExpression := "sem:iri(fn:replace(fn:string(.),'\\s+', ''))"
        for $concept in json:array-values($map-related-concepts)
        let $predicate_concept := map:get($concept, "predicate")
        let $concept-context := map:get($concept, "context")
        let $context := get-property-xpath($model, $ref-type-name, fn:tokenize($concept-context, "/"))
        let $expression := map:get($concept, "conceptExpression")
         let $conceptExpression :=
               if (fn:string($expression) eq "" or fn:empty($expression))
               then $defaultValueExpression
               else $expression
         let $contextWithValidation :=
               if (fn:string($context) eq ".")
               then $context
               else $context || "[xs:string(.) ne """"]"
        let $concept_class:=map:get($concept, "conceptClass")
        return
          <tde:template>
           <tde:context>{ $contextWithValidation}</tde:context>
          <tde:triples>
          <tde:triple>
            <tde:subject><tde:val>$subject-iri</tde:val></tde:subject>
            <tde:predicate><tde:val>sem:iri("{ $predicate_concept}")</tde:val></tde:predicate>
            <tde:object><tde:val>{ $conceptExpression}</tde:val></tde:object>
          </tde:triple>
         <tde:triple>
           <tde:subject><tde:val>sem:iri("{ model-graph-prefix($model) }/{ $ref-type-name }")</tde:val></tde:subject>
           <tde:predicate><tde:val>sem:iri("http://www.marklogic.com/data-hub#relatedConcept")</tde:val></tde:predicate>
           <tde:object><tde:val>sem:iri("{ $concept_class}")</tde:val></tde:object>
         </tde:triple>
         <tde:triple>
           <tde:subject><tde:val>sem:iri("{ $concept_class}")</tde:val></tde:subject>
           <tde:predicate><tde:val>sem:iri("http://www.marklogic.com/data-hub#conceptPredicate")</tde:val></tde:predicate>
           <tde:object><tde:val>sem:iri("{ $predicate_concept}")</tde:val></tde:object>
         </tde:triple>
       </tde:triples>
      </tde:template>
    ) else (),
     for $property-name in $related-entity-type-property-names
     let $property-info := map:get($ref-properties, $property-name)
     let $property-info := if (map:contains($property-info, "items")) then map:get($property-info, "items") else $property-info
     let $is-related-entity-type := map:contains($property-info, "relatedEntityType" )
     let $related-entity-type := map:get($property-info, "relatedEntityType" )
     where exists($related-entity-type)
     return
        <tde:template>
          <tde:context>{$property-name}[xs:string(.) ne ""]</tde:context>
          <tde:triples>
            <tde:triple>
            <tde:subject><tde:val>$subject-iri</tde:val></tde:subject>
            <tde:predicate><tde:val>sem:iri("{ model-graph-prefix($model) }/{ $ref-type-name }/{$property-name}")</tde:val></tde:predicate>
            <tde:object><tde:val>sem:iri(concat("{ $related-entity-type}/", fn:encode-for-uri(fn:string(.))))</tde:val></tde:object>
          </tde:triple>
         </tde:triples>
        </tde:template>
  )[tde:triples|tde:rows|tde:templates]
  return
      <tde:template>
          <tde:context>{if (fn:exists($parent-entity-type-name)) then "*/" else ()}{
            if ($ref-prefix-value) then
              "(" || $ref-prefix-value || $ref-type-name || "|" || $ref-type-name ||")"
            else
              $ref-type-name
          }</tde:context>
          <tde:rows>
            <tde:row>
              <tde:schema-name>{ $schema-name=>$secure-tde-name() }</tde:schema-name>
              <tde:view-name>{ $ref-type-name=>$secure-tde-name() }</tde:view-name>
              <tde:view-layout>sparse</tde:view-layout>
              <tde:columns>
                <tde:column>
                  <tde:name>dataHubIRI</tde:name>
                  <tde:scalar-type>IRI</tde:scalar-type>
                  <tde:val>$subject-iri</tde:val>
                </tde:column>
                {
                  if (fn:exists($parent-entity-type-name)) then (
                    <tde:column>
                      <tde:name>{ $secure-tde-name($schema-name || "_propertyName") }</tde:name>
                      <tde:scalar-type>string</tde:scalar-type>
                      <tde:val>fn:string(fn:node-name(..))</tde:val>
                    </tde:column>,
                    let $parent-type := $model => map:get("definitions")=>map:get($parent-entity-type-name)
                    let $parent-primary-key := $parent-type=>map:get("primaryKey")
                    let $parent-prefix := $parent-type=>map:get("namespacePrefix")
                    let $parent-prefix-value := if (fn:exists($parent-prefix)) then $parent-prefix || ":" else ()
                    let $parent-primary-key := $parent-type=>map:get("primaryKey")
                    let $parent-properties := $parent-type=>map:get("properties")
                    let $parent-primary-key-type := $parent-properties => map:get($parent-primary-key) => map:get("datatype")
                    return
                    <tde:column>
                      { comment { "This column joins to property",
                      $parent-primary-key, "of",
                      $parent-entity-type-name } }
                      <tde:name>{ $parent-entity-type-name=>$secure-tde-name() }_{ $parent-primary-key=>$secure-tde-name() }</tde:name>
                      <tde:scalar-type>{ if ($parent-primary-key-type="iri") then "IRI" else $parent-primary-key-type }</tde:scalar-type>
                      <tde:val>fn:head({if ($parent-prefix-value) then "(ancestor::" || $parent-prefix-value || $parent-entity-type-name || "|" || "ancestor::" || $parent-entity-type-name || ")" else "(ancestor::" || $parent-entity-type-name || ")"}/{ if ($parent-prefix-value) then  "("|| $parent-prefix-value || $parent-primary-key || "|" || $parent-primary-key || ")" else $parent-primary-key }[xs:string(.) ne ""])</tde:val>
                    </tde:column>
                  ) else (),
                  build-columns-for-definition("", "", $secure-tde-name, $model, $ref-type-name)
              }</tde:columns>
            </tde:row>
          </tde:rows>
          <tde:triples>
            <tde:triple>
              <tde:subject><tde:val>$subject-iri</tde:val></tde:subject>
              <tde:predicate><tde:val>$RDF_TYPE</tde:val></tde:predicate>
              <tde:object><tde:val>sem:iri("{ model-graph-prefix($model) }/{ $ref-type-name }")</tde:val></tde:object>
            </tde:triple>
            <tde:triple>
              <tde:subject><tde:val>$subject-iri</tde:val></tde:subject>
              <tde:predicate><tde:val>sem:iri("http://www.w3.org/2000/01/rdf-schema#isDefinedBy")</tde:val></tde:predicate>
              <tde:object><tde:val>fn:base-uri(.)</tde:val></tde:object>
            </tde:triple>
          </tde:triples>
          {
            if (fn:exists($sub-templates)) then
              <tde:templates>{
                let $distinct-contexts := fn:distinct-values($sub-templates/tde:context ! fn:string(.))
                for $context in $distinct-contexts
                let $templates := $sub-templates[tde:context eq $context]
                let $all-vars := $templates/tde:vars/tde:var
                let $vars :=
                  for $var at $pos in $all-vars
                  let $name := fn:string($var/tde:name)
                  where fn:empty(fn:subsequence($all-vars, $pos + 1)[fn:string(tde:name) eq $name])
                  return $var
                let $rows := $templates/tde:rows/tde:row
                let $triples := $templates/tde:triples/tde:triple
                let $child-templates := $templates/tde:templates/tde:template
                return
                  <tde:template>
                    <tde:context>{$context}</tde:context>
                    {if (fn:exists($vars)) then <tde:vars>{$vars}</tde:vars> else ()}
                    {if (fn:exists($rows)) then <tde:rows>{$rows}</tde:rows> else ()}
                    {if (fn:exists($triples)) then <tde:triples>{$triples}</tde:triples> else ()}
                    {if (fn:exists($child-templates)) then <tde:templates>{$child-templates}</tde:templates> else ()}
                  </tde:template>
              }</tde:templates>
            else ()
          }
        </tde:template>
};

declare %private function join-name-triples($model, $ref-type-name, $ref-prefix-value, $ref-primary-key) {
  let $model-type-iri := model-graph-prefix($model) || "/" || $ref-type-name
  let $ref-properties := $model => map:get("definitions") => map:get($ref-type-name) => map:get("properties")
  for $property-name in map:keys($ref-properties)
  let $property-info := map:get($ref-properties, $property-name)
  let $property-info := if (map:contains($property-info, "items")) then map:get($property-info, "items") else $property-info
  let $reference-to-entity-exists := fn:not($property-name = $ref-primary-key) and xdmp:exists(cts:search(
      fn:collection("http://marklogic.com/entity-services/models"),
    cts:json-property-scope-query("properties", cts:and-query((
            cts:json-property-value-query("relatedEntityType", $model-type-iri),
            cts:json-property-value-query("joinPropertyName", $property-name)
        )))
    ))
  where $reference-to-entity-exists
  return
    let $context := if ($ref-prefix-value) then "("|| $ref-prefix-value || $property-name || "|" || $property-name || ")" else $property-name
    let $subject :=  "sem:iri(concat(""" ||$model-type-iri || "/"", fn:encode-for-uri(xs:string(.))))"
    return  <tde:template>
      <tde:context>{ $context}[xs:string(.) ne ""]</tde:context>
      <tde:vars>
        <tde:var><tde:name>related-subject-iri</tde:name><tde:val>{$subject}</tde:val></tde:var>
      </tde:vars>
      <tde:triples>
        <tde:triple>
          <tde:subject><tde:val>$related-subject-iri</tde:val></tde:subject>
          <tde:predicate><tde:val>$RDF_TYPE</tde:val></tde:predicate>
          <tde:object><tde:val>sem:iri("{$model-type-iri}")</tde:val></tde:object>
        </tde:triple>
        <tde:triple>
          <tde:subject><tde:val>$related-subject-iri</tde:val></tde:subject>
          <tde:predicate><tde:val>sem:iri("http://www.w3.org/2000/01/rdf-schema#isDefinedBy")</tde:val></tde:predicate>
          <tde:object><tde:val>fn:base-uri(.)</tde:val></tde:object>
        </tde:triple>
      </tde:triples>
      </tde:template>
};

declare %private function build-columns-for-local-ref($column-prefix, $prefixed-path, $secure-tde-name, $model, $entity-type-name, $property-name) {
  let $ref-type-name := ref-type-name($model, $entity-type-name, $property-name)
  return build-columns-for-definition($column-prefix, $prefixed-path, $secure-tde-name, $model, $ref-type-name)
};

declare %private function build-columns-for-definition($column-prefix, $prefixed-path, $secure-tde-name, $model, $ref-type-name) {
  let $ref-type := $model=>map:get("definitions")=>map:get($ref-type-name)
  let $ref-prefix := $ref-type=>map:get("namespacePrefix")
  let $required-properties := $ref-type=>map:get("required")
  let $ref-properties := $ref-type=>map:get("properties")
  let $ref-prefix-value := if (fn:exists($ref-prefix[. ne ""])) then $ref-prefix || ":" else ()
  for $ref-property-name in map:keys($ref-properties)
  let $ref-property := $ref-properties => map:get($ref-property-name)
  let $additional-path :=
    $prefixed-path || (if ($ref-prefix-value) then
      "(" || $ref-prefix-value || $ref-property-name || "|" || $ref-property-name || ")"
    else
      $ref-property-name) || "/" || ref-prefixed-name($model, $ref-type-name, $ref-property-name) || "/"
  let $is-local-ref := fn:starts-with(fn:string($ref-property => map:get("$ref")), "#/definitions/")
  let $current-column-prefix := $column-prefix || $ref-property-name=>$secure-tde-name() || "_"
  let $thisPropertyIsRequired := fn:exists(index-of(json:array-values($required-properties), $ref-property-name))
  let $is-nullable :=
      if ($thisPropertyIsRequired)
      then <tde:nullable>false</tde:nullable>
      else <tde:nullable>true</tde:nullable>
  where fn:empty($ref-property => map:get("items"))
  return
    if ($is-local-ref) then (
      build-columns-for-local-ref($current-column-prefix, $additional-path, $secure-tde-name, $model, $ref-type-name, $ref-property-name)[fn:not(fn:ends-with(tde:name, "_DataHubGeneratedPrimaryKey"))]
    ) else (
      <tde:column>
        <tde:name>{$column-prefix || ($ref-property-name=>$secure-tde-name()) }</tde:name>
        <tde:scalar-type>{ let $dt := $ref-property => map:get("datatype") return if ($dt="iri") then "IRI" else $dt} </tde:scalar-type>
        <tde:val>{ $prefixed-path }{
          if ($ref-prefix-value) then "(" || $ref-prefix-value || $ref-property-name || "|" || $ref-property-name || ")" else $ref-property-name
          }</tde:val>
        {$is-nullable}
      </tde:column>
  )
};

declare %private function get-property-xpath($model as map:map, $entity-type-name as xs:string, $property-names)
{
  let $xpath := if(fn:empty($property-names)) then ""
  else
    let $curr-prop-name := $property-names[1]
    let $new-prop-names := fn:remove($property-names, 1)
    let $type := ref-type-name($model, $entity-type-name, $curr-prop-name)

    let $entity-type := $model=>map:get("definitions")=>map:get($entity-type-name)
    let $namespace-prefix := $entity-type=>map:get("namespacePrefix")
    let $prefix-value := if ($namespace-prefix) then $namespace-prefix || ":" else ""

    let $prop-name-with-ns := if ($prefix-value) then "("|| $prefix-value || $curr-prop-name || "|" || $curr-prop-name || ")" else $curr-prop-name
    let $type-with-ns := if ($prefix-value) then "("|| $prefix-value || $type || "|" || $type || ")" else $type

    let $curr-path := get-property-xpath($model, $type, $new-prop-names)
    return if(fn:empty($curr-path) or fn:string($curr-path) eq "") then
      if(fn:empty($type) or fn:string($type) eq "") then $prop-name-with-ns
      else fn:concat($prop-name-with-ns, "/", $type-with-ns)
    else fn:concat($prop-name-with-ns, "/", $type-with-ns, "/", $curr-path)
  return $xpath
};

declare %private function ref-prefixed-name(
    $model as map:map,
    $entity-type-name as xs:string,
    $property-name as xs:string
) as xs:string
{
    let $ref-type := ref-type( $model, $entity-type-name, $property-name )
    let $ref-name := ref-type-name($model, $entity-type-name, $property-name)
    let $namespace-prefix := $ref-type=>map:get("namespacePrefix")
    let $is-local-ref := is-local-reference($model, $entity-type-name, $property-name)
    return
        if ($namespace-prefix and $is-local-ref)
        then "(" || $namespace-prefix || ":" || $ref-name || "|" || $ref-name || ")"
        else $ref-name
};

declare %private function resolve-base-prefix(
    $info as map:map
) as xs:string
{
    replace(resolve-base-uri($info), "#", "/")
};


(:
 : Resolves a reference and returns its datatype
 : If the reference is external, return 'string'
 :)
declare %private function ref-datatype(
    $model as map:map,
    $entity-type-name as xs:string,
    $property-name as xs:string
) as xs:string
{
    let $ref-type := ref-type($model, $entity-type-name, $property-name)
    return
        if (is-local-reference($model, $entity-type-name, $property-name))
        then
            (: if the referent type has a primary key, use that type :)
            let $primary-key-property := map:get($ref-type, "primaryKey")
            return
                if (empty($primary-key-property))
                then "string"
                else map:get(
                        map:get(
                            map:get($ref-type, "properties"),
                            $primary-key-property),
                        "datatype")
        else "string"
};

declare %private function ref-type(
    $model as map:map,
    $entity-type-name as xs:string,
    $property-name as xs:string
) as map:map?
{
    $model
        =>map:get("definitions")
        =>map:get( ref-type-name($model, $entity-type-name, $property-name) )
};


(:
 : Given a model, an entity type name and a reference property,
 : return a reference's type name
 :)
declare function ref-type-name(
    $model as map:map,
    $entity-type-name as xs:string,
    $property-name as xs:string
) as xs:string
{
    let $property := $model
        =>map:get("definitions")
        =>map:get($entity-type-name)
        =>map:get("properties")
        =>map:get($property-name)
    let $ref-target := head( ($property=>map:get("$ref"),
                  $property=>map:get("items")=>map:get("$ref") ) )
    return functx:substring-after-last($ref-target, "/")
};


declare %private function is-local-reference(
    $model as map:map,
    $entity-type-name as xs:string,
    $property-name as xs:string
)
{
    let $top-id := map:get($model,"$id")
    let $property := $model
        =>map:get("definitions")
        =>map:get($entity-type-name)
        =>map:get("properties")
        =>map:get($property-name)
    let $id-target :=
      head( ($property=>map:get("$id"),
             $property=>map:get("items")=>map:get("$id") ) )
    let $id-target :=
      if (empty($id-target) or (exists($top-id) and $top-id=$id-target)) then ()
      else $id-target
    let $ref-target := head( ($property=>map:get("$ref"),
                              $property=>map:get("items")=>map:get("$ref") ) )
    return starts-with($ref-target, "#/definitions") and empty($id-target)
};


(: returns empty-sequence if no primary key :)
declare %private function ref-primary-key-name(
    $model as map:map,
    $entity-type-name as xs:string,
    $property-name as xs:string
) as xs:string?
{
    let $ref-type-name := ref-type-name($model, $entity-type-name, $property-name)
    let $ref-target := $model=>map:get("definitions")=>map:get($ref-type-name)
    return
        if (is-local-reference($model, $entity-type-name, $property-name)) then (
          if (map:contains($ref-target, "primaryKey"))
          then map:get($ref-target, "primaryKey")
          else ()
        ) else ()
};


declare %private function model-create(
    $model-descriptor
) as map:map
{
  let $tentative-result :=
    typeswitch ($model-descriptor)
    case document-node() return
        if ($model-descriptor/object-node())
        then xdmp:from-json($model-descriptor)
        else model-from-xml($model-descriptor/node())
    case element() return
        model-from-xml($model-descriptor)
    case object-node() return
        xdmp:from-json($model-descriptor)
    case map:map return $model-descriptor
    default return fn:error((), "ES-MODEL-INVALID")
  return
    if (is-modern-model($tentative-result))
    then $tentative-result
    else modernize($tentative-result)
};


declare function model-from-xml(
    $model as element(es:model)
) as map:map
{
    let $info := json:object()
        =>map:with("title", data($model/es:info/es:title))
        =>map:with("version", data($model/es:info/es:version))
        =>with-if-exists("baseUri", data($model/es:info/es:base-uri))
        =>with-if-exists("description", data($model/es:info/es:description))
    let $definitions :=
        let $d := json:object()
        let $_ :=
            for $entity-type-node in $model/es:definitions/*
            let $entity-type := json:object()
            let $properties := json:object()
            let $_ :=
                for $property-node in $entity-type-node/es:properties/*
                let $property-attributes := json:object()
                    =>with-if-exists("datatype", data($property-node/es:datatype))
                    =>with-if-exists("$ref", data($property-node/es:ref))
                    =>with-if-exists("description", data($property-node/es:description))
                    =>with-if-exists("collation", data($property-node/es:collation))
                    =>with-if-exists("$id", data($property-node/es:id))
                    =>with-if-exists("type", data($property-node/es:type))
                    =>with-if-exists("format", data($property-node/es:format))

                let $items-map := json:object()
                    =>with-if-exists("datatype", data($property-node/es:items/es:datatype))
                    =>with-if-exists("$ref", data($property-node/es:items/es:ref))
                    =>with-if-exists("description", data($property-node/es:items/es:description))
                    =>with-if-exists("collation", data($property-node/es:items/es:collation))
                    =>with-if-exists("$id", data($property-node/es:items/es:id))
                    =>with-if-exists("type", data($property-node/es:items/es:type))
                    =>with-if-exists("format", data($property-node/es:items/es:format))
                let $_ := if (count(map:keys($items-map)) gt 0)
                        then map:put($property-attributes, "items", $items-map)
                        else ()
                return map:put($properties, fn:local-name($property-node), $property-attributes)
            let $_ :=
                $entity-type
                  =>map:with("properties", $properties)
                  =>with-if-exists("primaryKey", data($entity-type-node/es:primary-key))
                  =>with-if-exists("required", json:to-array($entity-type-node/es:required/xs:string(.)))
                  =>with-if-exists("pii", json:to-array($entity-type-node/es:pii/xs:string(.)))
                  =>with-if-exists("namespace", $entity-type-node/es:namespace/xs:string(.))
                  =>with-if-exists("namespacePrefix", $entity-type-node/es:namespace-prefix/xs:string(.))
                  =>with-if-exists("rangeIndex", json:to-array($entity-type-node/es:range-index/xs:string(.)))
                  =>with-if-exists("pathRangeIndex", json:to-array($entity-type-node/es:path-range-index/xs:string(.)))
                  =>with-if-exists("elementRangeIndex", json:to-array($entity-type-node/es:element-range-index/xs:string(.)))
                  =>with-if-exists("wordLexicon", json:to-array($entity-type-node/es:word-lexicon/xs:string(.)))
                  =>with-if-exists("description", data($entity-type-node/es:description))
                  =>with-if-exists("$id", data($entity-type-node/es:id))
            return map:put($d, fn:local-name($entity-type-node), $entity-type)
        return $d

    let $properties :=
      let $top-entity := json:object()
      let $property-node := ($model/es:properties/*)[1] (: There can be only 1 :)
      return (
        (: {entity: { "$ref": "whatever" } } :)
        if (empty($property-node)) then () else (
          let $ref :=
            json:object()=>with-if-exists("$ref", data($property-node/es:ref))
          return json:object()=>map:with( fn:local-name($property-node), $ref )
        )
      )
    return json:object()
        =>map:with("lang","zxx")
        =>map:with("info", $info)
        =>map:with("definitions", $definitions)
        =>with-if-exists("$schema", data($model/es:schema))
        =>with-if-exists("$id", data($model/es:id))
        =>with-if-exists("required", json:to-array($model/es:required))
        =>with-if-exists("properties", $properties)
};



declare function
modernize($model as map:map) as map:map
{
  let $new-model := fix-references($model)
  let $top-entity := top-entity($new-model, true())
  return model-to-json-schema($new-model, $top-entity)
};

declare function
fix-references($model as map:map) as map:map
{
  let $_ :=
    let $top-id := map:get($model,"$id")
    for $entity-type in $model=>map:get("definitions")=>map:keys()
    let $entity := $model=>map:get("definitions")=>map:get($entity-type)
    return walk-to-fix-references($top-id, $entity)
  return $model
};

declare %private function with-if-exists(
    $map as map:map,
    $key-name as xs:string,
    $value as item()?
) as map:map
{
    typeswitch($value)
    case json:array return
        if (json:array-size($value) gt 0)
        then map:put($map, $key-name, $value)
        else ()
    default return
        if (exists($value))
        then map:put($map, $key-name, $value)
        else (),
    $map
};

(:
   Pick a top entity. If this is a modern model, we know already.
   If this is a legacy model, if there is only one entity, that's it.
   If there is only one entity that is NOT the target of a local reference,
   that must be it. Otherwise, we don't know, and we have to fall-back to
   lousy JSON Schemas, and lousy TDE template paths.
 :)
declare function
top-entity($model as map:map, $force as xs:boolean) as xs:string?
{
  let $entity-type-names := $model=>map:get("definitions")=>map:keys()
  let $model-title := $model=>map:get("info")=>map:get("title")
  return (
    if (is-modern-model($model)) then json:array-values($model=>map:get("required"))
    else if (count($entity-type-names)=1) then $entity-type-names
    else (
      let $local-refs := local-references($model)
      (: All the entities that are not the target of a local reference :)
      let $non-child-entities := $entity-type-names[not(. = $local-refs)]
      return (
        if (count($non-child-entities)=1) then (
          $non-child-entities
        ) else if ($force) then (
          (: No good reason for picking, but pick anyway :)
          fn:head(($model-title[. = $entity-type-names], $non-child-entities[1]))
        ) else () (: No basis for picking :)
      )
    )
  )
};

declare function model-to-json-schema(
    $model as map:map,
    $entity-name as xs:string?
) as map:map
{
  if (is-modern-model($model)) then (
    $model
  ) else if (empty($entity-name)) then (
    model-to-json-schema($model)
  ) else (
    (:
       Take the model and add
       "$schema": "http://json-schema.org/draft-07/schema#" (if no $schema)
       "properties": { <$entity-name> : {"$ref": "#/definitions/<$entity-name>"} },
       "required": [<$entity-name>]
       If there is no definition with that name, error.
       If there is a info/baseUri property, add equivalent $id (if no $id)
       Fix external references.
       [53510] put primaryKey into required array if it is not there already
    :)
    let $new-model := fix-primary-keys(fix-references($model))
    return (
      if (empty($new-model=>map:get("definitions")=>map:get($entity-name))) then (
        fn:error((), "ES-ENTITY-NOTFOUND", $entity-name)
      ) else (
        if (empty($new-model=>map:get("$schema")))
        then map:put($new-model, "$schema", "http://json-schema.org/draft-07/schema#")
        else (),
        map:put($new-model,"lang","zxx"),
        map:put($new-model, "properties",
          map:map()=>map:with($entity-name,
            map:map()=>map:with("$ref", "#/definitions/"||$entity-name))),
        map:put($new-model, "required", json:to-array($entity-name)),
        let $baseUri := $new-model=>map:get("info")=>map:get("baseUri")
        where not(empty($baseUri)) and empty($new-model=>map:get("$id"))
        return map:put($new-model, "$id", $baseUri),
        $new-model
      )
    )
  )
};

declare function
fix-primary-keys($model as map:map) as map:map
{
  let $_ :=
    for $entity-type in $model=>map:get("definitions")=>map:keys()
    let $entity := $model=>map:get("definitions")=>map:get($entity-type)
    return walk-to-fix-primary-keys($entity)
  return $model
};


(:
    primary keys should be required
 :)
declare %private function
walk-to-fix-primary-keys($model as map:map) as empty-sequence()
{
  let $required := $model=>map:get("required")
  let $primaryKey := $model=>map:get("primaryKey")
  where (not(empty($primaryKey)) and
         (empty($required) or not($primaryKey = json:array-values($required)))
        )
  return (
    map:put($model, "required", json:to-array((json:array-values($required), $primaryKey)))
  )
  ,
  for $property in map:get($model, "properties")=>map:keys()
  let $propspec := map:get($model, "properties")=>map:get($property)
  return (
    walk-to-fix-primary-keys($propspec),
    if (exists($propspec=>map:get("items")))
    then walk-to-fix-primary-keys($propspec=>map:get("items"))
    else ()
  )
};


declare function model-to-json-schema(
    $model as map:map
) as map:map
{
  if (is-modern-model($model)) then (
    $model
  ) else (
    (:
       Take the model and add
       "$schema": "http://json-schema.org/draft-07/schema#" (if no $schema)
       "oneOf": [
         {"properties": { <$entity-name1> : {"$ref": "#/definitions/<$entity-name1>"} },
          "required": [<$entity-name1>]
         },
         {"properties": { <$entity-name2> : {"$ref": "#/definitions/<$entity-name2">} },
         "required": [<$entity-name2>]
         },
         ...
       ]
       If there are no definitions, error.
       If there is only one definition, just add it directly instead of the
       "oneOf"
       If there is a info/baseUri property, add equivalent $id (if no $id)
       [53510] put primaryKey into required array if it is not there already
    :)
    let $new-model := fix-primary-keys(fix-references($model))
    return (
      if (empty($new-model=>map:get("definitions")=>map:keys())) then (
        fn:error((), "ES-DEFINITIONS")
      ) else (
        if (empty($new-model=>map:get("$schema")))
        then map:put($new-model, "$schema", "http://json-schema.org/draft-07/schema#")
        else (),
        map:put($new-model,"lang","zxx"),
        let $alts :=
          for $entity-name in $new-model=>map:get("definitions")=>map:keys()
          return
            map:map()=>
              map:with("properties",
                map:map()=>map:with($entity-name,
                  map:map()=>map:with("$ref", "#/definitions/"||$entity-name)))=>
              map:with("required",
                json:to-array($entity-name))
        return (
          if (count($alts)=1) then (
            map:put($new-model, "properties", $alts=>map:get("properties")),
            map:put($new-model, "required", $alts=>map:get("required"))
          ) else (
            map:put($new-model, "oneOf", json:to-array($alts))
          )
        ),
        let $baseUri := $new-model=>map:get("info")=>map:get("baseUri")
        where not(empty($baseUri)) and empty($new-model=>map:get("$id"))
        return map:put($new-model, "$id", $baseUri),
        $new-model
      )
    )
  )
};

(:
   Old style external references are not consistent with $ref
   e.g. { "$ref": "http://example.com/base/OrderDetails" } should be
   { "$id": "http://example.com/base", "$ref": "#/definitions/OrderDetails" }
 :)
declare %private function
walk-to-fix-references($top-id as xs:string?, $model as map:map) as empty-sequence()
{
  (: Only consider $ref if it is non-absolute and doesn't already have the # :)
  let $ref := $model=>map:get("$ref")
  where (not(empty($ref)) and
         not(starts-with($ref,"#")) and
         not(contains($ref,"#")))
  return (
    let $ref-name := functx:substring-after-last($ref, "/")
    let $new-ref := "#/definitions/"||$ref-name
    let $new-id := functx:substring-before-last($ref, "/")
    return (
      map:put($model, "$ref", $new-ref),
      map:put($model, "$id", $new-id)
    )
  )
  ,
  for $property in map:get($model, "properties")=>map:keys()
  let $propspec := map:get($model, "properties")=>map:get($property)
  return (
    walk-to-fix-references($top-id, $propspec),
    if (exists($propspec=>map:get("items")))
    then walk-to-fix-references($top-id, $propspec=>map:get("items"))
    else ()
  )
};

(:
   A well-constructed modern model looks like:
  {
    $schema: ...,
    info: {
      ...
    },
    definitions: {
      E1: {...},
      E2: {...},
    },
    properties: {"E1": {"$ref":"#definitions/properties/E1"}},
    required: ["E1"]
  }
  With one required top level entity
 :)
declare function
is-modern-model($model as map:map) as xs:boolean
{
  exists($model=>map:get("$schema")) and
  exists($model=>map:get("properties")) and
  count($model=>map:get("properties")=>map:keys())=1 and
  exists($model=>map:get("properties")=>map:get(($model=>map:get("properties")=>map:keys())[1])=>map:get("$ref")) and
  exists($model=>map:get("required")) and
  count(json:array-values($model=>map:get("required")))=1
};

declare %private function
local-references($model as map:map) as xs:string*
{
  let $top-id := map:get($model,"$id")
  for $entity-type in $model=>map:get("definitions")=>map:keys()
  let $entity := $model=>map:get("definitions")=>map:get($entity-type)
  return walk-for-ref($top-id, $entity)
};


declare %private function
walk-for-ref($top-id as xs:string?, $model as map:map) as xs:string*
{
  (: Only consider $ref if there is not a $id that makes it non-local :)
  let $id := map:get($model,"$id")
  let $id :=
    if (exists($top-id)) then (
      if (exists($id) and $id!=$top-id) then $id else ()
    ) else (
      $id
    )
  let $ref := map:get($model,"$ref")
  where (exists($ref) and starts-with($ref,"#"))
  return tokenize($ref,"/")[last()]
  ,
  for $property in map:get($model, "properties") ! map:keys(.)
  let $propspec := map:get($model, "properties")=>map:get($property)
  return (
    walk-for-ref($top-id, $propspec),
    if (exists($propspec=>map:get("items")))
    then walk-for-ref($top-id, $propspec=>map:get("items"))
    else ()
  )
};
